/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

/**
 * ToolItem is the base class of widgets that can be added to ToolBar objects.
 * 
 * <p>ToolItems are widgets that can appear on a toolbar. To create a toolbar 
 * item that contain something else than a button, use new {@link ToolItem#ToolItem()}. 
 * Use {@link org.gnu.gtk.Container#add(Widget)} to add a child widget to the tool item.
 *
 * <p>For toolbar items that contain buttons, see the {@link ToolButton}, {@link ToggleToolButton}
 * and {@link RadioToolButton} classes.
 * see {@link org.gnu.gtk.ToolBar} class for a description of the toolbar widget.
 */
public class ToolItem extends Bin {
	
	
	/**
	 * Construct a new ToolItem object.
	 */
	public ToolItem() {
		if (-1 == handle) {
			handle = gtk_tool_item_new();
			initializeEventHandlers();
		}
	}
	
	/**
	 * Construct a new ToolItem object from a handle to a native widget.
	 */
	ToolItem(int hndl) {
		this.handle = hndl;
		initializeEventHandlers();
	}

	/**
	 * Sets whether the ToolItem is to be allocated the same size as other
	 * homogeneous items.  The effect is that all homogeneous items will have
	 * the same width as the widest of the items.
	 * @param homogeneous <code>TRUE</code> if the item is the same size as other homogeneous items
	 */
	public void setHomogeneous(boolean homogeneous) {
		gtk_tool_item_set_homogeneous(handle, homogeneous);
	}
	
	/**
	 * Returns whether the item is the same size as other homogeneous items.
	 * @return <code>TRUE</code> if the item is the same size as other homogeneous items
	 */
	public boolean getHomogeneous() {
		return gtk_tool_item_get_homogeneous(handle);
	}
	
	/**
	 * Sets whether the ToolItem is allocated extra space when there is more room
	 * on the ToolBar than needed by the ToolItems.  The effect is that the item gets
	 * bigger when the ToolBar gets bigger and smaller when the ToolBar gets smaller.
	 * @param expand whether ToolItem is allocated extra space 
	 */
	public void setExpand(boolean expand) {
		gtk_tool_item_set_expand(handle, expand);
	}
	
	/**
	 * Returns whether ToolItem is allocated extra space.
	 * @return <code>TRUE</code> if ToolItem is allocated extra space
	 */
	public boolean getExpand() {
		return gtk_tool_item_get_expand(handle);
	}
	
	/**
	 * Sets the {@link ToolTips} object to be used for ToolItem, the text to be 
	 * displayed as tooltip on the item and the private text to be used. 
	 * @see ToolTips#setTip(Widget, String, String)
	 * @param tooltips the {@link ToolTips} object to be used
	 * @param tipText text to be used as tooltip text for ToolItem
	 * @param tipPrivate text to be used as private tooltip text 
	 */
	public void setTooltip(ToolTips toolTips, String tipText, String tipPrivate) {
		gtk_tool_item_set_tooltip(handle, toolTips.getHandle(), tipText, tipPrivate);
	}
	
	/**
	 * Sets whether ToolItem has a drag window. When <code>TRUE</code> the ToolItem 
	 * can be used as a drag source through {@link org.gnu.gdk.DragSource}. When ToolItem 
	 * has a drag window it will intercept all events, even those that would otherwise
	 * be sent to a child of ToolItem.
	 * @param useDragWindow whether ToolItem has a drag window
	 */
	public void setUseDragWindow(boolean useDragWindow) {
		gtk_tool_item_set_use_drag_window(handle, useDragWindow);
	}
	
	/**
	 * Returns whether ToolItem has a drag window. 
	 * @see ToolItem#setUseDragWindow(boolean)
	 * @return <code>TRUE</code> if toolitem uses a drag window
	 */
	public boolean getUseDragWindow() {
		return gtk_tool_item_get_use_drag_window(handle);
	}
	
	/**
	 * Sets whether ToolItem is visible when the toolbar is docked horizontally.
	 * @param visibleHorizontal whether ToolItem is visible when the toolbar is in horizontal mode
	 */
	public void setVisibleHorizontal(boolean visibleHorizontal) {
		gtk_tool_item_set_visible_horizontal(handle, visibleHorizontal);
	}
	
	/**
	 * Returns whether the ToolItem is visible on toolbars that are docked horizontally.
	 * @see ToolItem#setVisibleHorizontal(boolean)
	 * @return <code>TRUE</code> if ToolItem is visible on toolbars that are docked horizontally
	 */
	public boolean getVisibleHorizontal() {
		return gtk_tool_item_get_visible_horizontal(handle);
	}
	
	/**
	 * Sets whether ToolItem is visible when the toolbar is docked vertically. 
	 * Some tool items, such as text entries, are too wide to be useful on a vertically 
	 * docked toolbar. If <code>visibleVertical</code> is <code>FALSE</code> ToolItem will not 
	 * appear on toolbars that are docked vertically.
	 * @param visibleVertical whether toolitem is visible when the toolbar is in vertical mode
	 */
	public void setVisibleVertical(boolean visibleVertical) {
		gtk_tool_item_set_visible_vertical(handle, visibleVertical);
	}
	
	/**
	 * Returns whether ToolItem is visible on toolbars that are docked vertically. 
	 * @see ToolItem#setVisibleVertical(boolean)
	 * @return <code>TRUE</code> if ToolItem is visible on toolbars that are docked vertically
	 */
	public boolean getVisibleVertical() {
		return gtk_tool_item_get_visible_vertical(handle);
	}
	
	/**
	 * Sets whether ToolItem should be considered important. The {@link ToolButton} class 
	 * uses this property to determine whether to show or hide its label when the 
	 * toolbar style is {@link ToolBarStyle#_BOTH_HORIZ}. The result is that only 
	 * tool buttons with the IS_IMPORTANT property set have labels, an effect known 
	 * as "priority text".
	 * @param isImportant whether the tool item should be considered important
	 */
	public void setIsImportant(boolean isImportant) {
		gtk_tool_item_set_is_important(handle, isImportant);
	}
	
	/**
	 * Returns whether ToolItem is considered important. 
	 * @see ToolItem#setIsImportant(boolean)
	 * @return <code>TRUE</code> if ToolItem is considered important
	 */
	public boolean getIsImportant() {
		return gtk_tool_item_get_is_important(handle);
	}
	
	/**
	 * Returns the icon size used for ToolItem. Custom subclasses of ToolItem should
	 * call this function to find out what size icons they should use.
	 * @return a {@link IconSize} indicating the icon size used for ToolItem
	 */
	public IconSize getIconSize() {
		return IconSize.intern(gtk_tool_item_get_icon_size(handle));
	}
	
	/**
	 * Returns the orientation used for ToolItem. Custom subclasses of ToolItem should 
	 * call this function to find out what size icons they should use.
	 * @return a {@link Orientation} indicating the orientation used for ToolItem
	 */
	public Orientation getOrientation() {
		return Orientation.intern(gtk_tool_item_get_orientation(handle));
	}
	
	/**
	 * Returns the toolbar style used for ToolItem. Custom subclasses of ToolItem 
	 * should call this function in the handler of the GtkToolItem::toolbar_reconfigured
	 * signal to find out in what style the toolbar is displayed and change themselves 
	 * accordingly.
	 *
	 * <p>Possibilities are:
	 *
     * <p>GTK_TOOLBAR_BOTH, meaning the tool item should show both an icon and a label, stacked vertically
     * GTK_TOOLBAR_ICONS, meaning the toolbar shows only icons
     * GTK_TOOLBAR_TEXT, meaning the tool item should only show text
     * GTK_TOOLBAR_BOTH_HORIZ, meaning the tool item should show both an icon and 
     * a label, arranged horizontally (however, note the GtkToolButton::has_text_horizontally
     * that makes tool buttons not show labels when the toolbar style is GTK_TOOLBAR_BOTH_HORIZ. 
	 * @return a {@link ToolBarStyle} indicating the toolbar style used for ToolItem
	 */
	public ToolBarStyle getToolbarStyle() {
		return ToolBarStyle.intern(gtk_tool_item_get_toolbar_style(handle));
	}
	
	/**
	 * Returns the relief style of ToolItem. See {@link Button#setRelief(ReliefStyle)}. 
	 * Custom subclasses of ToolItem should call this function in the handler of the 
	 * GtkToolItem::toolbar_reconfigured signal to find out the relief style of buttons.
	 * @return a {@link ReliefStyle} indicating the relief style used for ToolItem
	 */
	public ReliefStyle getReliefStyle() {
		return ReliefStyle.intern(gtk_tool_item_get_relief_style(handle));
	}
	
	/**
	 * Returns the {@link MenuItem} that was last set by {@link ToolItem#setProxyMenuItem(String, MenuItem)}
	 * i.e. the {@link MenuItem} that is going to appear in the overflow menu.
	 * @return the {@link MenuItem} that is going to appear in the overflow menu for tool_item
	 */
	public MenuItem retrieveProxyMenuItem() {
		return new MenuItem(gtk_tool_item_retrieve_proxy_menu_item(handle));
	}
	
	/**
	 * If <code>menuItemId<code> matches the string passed to {@link ToolItem#setProxyMenuItem(String, MenuItem)}
	 * returns the corresponding {@link MenuItem}.
	 * Custom subclasses of ToolItem should use this function to update their menu 
	 * item when the ToolItem changes. That the <code>menuItemId</code> must match ensures 
	 * that a ToolItem will not inadvertently change a menu item that they did not 
	 * create.
	 * @param menuItemId a string used to identify the menu item
	 * @return the {@link MenuItem} passed to {@link ToolItem#setProxyMenuItem(String, MenuItem)} 
	 * if the menuItemId matches
	 */
	public MenuItem getProxyMenuItem(String menuItemId) {
		return new MenuItem(gtk_tool_item_get_proxy_menu_item(handle, menuItemId));
	}
	
	/**
	 * Sets the {@link MenuItem} used in the toolbar overflow menu. The <code>menuItemId</code>
	 * is used to identify the caller of this function and should also be used with 
	 * {@link ToolItem#getProxyMenuItem(String)}.
	 * @param menuItemId a string used to identify <code>menuItem</code>
	 * @param menuItem a {@link MenuItem} to be used in the overflow menu
	 */
	public void setProxyMenuItem(String menuItemId, MenuItem menuItem){ 
		gtk_tool_item_set_proxy_menu_item(handle, menuItemId, menuItem.getHandle());
	}
	
	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int gtk_tool_item_get_type();
	native static final protected int gtk_tool_item_new();
	native static final protected void gtk_tool_item_set_homogeneous(int toolitem, boolean homogeneous);
	native static final protected boolean gtk_tool_item_get_homogeneous(int toolitem);
	native static final protected void gtk_tool_item_set_expand(int toolitem, boolean expand);
	native static final protected boolean gtk_tool_item_get_expand(int toolitem);
	native static final protected void gtk_tool_item_set_tooltip(int toolitem, int tooltips, String tipText, String tipPrivate);
	native static final protected void gtk_tool_item_set_use_drag_window(int toolitem, boolean useDragWindow);
	native static final protected boolean gtk_tool_item_get_use_drag_window(int toolitem);	
	native static final protected void gtk_tool_item_set_visible_horizontal(int toolitem, boolean visibleHorizontal);
	native static final protected boolean gtk_tool_item_get_visible_horizontal(int toolitem);
	native static final protected void gtk_tool_item_set_visible_vertical(int toolitem, boolean visibleVertical);
	native static final protected boolean gtk_tool_item_get_visible_vertical(int toolitem);
	native static final protected boolean gtk_tool_item_get_is_important(int toolitem);
	native static final protected void gtk_tool_item_set_is_important(int toolitem, boolean isImportant);
	native static final protected int gtk_tool_item_get_icon_size(int toolitem);
	native static final protected int gtk_tool_item_get_orientation(int toolitem);
	native static final protected int gtk_tool_item_get_toolbar_style(int toolitem);
	native static final protected int gtk_tool_item_get_relief_style(int toolitem);
	native static final protected int gtk_tool_item_retrieve_proxy_menu_item(int toolitem);	
	native static final protected int gtk_tool_item_get_proxy_menu_item(int toolitem, String menuItemId);
	native static final protected void gtk_tool_item_set_proxy_menu_item(int toolitem, String menuItemId, int menuItem);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
}
