/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-attributes-xml.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdlib.h>
#include "guppi-convenient.h"
#include "guppi-string.h"
#include "guppi-attributes-xml.h"

typedef struct _AttrItem AttrItem;
struct _AttrItem {
  gint type;

  gboolean undef;

  union {
    gchar *string_data;
    gint int_data;
    guint32 uint32_data;
    double double_data;
    gboolean boolean_data;
    GDate date_data;
    gpointer pointer_data;
  } d;

  GuppiContextXML *import_ctx;
  void (*destroy_fn) (gpointer);
};

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_attributes_xml_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_attributes_xml_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
g_hash_free (gpointer key, gpointer val, gpointer user_data)
{
  gchar *name = (gchar *)key;
  AttrItem *item = (AttrItem *)val;

  static void attritem_clear_data (AttrItem *);

  attritem_clear_data (item);
  guppi_unref (item->import_ctx);
  guppi_free (name);
  guppi_free (item);
}

static void
guppi_attributes_xml_finalize (GtkObject *obj)
{
  GuppiAttributesXML *ax = GUPPI_ATTRIBUTES_XML (obj);

  guppi_finalized (obj);

  g_hash_table_foreach (ax->attr_table, g_hash_free, NULL);

  g_hash_table_destroy (ax->attr_table);
  ax->attr_table = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_attributes_xml_class_init (GuppiAttributesXMLClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_attributes_xml_get_arg;
  object_class->set_arg = guppi_attributes_xml_set_arg;
  object_class->finalize = guppi_attributes_xml_finalize;
}

static void
guppi_attributes_xml_init (GuppiAttributesXML *obj)
{
  obj->attr_table = g_hash_table_new (g_str_hash, g_str_equal);
}

GtkType
guppi_attributes_xml_get_type (void)
{
  static GtkType guppi_attributes_xml_type = 0;
  if (!guppi_attributes_xml_type) {
    static const GtkTypeInfo guppi_attributes_xml_info = {
      "GuppiAttributesXML",
      sizeof(GuppiAttributesXML),
      sizeof(GuppiAttributesXMLClass),
      (GtkClassInitFunc)guppi_attributes_xml_class_init,
      (GtkObjectInitFunc)guppi_attributes_xml_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_attributes_xml_type = gtk_type_unique (GTK_TYPE_OBJECT,
						 &guppi_attributes_xml_info);
  }
  return guppi_attributes_xml_type;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

GuppiAttributesXML *
guppi_attributes_xml_new (void)
{
  return
    GUPPI_ATTRIBUTES_XML (guppi_type_new (guppi_attributes_xml_get_type ()));
}

static AttrItem *
guppi_attributes_xml_get (GuppiAttributesXML *ax, const gchar *name)
{
  gpointer ptr;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), FALSE);
  g_return_val_if_fail (name, FALSE);
  
  if (ax->attr_table 
      && g_hash_table_lookup_extended (ax->attr_table, name, NULL, &ptr))
    return (AttrItem *)ptr;
  
  return NULL;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static gboolean
valid_type (gint type)
{
  switch (type) {

  case GUPPI_AX_INT:
  case GUPPI_AX_UINT32:
  case GUPPI_AX_DOUBLE:
  case GUPPI_AX_STRING:
  case GUPPI_AX_BOOLEAN:
  case GUPPI_AX_DATE:
  case GUPPI_AX_POINTER:
    return TRUE;

  default:
    return FALSE;

  }
}

static AttrItem *
attritem_new (gint type)
{
  AttrItem *item;

  g_return_val_if_fail (valid_type (type), NULL);

  item = guppi_new0 (AttrItem, 1);
  item->type = type;
  item->undef = TRUE;

  return item;
}

static void
attritem_clear_data (AttrItem *item)
{
  if (item == NULL)
    return;

  switch (item->type) {

  case GUPPI_AX_STRING:
    guppi_free (item->d.string_data);
    break;
    
  case GUPPI_AX_POINTER:
    if (item->destroy_fn)
      item->destroy_fn (item->d.pointer_data);
    break;

  default:
    /* do nothing */
  }

  item->undef = TRUE;
}

static void
attritem_free (AttrItem *item)
{
  if (item == NULL)
    return;

  attritem_clear_data (item);
  guppi_free (item);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

gboolean
guppi_attributes_xml_declare (GuppiAttributesXML *ax, const gchar* name,
			      gint type)
{
  AttrItem *item;
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), FALSE);
  g_return_val_if_fail (name, FALSE);
  g_return_val_if_fail (valid_type (type), FALSE);

  g_return_val_if_fail(guppi_attributes_xml_get (ax, name) == NULL, FALSE);

  item = attritem_new (type);

  if (ax->attr_table == NULL)
    ax->attr_table = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (ax->attr_table, guppi_strdup (name), item);

  return TRUE;
}

gboolean
guppi_attributes_xml_declared (GuppiAttributesXML *ax, const gchar *name)
{
  gint type;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), FALSE);
  g_return_val_if_fail (name, FALSE);

  type = guppi_attributes_xml_declared_type (ax, name);

  return type != GUPPI_AX_INVALID;
}

gint
guppi_attributes_xml_declared_type (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), GUPPI_AX_INVALID);
  g_return_val_if_fail (name, GUPPI_AX_INVALID);

  item = guppi_attributes_xml_get (ax, name);

  return item ? item->type : GUPPI_AX_INVALID;
}

void
guppi_attributes_xml_undeclare (GuppiAttributesXML *ax, const gchar *name)
{
  gpointer *orig_key = NULL;
  gpointer *value = NULL;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  if (! g_hash_table_lookup_extended (ax->attr_table, name, orig_key, value))
    return;
  
  g_hash_table_remove (ax->attr_table, name);
  guppi_free (*orig_key);
  attritem_free ((AttrItem *)*value);
}


gboolean
guppi_attributes_xml_defined (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), FALSE);
  g_return_val_if_fail (name, FALSE);

  item = guppi_attributes_xml_get (ax, name);

  return ! item->undef;

}

void
guppi_attributes_xml_undef (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  if (item ==  NULL)
    return;

  attritem_clear_data (item);
}

void
guppi_attributes_xml_set_special_import_context (GuppiAttributesXML *ax,
						 const gchar *name,
						 GuppiContextXML *ctx)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);
  
  item = guppi_attributes_xml_get (ax, name);
  if (item == NULL)
    return;

  guppi_refcounting_assign (item->import_ctx, ctx);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_string (GuppiAttributesXML *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_declare (ax, name, GUPPI_AX_STRING);
}


#define STRING_INVALID NULL
const gchar *
guppi_attributes_xml_get_string (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), STRING_INVALID);
  g_return_val_if_fail (name, STRING_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, STRING_INVALID);

  g_return_val_if_fail (! item->undef, STRING_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_STRING, STRING_INVALID);

  return item->d.string_data;
}

void
guppi_attributes_xml_set_string (GuppiAttributesXML *ax, const gchar *name,
				 const gchar *str)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_STRING);

  if (item->d.string_data)
    guppi_free (item->d.string_data);
  item->undef = FALSE;
  item->d.string_data = guppi_strdup (str);
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_int (GuppiAttributesXML *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);
  
  guppi_attributes_xml_declare (ax, name, GUPPI_AX_INT);
}
	
#define INT_INVALID 0
gint
guppi_attributes_xml_get_int (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), INT_INVALID);
  g_return_val_if_fail (name, INT_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, INT_INVALID);

  g_return_val_if_fail (! item->undef, INT_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_INT, INT_INVALID);

  return item->d.int_data;
}

void
guppi_attributes_xml_set_int (GuppiAttributesXML *ax, const gchar *name,
			      gint n)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_INT);

  item->undef = FALSE;
  item->d.int_data = n;
}


/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_uint32 (GuppiAttributesXML *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);
  
  guppi_attributes_xml_declare (ax, name, GUPPI_AX_UINT32);
}
	
#define UINT32_INVALID 0
guint32
guppi_attributes_xml_get_uint32 (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), UINT32_INVALID);
  g_return_val_if_fail (name, UINT32_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, UINT32_INVALID);

  g_return_val_if_fail (! item->undef, UINT32_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_UINT32, UINT32_INVALID);

  return item->d.uint32_data;
}

void
guppi_attributes_xml_set_uint32 (GuppiAttributesXML *ax, const gchar *name,
				 guint32 n)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_UINT32);

  item->undef = FALSE;
  item->d.uint32_data = n;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_double (GuppiAttributesXML *ax,
				     const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_declare (ax, name, GUPPI_AX_DOUBLE);
}

#define DOUBLE_INVALID 0.
double
guppi_attributes_xml_get_double (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;
  
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), DOUBLE_INVALID);
  g_return_val_if_fail (name, DOUBLE_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, DOUBLE_INVALID);

  g_return_val_if_fail (! item->undef, DOUBLE_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_DOUBLE, DOUBLE_INVALID);

  return item->d.double_data;
}

void
guppi_attributes_xml_set_double (GuppiAttributesXML *ax, const gchar *name,
				 double x)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_DOUBLE);

  item->undef = FALSE;
  item->d.double_data = x;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_boolean (GuppiAttributesXML *ax,
				      const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_declare (ax, name, GUPPI_AX_BOOLEAN);
}

#define BOOLEAN_INVALID FALSE
gboolean
guppi_attributes_xml_get_boolean (GuppiAttributesXML *ax,
				  const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), BOOLEAN_INVALID);
  g_return_val_if_fail (name, BOOLEAN_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, BOOLEAN_INVALID);

  g_return_val_if_fail (! item->undef, BOOLEAN_INVALID);
  g_return_val_if_fail ( item->type == GUPPI_AX_BOOLEAN, BOOLEAN_INVALID);

  return item->d.boolean_data;
}

void
guppi_attributes_xml_set_boolean (GuppiAttributesXML *ax,
				  const gchar *name,
				  gboolean x)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_BOOLEAN);
  
  item->undef = FALSE;
  item->d.boolean_data = x;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

void
guppi_attributes_xml_declare_date (GuppiAttributesXML *ax,
				   const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_declare (ax, name, GUPPI_AX_DATE);
}

#define DATE_INVALID NULL
const GDate *
guppi_attributes_xml_get_date (GuppiAttributesXML *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), DATE_INVALID);
  g_return_val_if_fail (name, DATE_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, DATE_INVALID);

  g_return_val_if_fail (! item->undef, DATE_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_DATE, DATE_INVALID);

  return &(item->d.date_data);
}

void
guppi_attributes_xml_set_date (GuppiAttributesXML *ax, const gchar *name,
			       const GDate *dt)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);
  
  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_DATE);

  item->undef = FALSE;
  item->d.date_data = *dt;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

#define POINTER_INVALID NULL
void
guppi_attributes_xml_declare_pointer (GuppiAttributesXML *ax,
				      const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_declare (ax, name, GUPPI_AX_POINTER);
}

gpointer
guppi_attributes_xml_get_pointer (GuppiAttributesXML *ax,
				  const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), POINTER_INVALID);
  g_return_val_if_fail (name, POINTER_INVALID);

  item = guppi_attributes_xml_get (ax, name);
  g_return_val_if_fail (item, POINTER_INVALID);

  g_return_val_if_fail (! item->undef, POINTER_INVALID);
  g_return_val_if_fail (item->type == GUPPI_AX_POINTER, POINTER_INVALID);

  return item->d.pointer_data;
}

void
guppi_attributes_xml_set_pointer (GuppiAttributesXML *ax,
				  const gchar *name,
				  gpointer ptr)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_set_pointer_destroy_fn (ax, name, ptr, NULL);
}

void
guppi_attributes_xml_set_pointer_free (GuppiAttributesXML *ax,
				  const gchar *name,
				  gpointer ptr)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_set_pointer_destroy_fn (ax, name, ptr, guppi_free_fn);
}

void
guppi_attributes_xml_set_pointer_unref (GuppiAttributesXML *ax,
					const gchar *name,
					gpointer ptr)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  guppi_attributes_xml_set_pointer_destroy_fn (ax, name, ptr, guppi_unref_fn);
}

void
guppi_attributes_xml_set_pointer_destroy_fn (GuppiAttributesXML *ax,
					     const gchar *name,
					     gpointer ptr,
					     void (*fn) (gpointer))
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax));
  g_return_if_fail (name);

  item = guppi_attributes_xml_get (ax, name);
  g_return_if_fail (item && item->type == GUPPI_AX_POINTER);

  if (item->destroy_fn)
    item->destroy_fn (item->d.pointer_data);

  item->undef = FALSE;
  item->d.pointer_data = ptr;
  item->destroy_fn = fn;
}

/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

enum {
  UNKNOWN_TAG = 0,
  ATTRIBUTE_TAG,
  NAME_TAG,
  VALUE_TAG
};

static gint
tag (const CHAR *name)
{
  g_return_val_if_fail (name, UNKNOWN_TAG);

  if (!strcmp (name, "gpi:Attribute"))
    return ATTRIBUTE_TAG;
  if (!strcmp (name, "gpi:name"))
    return NAME_TAG;
  if (!strcmp (name, "gpi:value"))
    return VALUE_TAG;
  return UNKNOWN_TAG;
}

typedef struct _AttrContext AttrContext;
struct _AttrContext {
  gint current_state;
  gchar *current_name;
  gint next_value_type;

  GuppiAttributesXML *ax;
};

static AttrContext *
ac_new (GuppiAttributesXML *ax)
{
  AttrContext *ac;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), NULL);

  ac = guppi_new0 (AttrContext, 1);

  ac->ax = ax;
  guppi_ref (ac->ax);
  
  return ac;
}

static void
ac_destroy (gpointer ud)
{
  AttrContext *ac = (AttrContext *)ud;

  g_return_if_fail (ac);

  guppi_free (ac->current_name);
  guppi_unref0 (ac->ax);
  guppi_free (ac);
}

static GuppiContextXML *
ac_begin_element (const CHAR *name, const CHAR **tag_attrs, gpointer ud)
{
  AttrContext *ac = (AttrContext *)ud;
  gint t;

  g_return_val_if_fail (ac, NULL);
  g_return_val_if_fail (name, NULL);

  t = tag (name);

  switch (t) {

  case UNKNOWN_TAG:
    g_warning ("Unrecognized XML tag \"%s\"", name);
    return NULL;
    
  case ATTRIBUTE_TAG:
    if (ac->current_state != UNKNOWN_TAG) {
      g_warning ("Illegal nesting of gpi:Attribute tags");
      return NULL;
    }
    break;
      
  case NAME_TAG:
  case VALUE_TAG:
    if (ac->current_state != ATTRIBUTE_TAG) {
      g_warning ("Illegal nesting of %s tags",
		 t  == NAME_TAG ? "gpi:name" : "gpi:value");
      return NULL;
    }
    break;

  default:
    g_assert_not_reached ();
  }

  ac->current_state = t;

  if (t == VALUE_TAG) {
    AttrItem *item = guppi_attributes_xml_get (ac->ax, ac->current_name);
    if (item && item->import_ctx)
      return item->import_ctx;
  }
    
  return NULL;
}

static void
ac_chars_value (const CHAR *str, gint len, gpointer ud)
{
  AttrContext *ac = (AttrContext *)ud;
  gchar *s;

  gint dt_m, dt_d, dt_y;
  GDate dt;

  g_return_if_fail (str);
  g_return_if_fail (ac);

  s = guppi_strndup (str, len);
  
  switch (ac->next_value_type) {

  case GUPPI_AX_STRING:

    guppi_attributes_xml_set_string (ac->ax, ac->current_name, s);
    break;

  case GUPPI_AX_INT:
    
    guppi_attributes_xml_set_int (ac->ax, ac->current_name, atoi (s));
    break;

  case GUPPI_AX_UINT32:

    guppi_attributes_xml_set_uint32 (ac->ax, ac->current_name,
				     (guint32) atoi (s));
    break;

  case GUPPI_AX_DOUBLE:

    guppi_attributes_xml_set_double (ac->ax, ac->current_name, atof (s));
    break;

  case GUPPI_AX_BOOLEAN:
    guppi_attributes_xml_set_boolean (ac->ax, ac->current_name,
				      guppi_string2boolean (s));
    break;

  case GUPPI_AX_DATE:

    if (sscanf (s, "%d-%d-%d", &dt_y, &dt_m, &dt_d) == 3) {
      g_date_set_dmy (&dt, dt_d, dt_m, dt_y);
      if (g_date_valid (&dt))
	guppi_attributes_xml_set_date (ac->ax, ac->current_name, &dt);
      else
	g_warning ("Illegal date: %s", s);
    } else {
      g_warning ("Ill-formed date: %s", s);
    }
    
    break;

  default:
    g_assert_not_reached();
  }

  guppi_free (s);
}

static void
ac_characters (const CHAR *str, gint len, gpointer ud)
{
  AttrContext *ac = (AttrContext *)ud;

  g_return_if_fail (str);
  g_return_if_fail (ac);

  switch (ac->current_state) {
  case ATTRIBUTE_TAG:
  case UNKNOWN_TAG:
    g_warning ("Stray characters!");
    break;

  case NAME_TAG:
    if (ac->current_name) 
      g_warning ("Clobbering name %s", ac->current_name);
    guppi_free (ac->current_name);
    ac->current_name = guppi_strndup (str, len);
    break;

  case VALUE_TAG:
    ac_chars_value (str, len, ud);
    break;

  default:
    g_assert_not_reached ();
  }
}

static void
ac_end_element (const CHAR *name, gpointer ud)
{
  AttrContext *ac = (AttrContext *)ud;

  g_return_if_fail (ac);
  g_return_if_fail (name);

  switch (tag (name)) {
  case ATTRIBUTE_TAG:
    ac->current_state = UNKNOWN_TAG;
    guppi_free (ac->current_name);
    ac->current_name = NULL;
    break;

  case NAME_TAG:
    ac->next_value_type = guppi_attributes_xml_declared_type (ac->ax,
							      ac->current_name);
    //g_assert (ac->next_value_type != GUPPI_AX_INVALID);

    if (ac->next_value_type == GUPPI_AX_INVALID)
      g_warning ("Unknown name: %s", ac->current_name);

    ac->current_state = ATTRIBUTE_TAG;
    break;

  case VALUE_TAG:
    ac->current_state = ATTRIBUTE_TAG;
    break;

  case UNKNOWN_TAG:
    g_warning ("Unknown end tag: %s", name);
    break;

  default:
    g_assert_not_reached ();
  }

}

GuppiContextXML *
guppi_attributes_xml_build_context (GuppiAttributesXML *ax)
{
  GuppiContextXML *ctx;
  AttrContext *ac;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES_XML (ax), NULL);

  ctx = guppi_context_xml_new ();

  guppi_context_xml_set_begin_element_fn (ctx, ac_begin_element);
  guppi_context_xml_set_characters_fn (ctx, ac_characters);
  guppi_context_xml_set_end_element_fn (ctx, ac_end_element);

  ac = ac_new (ax);
  guppi_context_xml_set_user_data_destroy_fn (ctx, ac, ac_destroy);


  return ctx;
}
