/* This is -*- C -*- */
/* $Id: guppi-seq-boolean.c,v 1.15 2000/12/14 20:23:04 trow Exp $ */

/*
 * guppi-seq-boolean.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <guppi-memory.h>
#include "guppi-seq-boolean.h"
#include "guppi-seq-boolean-impl.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_boolean_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_boolean_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}


static void
write_xml_element (GuppiSeq *seq, gint i, GuppiOutputXML *out)
{
  gboolean x = guppi_seq_boolean_get (GUPPI_SEQ_BOOLEAN (seq), i);
  guppi_output_xml_empty_tag (out, x ? "true" : "false");
}

static GuppiData *
create_xml_object (GuppiAttributesXML *ax)
{
  return guppi_seq_boolean_new ();
}

static GuppiContextXML *
found_true (const CHAR *name, const CHAR **attr, gpointer ud)
{
  GuppiSeqBoolean *seq = GUPPI_SEQ_BOOLEAN (ud);
  guppi_seq_boolean_append (seq, TRUE);
  return NULL;
}

static GuppiContextXML *
found_false (const CHAR *name, const CHAR **attr, gpointer ud)
{
  GuppiSeqBoolean *seq = GUPPI_SEQ_BOOLEAN (ud);
  guppi_seq_boolean_append (seq, FALSE);
  return NULL;
}
    
static GuppiContextXML *
content_xml_context (GuppiData *d, GuppiAttributesXML *ax)
{
  GuppiContextXML *ctx;

  ctx = guppi_context_xml_new ();
  guppi_context_xml_add_begin_element_fn (ctx, "gpi:true", found_true);
  guppi_context_xml_add_begin_element_fn (ctx, "gpi:false", found_false);
  guppi_context_xml_set_user_data (ctx, d);

  return ctx;
}

static void
guppi_seq_boolean_class_init (GuppiSeqBooleanClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiSeqClass *seq_class = GUPPI_SEQ_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ);

  data_class->type_name = _("Boolean Sequence");
  data_class->impl_type = GUPPI_TYPE_SEQ_BOOLEAN_IMPL;
  data_class->create_xml_object = create_xml_object;
  data_class->content_xml_context = content_xml_context;

  seq_class->write_xml_element = write_xml_element;
    
  object_class->get_arg = guppi_seq_boolean_get_arg;
  object_class->set_arg = guppi_seq_boolean_set_arg;
  object_class->finalize = guppi_seq_boolean_finalize;
}

static void
guppi_seq_boolean_init (GuppiSeqBoolean * obj)
{

}

GtkType guppi_seq_boolean_get_type (void)
{
  static GtkType guppi_seq_boolean_type = 0;
  if (!guppi_seq_boolean_type) {
    static const GtkTypeInfo guppi_seq_boolean_info = {
      "GuppiSeqBoolean",
      sizeof (GuppiSeqBoolean),
      sizeof (GuppiSeqBooleanClass),
      (GtkClassInitFunc) guppi_seq_boolean_class_init,
      (GtkObjectInitFunc) guppi_seq_boolean_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_boolean_type =
      gtk_type_unique (GUPPI_TYPE_SEQ, &guppi_seq_boolean_info);
  }
  return guppi_seq_boolean_type;
}

GuppiData *
guppi_seq_boolean_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_BOOLEAN, NULL, 0, NULL);
}

/*****************************************************************************/

typedef struct _GuppiDataOp_Boolean GuppiDataOp_Boolean;
struct _GuppiDataOp_Boolean {
  GuppiDataOp op;

  gint i;
  gsize N;
  gboolean x;

  const gint *iptr;
  const GuppiSeqBoolean *other;
};

static void
op_set (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  impl_class->set (impl, bop->i, bop->x);
}

static void
op_set_all (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set_all);
  impl_class->set_all (impl, bop->x);
}

static void
op_set_many (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->set_many) {
    impl_class->set_many (impl, bop->iptr, bop->N, bop->x);
  } else {
    gint i = 0, i0, i1;
    guppi_seq_bounds (GUPPI_SEQ (d), &i0, &i1);
    g_assert (impl_class->set);
    for (i = 0; i < bop->N; ++i) {
      gint j = bop->iptr[i];
      if (i0 <= j && j <= i1)
	impl_class->set (impl, j, bop->x);
    }
  }
}

static void
op_insert (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->insert)
    impl_class->insert (impl, bop->i, bop->x);
  else if (impl_class->insert_many)
    impl_class->insert_many (impl, bop->i, bop->x, 1);
  else
    g_assert_not_reached ();
}

static void
op_insert_many (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert_many);
  impl_class->insert_many (impl, bop->i, bop->x, bop->N);
}

static void
op_and (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImpl *other_impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  other_impl =
    GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (bop->other)));

  g_assert (impl_class->bitwise_and);
  impl_class->bitwise_and (impl, other_impl);
}

static void
op_or (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImpl *other_impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  other_impl =
    GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (bop->other)));

  g_assert (impl_class->bitwise_or);
  impl_class->bitwise_or (impl, other_impl);
}

static void
op_xor (GuppiData * d, GuppiDataOp * op)
{
  GuppiDataOp_Boolean *bop = (GuppiDataOp_Boolean *) op;
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImpl *other_impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  other_impl =
    GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (bop->other)));

  g_assert (impl_class->bitwise_xor);
  impl_class->bitwise_xor (impl, other_impl);
}

static void
op_not (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->bitwise_not);
  impl_class->bitwise_not (impl);
}

/*****************************************************************************/

GuppiData *
guppi_seq_boolean_new_aligned (const GuppiSeq * seq)
{
  GuppiSeqBoolean *aligned;

  if (seq == NULL)
    return NULL;

  aligned = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());

  guppi_seq_boolean_append_many (aligned, FALSE, guppi_seq_size (seq));
  guppi_seq_set_min_index (GUPPI_SEQ (aligned), guppi_seq_min_index (seq));

  return GUPPI_DATA (aligned);
}

gboolean guppi_seq_boolean_get (const GuppiSeqBoolean * seq, gint i)
{
  const GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, FALSE);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i), FALSE);

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);
  return (impl_class->get) (impl, i);
}

void
guppi_seq_boolean_set (GuppiSeqBoolean * seq, gint i, gboolean x)
{
  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  if (guppi_seq_boolean_get (seq, i) != x) {
    GuppiDataOp_Boolean op;
    op.op.op = op_set;
    op.i = i;
    op.x = x;
    guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) & op);
  }
}

void
guppi_seq_boolean_set_all (GuppiSeqBoolean * seq, gboolean x)
{
  GuppiDataOp_Boolean op;
  gint i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  guppi_seq_bounds (GUPPI_SEQ (seq), &i0, &i1);
  op.op.op = op_set_all;
  op.x = x;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_set_many (GuppiSeqBoolean * seq,
			    gint * indices, gsize N, gboolean x)
{
  GuppiDataOp_Boolean op;
  gint i, i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (N == 0)
    return;

  g_return_if_fail (indices != NULL);

  i0 = i1 = indices[0];
  for (i = 1; i < N; ++i) {
    if (indices[i] < i0)
      i0 = indices[i];
    if (indices[i] > i1)
      i1 = indices[i];
  }

  op.op.op = op_set_many;
  op.iptr = indices;
  op.N = N;
  op.x = x;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_clear (GuppiSeqBoolean * seq)
{
  guppi_seq_boolean_set_all (seq, FALSE);
}

void
guppi_seq_boolean_prepend (GuppiSeqBoolean * seq, gboolean x)
{
  gint i;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  i = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_boolean_insert (seq, i, x);
}

void
guppi_seq_boolean_prepend_many (GuppiSeqBoolean * seq, gboolean x, gsize n)
{
  gint i;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (n == 0)
    return;

  i = guppi_seq_min_index (GUPPI_SEQ (seq));

  guppi_seq_boolean_insert_many (seq, i, x, n);
}

void
guppi_seq_boolean_append (GuppiSeqBoolean * seq, gboolean x)
{
  gint i;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  i = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_boolean_insert (seq, i + 1, x);
}

void
guppi_seq_boolean_append_many (GuppiSeqBoolean * seq, gboolean x, gsize n)
{
  gint i;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (n == 0)
    return;

  i = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_boolean_insert_many (seq, i + 1, x, n);
}

void
guppi_seq_boolean_insert (GuppiSeqBoolean * seq, gint i, gboolean x)
{
  GuppiDataOp_Boolean op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_insert;
  op.i = i;
  op.x = x;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_insert_many (GuppiSeqBoolean * seq, gint i, gboolean x,
			       gsize N)
{
  GuppiDataOp_Boolean op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (N == 0)
    return;

  op.op.op = op_insert_many;
  op.i = i;
  op.x = x;
  op.N = N;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, N, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_bitwise_and (GuppiSeqBoolean * seq,
			       const GuppiSeqBoolean * other)
{
  GuppiDataOp_Boolean op;
  gint i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  guppi_seq_common_bounds (GUPPI_SEQ (seq), GUPPI_SEQ (other), &i0, &i1);

  op.op.op = op_and;
  op.other = other;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_bitwise_or (GuppiSeqBoolean * seq,
			      const GuppiSeqBoolean * other)
{
  GuppiDataOp_Boolean op;
  gint i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  guppi_seq_common_bounds (GUPPI_SEQ (seq), GUPPI_SEQ (other), &i0, &i1);

  op.op.op = op_or;
  op.other = other;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_bitwise_xor (GuppiSeqBoolean * seq,
			       const GuppiSeqBoolean * other)
{
  GuppiDataOp_Boolean op;
  gint i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  guppi_seq_common_bounds (GUPPI_SEQ (seq), GUPPI_SEQ (other), &i0, &i1);

  op.op.op = op_xor;
  op.other = other;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

void
guppi_seq_boolean_bitwise_not (GuppiSeqBoolean * seq)
{
  GuppiDataOp_Boolean op;
  gint i0, i1;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_BOOLEAN (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  guppi_seq_bounds (GUPPI_SEQ (seq), &i0, &i1);

  op.op.op = op_not;

  guppi_seq_changed_set (GUPPI_SEQ (seq), i0, i1, (GuppiDataOp *) & op);
}

gint guppi_seq_boolean_first_true (const GuppiSeqBoolean * seq)
{
  const GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;
  gint i, i0, i1;

  g_return_val_if_fail (seq != NULL, 1);

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  i0 = guppi_seq_min_index (GUPPI_SEQ (seq));

  if (impl_class->next_true) {
    return (impl_class->next_true) (impl, i0 - 1);
  }

  g_assert (impl_class->get);
  i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
  for (i = i0; i <= i1 && (impl_class->get) (impl, i); ++i);
  return i;
}

gint guppi_seq_boolean_next_true (const GuppiSeqBoolean * seq, gint i)
{
  const GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;
  gint i1;

  g_return_val_if_fail (seq != NULL, 1);

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
  if (i1 < i)
    return i;

  if (impl_class->next_true) {
    return (impl_class->next_true) (impl, i);
  }

  g_assert (impl_class->get);

  while (i < i1) {
    ++i;
    if ((impl_class->get) (impl, i))
      return i;
  }
  return i;
}

gint
guppi_seq_boolean_first_jointly_true (const GuppiSeqBoolean * a,
				      const GuppiSeqBoolean * b)
{
  gint i, i0, i1;

  g_return_val_if_fail (a != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_SEQ_BOOLEAN (a), 0);

  g_return_val_if_fail (b != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_SEQ_BOOLEAN (b), 0);

  guppi_seq_common_bounds (GUPPI_SEQ (a), GUPPI_SEQ (b), &i0, &i1);

  i = i0;
  if (!guppi_seq_boolean_get (a, i))
    i = guppi_seq_boolean_next_true (a, i);

  while (i <= i1 && !guppi_seq_boolean_get (b, i))
    i = guppi_seq_boolean_next_true (a, i);

  return i;
}

gint
guppi_seq_boolean_next_jointly_true (const GuppiSeqBoolean * a,
				     const GuppiSeqBoolean * b, gint i)
{
  gint i0, i1;
  gint ia, ib;

  g_return_val_if_fail (a != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_SEQ_BOOLEAN (a), 0);

  g_return_val_if_fail (b != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_SEQ_BOOLEAN (b), 0);

  guppi_seq_common_bounds (GUPPI_SEQ (a), GUPPI_SEQ (b), &i0, &i1);

  while (i <= i1) {
    ia = guppi_seq_boolean_next_true (a, i);
    ib = guppi_seq_boolean_next_true (b, i);

    if (ia <= i1 && ib <= i1) {
      if (ia == ib)
	return ia;
      else if (ia < ib && guppi_seq_boolean_get (a, ib))
	return ib;
      else if (ia > ib && guppi_seq_boolean_get (b, ia))
	return ia;
    }

    i = MAX (ia, ib);
  }

  return i;
}

gsize guppi_seq_boolean_true_count (const GuppiSeqBoolean * seq)
{
  const GuppiSeqBooleanImpl *impl;
  GuppiSeqBooleanImplClass *impl_class;
  gint i, i0, i1;
  gsize count = 0;

  g_return_val_if_fail (seq != NULL, 0);

  impl = GUPPI_SEQ_BOOLEAN_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->true_count) {
    return (impl_class->true_count) (impl);
  }

  g_assert (impl_class->get);

  guppi_seq_indices (GUPPI_SEQ (seq), &i0, &i1);

  for (i = i0; i <= i1; ++i) {
    if ((impl_class->get) (impl, i))
      ++count;
  }

  return count;
}

/* $Id: guppi-seq-boolean.c,v 1.15 2000/12/14 20:23:04 trow Exp $ */
