/* vim: set sw=8: */

/*
 * seq-notify.c: Handlers for sequence change notification.
 *
 * Copyright (C) 2000 Jody Goldberg (jgoldberg@home.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "corba-graph-series.h"
#include "gnumeric-graphs.h"
#include <bonobo.h>
#include <gtk/gtkobject.h>
#include <guppi-seq-scalar.h>
#include <guppi-seq-date.h>
#include <guppi-seq-string.h>
#include <guppi-useful.h>

struct _SeqNotify {
	GtkObject obj;

	SeqNotifyType type;
	GuppiData *data;

	CORBA_Object    notify_ref;
	union {
		POA_GNOME_Gnumeric_VectorScalarNotify	scalar;
		POA_GNOME_Gnumeric_VectorDateNotify	date;
		POA_GNOME_Gnumeric_VectorStringNotify	string;
	} servant;

	/* The remote server monitoring this series */
	union {
		GNOME_Gnumeric_VectorScalarNotify	scalar;
		GNOME_Gnumeric_VectorDateNotify		date;
		GNOME_Gnumeric_VectorStringNotify	string;
	} subscriber;
};

typedef struct {
	GtkObjectClass parent_class;
} SeqNotifyClass;

#define SEQ_NOTIFY_TYPE        (seq_notify_get_type ())
#define SEQ_NOTIFY(o)          (GTK_CHECK_CAST ((o), SEQ_NOTIFY_TYPE, SeqNotify))
#define SEQ_NOTIFY_CLASS(k)    (GTK_CHECK_CLASS_CAST((k), SEQ_NOTIFY_TYPE, SeqNotifyClass))
#define IS_SEQ_NOTIFY(o)       (GTK_CHECK_TYPE ((o), SEQ_NOTIFY_TYPE))
#define IS_SEQ_NOTIFY_CLASS(k) (GTK_CHECK_CLASS_TYPE ((k), SEQ_NOTIFY_TYPE))
#define SERVANT_TO_SEQ_NOTIFY(ptr) (SeqNotify *)(((void *)ptr) - GTK_STRUCT_OFFSET(SeqNotify, servant))

static GtkType seq_notify_get_type (void);

/***************************************************************************/

static void
impl_vector_notify_remove (PortableServer_Servant servant,
			   CORBA_Environment *ev)
{
	SeqNotify *seq = SERVANT_TO_SEQ_NOTIFY (servant);

	printf ("Guppi : remove %p\n", seq);
}

static void
impl_vector_notify_nameChanged (PortableServer_Servant servant,
				const CORBA_char *name, CORBA_Environment *ev)
{
	SeqNotify *seq = SERVANT_TO_SEQ_NOTIFY (servant);

	printf ("Guppi : name change %p\n", seq);
}

static void
impl_vector_scalar_notify_changed (PortableServer_Servant servant,
				   const CORBA_short start,
				   const GNOME_Gnumeric_SeqScalar *values,
				   CORBA_Environment *ev)
{
	SeqNotify *seq = SERVANT_TO_SEQ_NOTIFY (servant);

	printf ("Guppi : scalar changed %p\n", seq);
	guppi_seq_scalar_set_many (GUPPI_SEQ_SCALAR (seq->data), 0,
				   values->_buffer, sizeof(double),
				   values->_length);
}

static void
impl_vector_date_notify_changed (PortableServer_Servant servant,
				 const CORBA_short start,
				 const GNOME_Gnumeric_SeqDate *vals,
				 CORBA_Environment *ev)
{
	SeqNotify *seq = SERVANT_TO_SEQ_NOTIFY (servant);

	printf ("Guppi : date changed %p\n", seq);
}

static void
impl_vector_string_notify_changed (PortableServer_Servant servant,
				   const CORBA_short start,
				   const GNOME_Gnumeric_SeqString *vals,
				   CORBA_Environment *ev)
{
	int i;
	SeqNotify *seq = SERVANT_TO_SEQ_NOTIFY (servant);

	printf ("Guppi : string changed %p\n", seq);

	/* TODO : Do we need a set_many */
	i = vals->_length;
	while (i-- > 0 )
		guppi_seq_string_set (GUPPI_SEQ_STRING (seq->data),
				      i, vals->_buffer[i]);
}

static POA_GNOME_Gnumeric_VectorScalarNotify__vepv vector_scalar_notify_vepv;
static POA_GNOME_Gnumeric_VectorDateNotify__vepv vector_date_notify_vepv;
static POA_GNOME_Gnumeric_VectorStringNotify__vepv vector_string_notify_vepv;

static void
corba_implementation_classes_init (void)
{
	static POA_GNOME_Gnumeric_VectorNotify__epv 	  notify_epv;
	static POA_GNOME_Gnumeric_VectorScalarNotify__epv scalar_epv;
	static POA_GNOME_Gnumeric_VectorDateNotify__epv	  date_epv;
	static POA_GNOME_Gnumeric_VectorStringNotify__epv string_epv;

	notify_epv.remove      = &impl_vector_notify_remove;
	notify_epv.nameChanged = &impl_vector_notify_nameChanged;

	scalar_epv.valueChanged = &impl_vector_scalar_notify_changed;
	vector_scalar_notify_vepv.GNOME_Gnumeric_VectorNotify_epv =
		&notify_epv;
	vector_scalar_notify_vepv.GNOME_Gnumeric_VectorScalarNotify_epv =
		&scalar_epv;

	date_epv.valueChanged = & impl_vector_date_notify_changed;
	vector_date_notify_vepv.GNOME_Gnumeric_VectorNotify_epv =
		&notify_epv;
	vector_date_notify_vepv.GNOME_Gnumeric_VectorDateNotify_epv =
		&date_epv;

	string_epv.valueChanged = & impl_vector_string_notify_changed;
	vector_string_notify_vepv.GNOME_Gnumeric_VectorNotify_epv =
		&notify_epv;
	vector_string_notify_vepv.GNOME_Gnumeric_VectorStringNotify_epv =
		&string_epv;
}

static void
seq_notify_destroy (GtkObject *obj)
{
	printf ("destroy %p\n", obj);
}

static void
seq_notify_class_init (GtkObjectClass *object_class)
{
	static GtkObjectClass *seq_notify_parent_class = NULL;

	seq_notify_parent_class = gtk_type_class (gtk_object_get_type ());
	object_class->destroy = &seq_notify_destroy;

	corba_implementation_classes_init ();
}

static void
seq_notify_init (GtkObject *object)
{
	SeqNotify *notify = SEQ_NOTIFY (object);

	//notify->type;
	notify->notify_ref = CORBA_OBJECT_NIL;
}

static GtkType
seq_notify_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"SeqNotify",
			sizeof (SeqNotify),
			sizeof (SeqNotifyClass),
			(GtkClassInitFunc) seq_notify_class_init,
			(GtkObjectInitFunc) seq_notify_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (gtk_object_get_type (), &info);
	}

	return type;
}

SeqNotify *
seq_notify_new (SeqNotifyType type, CORBA_Object subscriber)
{
	SeqNotify *seq;

	CORBA_Environment ev;
	PortableServer_Servant serv = CORBA_OBJECT_NIL;

	seq = guppi_type_new (seq_notify_get_type ());

	CORBA_exception_init (&ev);

	switch (type) {
	case SERIES_SCALAR :
		serv = &seq->servant.scalar;
		seq->servant.scalar.vepv = &vector_scalar_notify_vepv;
		POA_GNOME_Gnumeric_VectorScalarNotify__init (serv, &ev);

		seq->subscriber.scalar = subscriber;
		seq->data = guppi_seq_scalar_new ();
		printf ("Guppi : scalar new %p\n", seq);
		break;

	case SERIES_DATE :
		serv = &seq->servant.date;
		seq->servant.date.vepv = &vector_date_notify_vepv;
		POA_GNOME_Gnumeric_VectorDateNotify__init (serv, &ev);

		seq->subscriber.date = subscriber;
		seq->data = guppi_seq_date_new ();
		printf ("Guppi : date new %p\n", seq);
		break;

	case SERIES_STRING :
		serv = &seq->servant.string;
		seq->servant.string.vepv = &vector_string_notify_vepv;
		POA_GNOME_Gnumeric_VectorStringNotify__init (serv, &ev);

		seq->subscriber.string = subscriber;
		seq->data = guppi_seq_string_new ();
		printf ("Guppi : string new %p\n", seq);
		break;

	default :
		g_assert_not_reached ();
	};

	if (ev._major == CORBA_NO_EXCEPTION) {
		PortableServer_POA poa = bonobo_poa ();
		PortableServer_ObjectId *oid;

		oid = PortableServer_POA_activate_object (poa, serv, &ev);
		CORBA_free (oid);
		seq->notify_ref = PortableServer_POA_servant_to_reference (poa, serv, &ev);
	}
	CORBA_exception_free (&ev);

	return seq;
}

GuppiData *
seq_notify_get_data (SeqNotify *seq)
{
	return seq->data;
}

CORBA_Object
seq_notify_get_servant (SeqNotify *notify)
{
	g_return_val_if_fail (IS_SEQ_NOTIFY (notify), CORBA_OBJECT_NIL);

	return notify->notify_ref;
}
