/* This is -*- C -*- */
/* $Id: guppi-text-state.c,v 1.13 2000/12/14 20:23:01 trow Exp $ */

/*
 * guppi-text-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <glade/glade.h>
#include <guppi-convenient.h>
#include <guppi-useful.h>
#include "guppi-text-state.h"
#include "guppi-text-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_TEXT,
  ARG_COLOR,
  ARG_ANGLE,
  ARG_FONT,
  ARG_FONT_SIZE,
  ARG_FILLED,
  ARG_OUTLINE,
  ARG_OUTLINE_WIDTH,
  ARG_TOP_MARGIN,
  ARG_BOTTOM_MARGIN,
  ARG_LEFT_MARGIN,
  ARG_RIGHT_MARGIN,
};

static void
guppi_text_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (obj);

  switch (arg_id) {

  case ARG_TEXT:
    GTK_VALUE_STRING (*arg) = guppi_text_state_text (state);
    break;

  case ARG_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_text_state_color (state);
    break;

  case ARG_ANGLE:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_rot_angle (state);
    break;

  case ARG_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_text_state_font (state);
    break;

  case ARG_FONT_SIZE:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_font_size (state);
    break;

  case ARG_FILLED:
    GTK_VALUE_BOOL (*arg) = guppi_text_state_filled (state);
    break;

  case ARG_OUTLINE:
    GTK_VALUE_BOOL (*arg) = guppi_text_state_outline (state);
    break;

  case ARG_OUTLINE_WIDTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_outline_width (state);
    break;

  case ARG_TOP_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_top_margin (state);
    break;

  case ARG_BOTTOM_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_bottom_margin (state);
    break;

  case ARG_LEFT_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_left_margin (state);
    break;

  case ARG_RIGHT_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_text_state_right_margin (state);
    break;

  default:
    break;
  };
}

static void
guppi_text_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (obj);

  switch (arg_id) {

  case ARG_TEXT:
    guppi_text_state_set_text (state, GTK_VALUE_STRING (*arg));
    break;

  case ARG_COLOR:
    guppi_text_state_set_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_ANGLE:
    guppi_text_state_set_rot_angle (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_FONT:
    guppi_text_state_set_font (state, GNOME_FONT (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_FONT_SIZE:
    guppi_text_state_set_font_size (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_FILLED:
    guppi_text_state_set_filled (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_OUTLINE:
    guppi_text_state_set_outline (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_OUTLINE_WIDTH:
    guppi_text_state_set_outline_width (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_TOP_MARGIN:
    guppi_text_state_set_top_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BOTTOM_MARGIN:
    guppi_text_state_set_bottom_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_LEFT_MARGIN:
    guppi_text_state_set_left_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_RIGHT_MARGIN:
    guppi_text_state_set_right_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_text_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_text_state_finalize (GtkObject * obj)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (obj);

  guppi_free0 (state->text);
  guppi_unref0 (state->font);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*****************************************************************************/

static GuppiElementView *
make_view (GuppiElementState * state)
{
  return GUPPI_ELEMENT_VIEW (guppi_type_new (GUPPI_TYPE_TEXT_VIEW));
}

/*****************************************************************************/

/*** Code for config widget ***/

static void
push_state_to_widget (GuppiTextState * state, GladeXML * xml)
{
  GtkWidget *w;
  guint8 r, g, b, a;
  gboolean outline;

  w = glade_xml_get_widget (xml, "text_entry");
  gtk_entry_set_text (GTK_ENTRY (w), guppi_text_state_text (state));


  w = glade_xml_get_widget (xml, "font_picker");
  gtk_widget_set_sensitive (w, FALSE);


  w = glade_xml_get_widget (xml, "color_picker");
  UINT_TO_RGBA (guppi_text_state_color (state), &r, &g, &b, &a);
  gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (w), r, g, b, a);


  outline = guppi_text_state_outline (state);
  w = glade_xml_get_widget (xml, "outline_check_button");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), outline);

  w = glade_xml_get_widget (xml, "thickness_spinner");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
			     guppi_text_state_outline_width (state));
  gtk_widget_set_sensitive (w, outline);

  w = glade_xml_get_widget (xml, "thickness_label");
  gtk_widget_set_sensitive (w, outline);
}

static void
text_cb (GtkEntry * entry, GuppiTextState * state)
{
  guppi_text_state_set_text (state, gtk_entry_get_text (entry));
}

static void
font_cb (GnomeFontPicker * fp, const gchar * font_name,
	 GuppiTextState * state)
{
#if 0
  GnomeFont *font;
  font = gnome_font_new_from_full_name (font_name);
  guppi_text_state_set_font (state, font);
  guppi_unref (font);
#endif
}

static void
color_cb (GnomeColorPicker * cp,
	  guint r, guint g, guint b, guint a, GuppiTextState * state)
{
  guint32 color;

  color = RGBA_TO_UINT (r >> 8, g >> 8, b >> 8, a >> 8);
  guppi_text_state_set_color (state, color);
}

static void
outline_checkbox_cb (GtkToggleButton * b, GuppiTextState * state)
{
  guppi_text_state_set_outline (state, gtk_toggle_button_get_active (b));
}

static void
outline_spinner_cb (GtkSpinButton * sb, GuppiTextState * state)
{
  double w = gtk_spin_button_get_value_as_float (sb);

  if (guppi_text_state_outline (state))
    guppi_text_state_set_outline_width (state, w);
}

static void
destroy_cb (GtkObject * w, GladeXML * xml)
{
  gtk_signal_disconnect_by_func (GTK_OBJECT (gtk_object_get_user_data (w)),
				 push_state_to_widget, xml);
}

static void
connect_signals (GuppiTextState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "text_entry");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (text_cb), state);

  w = glade_xml_get_widget (xml, "font_picker");
  gtk_signal_connect (GTK_OBJECT (w), "font_set",
		      GTK_SIGNAL_FUNC (font_cb), state);

  w = glade_xml_get_widget (xml, "outline_check_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (outline_checkbox_cb), state);

  w = glade_xml_get_widget (xml, "thickness_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (outline_spinner_cb), state);

  w = glade_xml_get_widget (xml, "color_picker");
  gtk_signal_connect (GTK_OBJECT (w), "color_set",
		      GTK_SIGNAL_FUNC (color_cb), state);
}


static GtkWidget *
config_widget_cb (gpointer user_data)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (user_data);
  const gchar *glade_xml_path;
  GladeXML *xml;
  GtkWidget *w;

  glade_xml_path = guppi_glade_path ("guppi-text-state.glade");
  g_return_val_if_fail (glade_xml_path != NULL, NULL);

  xml = glade_xml_new (glade_xml_path, "text_state_control");
  g_return_val_if_fail (xml != NULL, NULL);

  push_state_to_widget (state, xml);

  connect_signals (state, xml);
  gtk_signal_connect (GTK_OBJECT (state), "changed",
		      GTK_SIGNAL_FUNC (push_state_to_widget), xml);

  w = glade_xml_get_widget (xml, "text_state_control");
  gtk_object_set_user_data (GTK_OBJECT (w), state);

  gtk_signal_connect (GTK_OBJECT (w), "destroy",
		      GTK_SIGNAL_FUNC (destroy_cb), xml);

  return w;
}

static GuppiConfigItem *
config_tree (GuppiElementState * state)
{
  GuppiConfigItem *top_node;

  top_node = guppi_config_item_new (_("Configure Text"),
				    _("Configure"), config_widget_cb, state);
  return top_node;
}

static void
write_xml_attributes (GuppiElementState *ges, GuppiOutputXML *out)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (ges);

  if (GUPPI_ELEMENT_STATE_CLASS (parent_class)->write_xml_attributes)
    GUPPI_ELEMENT_STATE_CLASS (parent_class)->write_xml_attributes (ges, out);
  
  if (guppi_text_state_text (state))
    guppi_output_xml_attribute (out, "text", guppi_text_state_text (state));

  guppi_output_xml_attributef (out, "color", "%u", 
			       guppi_text_state_color (state));

  guppi_output_xml_attributef (out, "angle", "%g",
			       guppi_text_state_rot_angle (state));

  guppi_output_xml_attribute (out, "filled", 
			      guppi_text_state_filled (state) ? "yes" : "no");
  
  if (guppi_text_state_outline_width (state) > 0)
    guppi_output_xml_attributef (out, "outline_width", "%g",
				 guppi_text_state_outline_width (state));

  if (guppi_text_state_top_margin (state) > 0)
    guppi_output_xml_attributef (out, "top_margin", "%g",
				 guppi_text_state_top_margin (state));

  if (guppi_text_state_bottom_margin (state) > 0)
    guppi_output_xml_attributef (out, "bottom_margin", "%g",
				 guppi_text_state_bottom_margin (state));

  if (guppi_text_state_left_margin (state) > 0 )
    guppi_output_xml_attributef (out, "left_margin", "%g",
				 guppi_text_state_left_margin (state));

  if (guppi_text_state_right_margin (state) > 0)
    guppi_output_xml_attributef (out, "right_margin", "%g",
				 guppi_text_state_right_margin (state));
			      
}

static void
prepare_xml_attributes (GuppiAttributesXML *ax)
{
  guppi_attributes_xml_declare_string (ax, "text");
  guppi_attributes_xml_declare_uint32 (ax, "color"); 
  guppi_attributes_xml_declare_double (ax, "angle");
  guppi_attributes_xml_declare_boolean (ax, "filled");
  guppi_attributes_xml_declare_double (ax, "outline_width");
  guppi_attributes_xml_declare_double (ax, "top_margin");
  guppi_attributes_xml_declare_double (ax, "bottom_margin");
  guppi_attributes_xml_declare_double (ax, "left_margin");
  guppi_attributes_xml_declare_double (ax, "right_margin");
}

static void
init_xml_object (GuppiElementState *ges, GuppiAttributesXML *ax)
{
  GuppiTextState *state = GUPPI_TEXT_STATE (ges);

  if (guppi_attributes_xml_defined (ax, "text"))
    state->text = guppi_strdup (guppi_attributes_xml_get_string (ax, "text"));

  if (guppi_attributes_xml_defined (ax, "color"))
    state->text_color = guppi_attributes_xml_get_uint32 (ax, "color");

  if (guppi_attributes_xml_defined (ax, "angle"))
    state->rot_angle = guppi_attributes_xml_get_double (ax, "angle");

  if (guppi_attributes_xml_defined (ax, "filled"))
    state->filled_text = guppi_attributes_xml_get_boolean (ax, "filled");

  if (guppi_attributes_xml_defined (ax, "outline_width"))
    state->outline_width = guppi_attributes_xml_get_double (ax, "outline_width");
  
  if (guppi_attributes_xml_defined (ax, "top_margin"))
    state->top_margin = guppi_attributes_xml_get_double (ax, "top_margin");

  if (guppi_attributes_xml_defined (ax, "bottom_margin"))
    state->top_margin = guppi_attributes_xml_get_double (ax, "bottom_margin");

  if (guppi_attributes_xml_defined (ax, "left_margin"))
    state->top_margin = guppi_attributes_xml_get_double (ax, "left_margin");

  if (guppi_attributes_xml_defined (ax, "right_margin"))
    state->top_margin = guppi_attributes_xml_get_double (ax, "right_margin");
}


/*****************************************************************************/

static void
guppi_text_state_class_init (GuppiTextStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_text_state_get_arg;
  object_class->set_arg = guppi_text_state_set_arg;
  object_class->destroy = guppi_text_state_destroy;
  object_class->finalize = guppi_text_state_finalize;

  state_class->name = _("Text");

  state_class->make_view = make_view;
  state_class->config_tree = config_tree;
  state_class->write_xml_attributes = write_xml_attributes;
  state_class->prepare_xml_attributes = prepare_xml_attributes;
  state_class->init_xml_object = init_xml_object;

  gtk_object_add_arg_type ("GuppiTextState::text",
			   GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_TEXT);
  gtk_object_add_arg_type ("GuppiTextState::color_rgba",
			   GTK_TYPE_UINT, GTK_ARG_READWRITE, ARG_COLOR);
  gtk_object_add_arg_type ("GuppiTextState::angle",
			   GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_ANGLE);
  gtk_object_add_arg_type ("GuppiTextState::font",
			   GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_FONT);
  gtk_object_add_arg_type ("GuppiTextState::font_size",
			   GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_FONT_SIZE);
  gtk_object_add_arg_type ("GuppiTextState::filled",
			   GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_FILLED);
  gtk_object_add_arg_type ("GuppiTextState::outline",
			   GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_OUTLINE);
  gtk_object_add_arg_type ("GuppiTextState::outline_width",
			   GTK_TYPE_DOUBLE,
			   GTK_ARG_READWRITE, ARG_OUTLINE_WIDTH);
}

static void
guppi_text_state_init (GuppiTextState * obj)
{
  /* Some silly defaults */
  obj->text = guppi_strdup ("");
  obj->text_color = 0x000000ff;
  obj->rot_angle = 0;

  obj->font = guppi_default_font ();
  guppi_ref (obj->font);

  obj->filled_text = TRUE;
  obj->outline_width = 0.5;

  obj->top_margin = guppi_in2pt (1.0 / 64);
  obj->bottom_margin = guppi_in2pt (1.0 / 64);
  obj->left_margin = guppi_in2pt (1.0 / 64);
  obj->right_margin = guppi_in2pt (1.0 / 64);
}

GtkType guppi_text_state_get_type (void)
{
  static GtkType guppi_text_state_type = 0;
  if (!guppi_text_state_type) {
    static const GtkTypeInfo guppi_text_state_info = {
      "GuppiTextState",
      sizeof (GuppiTextState),
      sizeof (GuppiTextStateClass),
      (GtkClassInitFunc) guppi_text_state_class_init,
      (GtkObjectInitFunc) guppi_text_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_text_state_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &guppi_text_state_info);
  }
  return guppi_text_state_type;
}

GuppiElementState *
guppi_text_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_text_state_get_type ()));
}

void
guppi_text_state_set_text (GuppiTextState * state, const gchar * text)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (text != NULL);

  if (strcmp (state->text, text)) {
    guppi_free (state->text);
    state->text = guppi_strdup (text);
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_color (GuppiTextState * state, guint32 rgba)
{
  g_return_if_fail (state != NULL);

  if (state->text_color != rgba) {
    state->text_color = rgba;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_rot_angle (GuppiTextState * state, double theta)
{
  g_return_if_fail (state != NULL);

  if (state->rot_angle != theta) {
    state->rot_angle = theta;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_font (GuppiTextState * state, GnomeFont * font)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (font != NULL);

  if (state->font != font) {
    guppi_refcounting_assign (state->font, font);
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_font_size (GuppiTextState * state, double sz)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (sz >= 0);

  if (state->font && gnome_font_get_size (state->font) != sz) {
    GnomeFont *new_font;
    new_font = gnome_font_new (gnome_font_get_name (state->font), sz);
    guppi_text_state_set_font (state, new_font);
    guppi_unref (new_font);
  }
}

void
guppi_text_state_set_filled (GuppiTextState * state, gboolean x)
{
  g_return_if_fail (state != NULL);

  if (state->filled_text != x) {
    state->filled_text = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_outline (GuppiTextState * state, gboolean x)
{
  g_return_if_fail (state != NULL);

  if (state->filled_text != !x) {
    state->filled_text = !x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_outline_width (GuppiTextState * state, double w)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (w > 0);

  if (state->outline_width != w) {
    state->outline_width = w;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_top_margin (GuppiTextState * state, double m)
{
  g_return_if_fail (state != NULL);

  if (state->top_margin != m) {
    state->top_margin = m;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_bottom_margin (GuppiTextState * state, double m)
{
  g_return_if_fail (state != NULL);

  if (state->bottom_margin != m) {
    state->bottom_margin = m;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_left_margin (GuppiTextState * state, double m)
{
  g_return_if_fail (state != NULL);

  if (state->left_margin != m) {
    state->left_margin = m;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_text_state_set_right_margin (GuppiTextState * state, double m)
{
  g_return_if_fail (state != NULL);

  if (state->right_margin != m) {
    state->right_margin = m;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}


/* $Id: guppi-text-state.c,v 1.13 2000/12/14 20:23:01 trow Exp $ */
