/* This is -*- C -*- */
/* $Id: guppi-scatter-style.c,v 1.17 2000/12/14 20:22:59 trow Exp $ */

/*
 * guppi-scatter-style.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <glade/glade.h>
#include <guppi-useful.h>
#include "guppi-scatter-style.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_ALLOW_COLOR_GRAD,
  ARG_REV_COLOR_GRAD,
  ARG_COLOR_GRAD,
  ARG_ALLOW_SIZE1_GRAD,
  ARG_REV_SIZE1_GRAD,
  ARG_SIZE1_RANGE,
  ARG_ALLOW_SIZE2_GRAD,
  ARG_REV_SIZE2_GRAD,
  ARG_SIZE2_RANGE
};

static void
guppi_scatter_style_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiScatterStyle *sty = GUPPI_SCATTER_STYLE (obj);

  switch (arg_id) {

  case ARG_ALLOW_COLOR_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_allow_color_gradient (sty);
    break;

  case ARG_REV_COLOR_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_reverse_color_gradient (sty);
    break;

  case ARG_COLOR_GRAD:
    GTK_VALUE_POINTER (*arg) = guppi_scatter_style_color_gradient (sty);
    break;

  case ARG_ALLOW_SIZE1_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_allow_size1_gradient (sty);
    break;

  case ARG_REV_SIZE1_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_reverse_size1_gradient (sty);
    break;

  case ARG_SIZE1_RANGE:
    GTK_VALUE_DOUBLE (*arg) = guppi_scatter_style_size1_gradient_range (sty);
    break;

  case ARG_ALLOW_SIZE2_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_allow_size2_gradient (sty);
    break;

  case ARG_REV_SIZE2_GRAD:
    GTK_VALUE_BOOL (*arg) = guppi_scatter_style_reverse_size2_gradient (sty);
    break;

  case ARG_SIZE2_RANGE:
    GTK_VALUE_DOUBLE (*arg) = guppi_scatter_style_size2_gradient_range (sty);
    break;

  default:
    break;
  }
}

static void
guppi_scatter_style_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiScatterStyle *sty = GUPPI_SCATTER_STYLE (obj);

  switch (arg_id) {

  case ARG_ALLOW_COLOR_GRAD:
    guppi_scatter_style_set_allow_color_gradient (sty, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_REV_COLOR_GRAD:
    guppi_scatter_style_set_reverse_color_gradient (sty,
						    GTK_VALUE_BOOL (*arg));
    break;

  case ARG_COLOR_GRAD:
    guppi_scatter_style_set_color_gradient (sty,
					    GUPPI_COLOR_GRADIENT
					    (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_ALLOW_SIZE1_GRAD:
    guppi_scatter_style_set_allow_size1_gradient (sty, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_REV_SIZE1_GRAD:
    guppi_scatter_style_set_reverse_size1_gradient (sty,
						    GTK_VALUE_BOOL (*arg));
    break;

  case ARG_SIZE1_RANGE:
    guppi_scatter_style_set_size1_gradient_range (sty,
						  GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_ALLOW_SIZE2_GRAD:
    guppi_scatter_style_set_allow_size2_gradient (sty, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_REV_SIZE2_GRAD:
    guppi_scatter_style_set_reverse_size2_gradient (sty,
						    GTK_VALUE_BOOL (*arg));
    break;

  case ARG_SIZE2_RANGE:
    guppi_scatter_style_set_size2_gradient_range (sty,
						  GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  }
}

static void
guppi_scatter_style_finalize (GtkObject * obj)
{
  GuppiScatterStyle *ss = GUPPI_SCATTER_STYLE (obj);

  if (ss->color_grad) {

    gtk_signal_disconnect_by_func (GTK_OBJECT (ss->color_grad),
				   guppi_style_changed_delayed, ss);

    guppi_unref (ss->color_grad);
  }

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/****************************************************************************/

/*** Edit Widget Stuff ***/

static void
push_state_to_widget (GuppiScatterStyle * ss, GladeXML * xml)
{
  const GuppiMarkerInfo *info;
  GtkWidget *w;
  GuppiColorGradient *cg;
  gboolean cgrad_use_int;
  gboolean cgrad_use_a;
  guint32 trans_color;

  g_return_if_fail (ss != NULL);
  g_return_if_fail (xml != NULL);

  info = guppi_marker_info (guppi_style_marker (GUPPI_STYLE (ss)));

  /* Set up size gradient boxes */
  w = glade_xml_get_widget (xml, "size1_gradient");
  gtk_widget_set_sensitive (w, info->size1_desc != NULL);

  w = glade_xml_get_widget (xml, "size1_label");
  gtk_label_set_text (GTK_LABEL (w),
		      info->size1_desc ? info->size1_desc : _("N/A"));

  w = glade_xml_get_widget (xml, "size1_activate_button");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				guppi_scatter_style_allow_size1_gradient
				(ss));

  w = glade_xml_get_widget (xml, "size1_spinner");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
			     guppi_scatter_style_size1_gradient_range (ss));

  w = glade_xml_get_widget (xml, "size2_gradient");
  gtk_widget_set_sensitive (w, info->size2_desc != NULL);

  w = glade_xml_get_widget (xml, "size2_label");
  gtk_label_set_text (GTK_LABEL (w),
		      info->size2_desc ? info->size2_desc : _("N/A"));

  /* Color Gradient Controls */

  cg = guppi_scatter_style_color_gradient (ss);
  cgrad_use_int = FALSE;
  cgrad_use_a = FALSE;

  w = glade_xml_get_widget (xml, "transition_color_picker");
  gtk_widget_set_sensitive (w, FALSE);

  w = NULL;
  switch (guppi_color_gradient_type (cg)) {

  case GUPPI_COLOR_GRADIENT_NONE:
    w = glade_xml_get_widget (xml, "none_button");
    break;

  case GUPPI_COLOR_GRADIENT_TWO:
    trans_color = guppi_color_gradient_get_node (cg, 1);

    w = glade_xml_get_widget (xml, "transition_color_picker");
    gtk_widget_set_sensitive (w, TRUE);
    gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (w),
			       UINT_RGBA_R (trans_color),
			       UINT_RGBA_G (trans_color),
			       UINT_RGBA_B (trans_color),
			       UINT_RGBA_A (trans_color));


    w = glade_xml_get_widget (xml, "transition_button");

    break;

  case GUPPI_COLOR_GRADIENT_FADE_IN:
    w = glade_xml_get_widget (xml, "fade_in_button");
    break;

  case GUPPI_COLOR_GRADIENT_FADE_OUT:
    w = glade_xml_get_widget (xml, "fade_out_button");
    break;

  case GUPPI_COLOR_GRADIENT_FIRE:
    w = glade_xml_get_widget (xml, "fire_button");
    cgrad_use_int = TRUE;
    cgrad_use_a = TRUE;
    break;

  case GUPPI_COLOR_GRADIENT_ICE:
    w = glade_xml_get_widget (xml, "ice_button");
    cgrad_use_int = TRUE;
    cgrad_use_a = TRUE;
    break;

  case GUPPI_COLOR_GRADIENT_THERMAL:
    w = glade_xml_get_widget (xml, "thermal_button");
    cgrad_use_int = TRUE;
    cgrad_use_a = TRUE;
    break;

  case GUPPI_COLOR_GRADIENT_SPECTRUM:
    w = glade_xml_get_widget (xml, "spectrum_button");
    cgrad_use_int = TRUE;
    cgrad_use_a = TRUE;
    break;

  case GUPPI_COLOR_GRADIENT_CUSTOM:
    g_message ("Can't handle custom color gradients now.");

  default:
    g_assert_not_reached ();
  }

  if (w != NULL)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

  w = glade_xml_get_widget (xml, "intensity_label");
  gtk_widget_set_sensitive (w, cgrad_use_int);

  w = glade_xml_get_widget (xml, "intensity_spinner");
  gtk_widget_set_sensitive (w, cgrad_use_int);

  if (cgrad_use_int)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
			       guppi_color_gradient_intensity (cg));


  w = glade_xml_get_widget (xml, "alpha_label");
  gtk_widget_set_sensitive (w, cgrad_use_a);

  w = glade_xml_get_widget (xml, "alpha_spinner");
  gtk_widget_set_sensitive (w, cgrad_use_a);

  if (cgrad_use_a)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
			       guppi_color_gradient_alpha (cg));


}

static void
spin1_activate_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  gboolean x = gtk_toggle_button_get_active (button);
  guppi_scatter_style_set_allow_size1_gradient (ss, x);
}

static void
spin1_range_cb (GtkSpinButton * spinner, GuppiScatterStyle * ss)
{
  double x = gtk_spin_button_get_value_as_float (spinner);
  guppi_scatter_style_set_size1_gradient_range (ss, x);
}

static void
spin2_activate_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  gboolean x = gtk_toggle_button_get_active (button);
  guppi_scatter_style_set_allow_size2_gradient (ss, x);
}

static void
spin2_range_cb (GtkSpinButton * spinner, GuppiScatterStyle * ss)
{
  double x = gtk_spin_button_get_value_as_float (spinner);
  guppi_scatter_style_set_size2_gradient_range (ss, x);
}

static void
cg_none_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_reset (cg);
}

static void
cg_transition_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);
  GnomeColorPicker *picker;
  guint8 r, g, b, a;

  if (!gtk_toggle_button_get_active (button))
    return;

  picker =
    GNOME_COLOR_PICKER (gtk_object_get_user_data (GTK_OBJECT (button)));
  gnome_color_picker_get_i8 (picker, &r, &g, &b, &a);

  guppi_color_gradient_set2 (cg,
			     guppi_style_color (GUPPI_STYLE (ss)),
			     RGBA_TO_UINT (r, g, b, a));
}

static void
cg_transition_color_picker_cb (GnomeColorPicker * picker,
			       guint r, guint g, guint b, guint a,
			       GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (guppi_color_gradient_type (cg) != GUPPI_COLOR_GRADIENT_TWO)
    return;

  guppi_color_gradient_set2 (cg,
			     guppi_style_color (GUPPI_STYLE (ss)),
			     RGBA_TO_UINT (r >> 8, g >> 8, b >> 8, a >> 8));
}


static void
cg_fade_in_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_fade_in (cg, guppi_style_color (GUPPI_STYLE (ss)));
}

static void
cg_fade_out_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_fade_out (cg,
				     guppi_style_color (GUPPI_STYLE (ss)));
}


static void
cg_spectrum_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_spectrum (cg,
				     guppi_color_gradient_intensity (cg),
				     guppi_color_gradient_alpha (cg));
}

static void
cg_thermal_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_thermal (cg,
				    guppi_color_gradient_intensity (cg),
				    guppi_color_gradient_alpha (cg));
}

static void
cg_fire_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_fire (cg,
				 guppi_color_gradient_intensity (cg),
				 guppi_color_gradient_alpha (cg));
}

static void
cg_ice_cb (GtkToggleButton * button, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);

  if (!gtk_toggle_button_get_active (button))
    return;

  guppi_color_gradient_set_ice (cg,
				guppi_color_gradient_intensity (cg),
				guppi_color_gradient_alpha (cg));
}

static void
intensity_spinner_cb (GtkSpinButton * spinner, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);
  double x = gtk_spin_button_get_value_as_float (spinner);
  guppi_color_gradient_set_intensity (cg, x);
}

static void
alpha_spinner_cb (GtkSpinButton * spinner, GuppiScatterStyle * ss)
{
  GuppiColorGradient *cg = guppi_scatter_style_color_gradient (ss);
  double x = gtk_spin_button_get_value_as_float (spinner);
  guppi_color_gradient_set_alpha (cg, x);
}

static void
connect_signals (GuppiScatterStyle * ss, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "size1_activate_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (spin1_activate_cb), ss);

  w = glade_xml_get_widget (xml, "size1_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (spin1_range_cb), ss);

  w = glade_xml_get_widget (xml, "size2_activate_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (spin2_activate_cb), ss);

  w = glade_xml_get_widget (xml, "size2_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (spin2_range_cb), ss);

  w = glade_xml_get_widget (xml, "none_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_none_cb), ss);

  w = glade_xml_get_widget (xml, "transition_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_transition_cb), ss);

  w = glade_xml_get_widget (xml, "transition_color_picker");
  gtk_signal_connect (GTK_OBJECT (w), "color_set",
		      GTK_SIGNAL_FUNC (cg_transition_color_picker_cb), ss);

  w = glade_xml_get_widget (xml, "fade_in_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_fade_in_cb), ss);

  w = glade_xml_get_widget (xml, "fade_out_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_fade_out_cb), ss);

  w = glade_xml_get_widget (xml, "spectrum_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_spectrum_cb), ss);

  w = glade_xml_get_widget (xml, "thermal_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_thermal_cb), ss);

  w = glade_xml_get_widget (xml, "fire_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_fire_cb), ss);

  w = glade_xml_get_widget (xml, "ice_button");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (cg_ice_cb), ss);

  w = glade_xml_get_widget (xml, "intensity_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (intensity_spinner_cb), ss);

  w = glade_xml_get_widget (xml, "alpha_spinner");
  gtk_signal_connect (GTK_OBJECT (w), "changed",
		      GTK_SIGNAL_FUNC (alpha_spinner_cb), ss);

}

static void
destroy_cb (GtkWidget * w, GladeXML * xml)
{
  GuppiScatterStyle *ss =
    GUPPI_SCATTER_STYLE (gtk_object_get_user_data (GTK_OBJECT (w)));
  gtk_signal_disconnect_by_func (GTK_OBJECT (ss),
				 GTK_SIGNAL_FUNC (push_state_to_widget), xml);
}

static GtkWidget *
edit (GuppiStyle * style)
{
  GuppiScatterStyle *ss;
  const gchar *xml_path;
  GladeXML *xml;
  GtkWidget *w;
  GtkWidget *empty_box;

  g_return_val_if_fail (style != NULL, NULL);

  ss = GUPPI_SCATTER_STYLE (style);

  xml_path = guppi_glade_path ("guppi-scatter-style.glade");
  if (xml_path == NULL) {
    g_warning ("File guppi-scatter-style.glade not found.");
    return NULL;
  }

  xml = glade_xml_new (xml_path, "scatter_style");
  if (xml == NULL) {
    g_warning ("Glade could not process guppi-scatter-style.glade.");
    return NULL;
  }

  /* Insert widget to edit "base class" parts of GuppiScatterStyle */
  empty_box = glade_xml_get_widget (xml, "empty_box");
  if (empty_box != NULL) {
    GtkWidget *(*edit_widget_fn) (GuppiStyle *);
    GtkWidget *ew;

    edit_widget_fn = GUPPI_STYLE_CLASS (parent_class)->edit_widget;

    ew = edit_widget_fn (style);
    gtk_container_add (GTK_CONTAINER (empty_box), ew);
  } else {
    g_warning ("Couldn't build GuppiScatterStyle parent class edit widget.");
  }

  /* A little hackiness */
  gtk_object_set_user_data (GTK_OBJECT
			    (glade_xml_get_widget (xml, "transition_button")),
			    glade_xml_get_widget (xml,
						  "transition_color_picker"));

  /* Connect all widget signals. */

  connect_signals (ss, xml);

  /* On a changed signal, push out our style's state to the widget
     to keep the UI in sync. */

  gtk_signal_connect (GTK_OBJECT (ss),
		      "changed", GTK_SIGNAL_FUNC (push_state_to_widget), xml);

  /* A little more hackiness, to make sure that this signal gets
     disconnected when our widget is destroyed. */
  w = glade_xml_get_widget (xml, "scatter_style");
  gtk_object_set_user_data (GTK_OBJECT (w), ss);
  gtk_signal_connect (GTK_OBJECT (w),
		      "destroy", GTK_SIGNAL_FUNC (destroy_cb), xml);

  push_state_to_widget (ss, xml);	/* Initial sync. */


  /* Return our top-level widget, which is still in w */

  return w;
}

/****************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiScatterStyle::" str, t, GTK_ARG_READWRITE, symb)

static void
guppi_scatter_style_class_init (GuppiScatterStyleClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiStyleClass *style_class = GUPPI_STYLE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_STYLE);

  object_class->get_arg = guppi_scatter_style_get_arg;
  object_class->set_arg = guppi_scatter_style_set_arg;
  object_class->finalize = guppi_scatter_style_finalize;
  style_class->edit_widget = edit;

  add_arg ("allow_color_gradient", GTK_TYPE_BOOL, ARG_ALLOW_COLOR_GRAD);
  add_arg ("reverse_color_gradient", GTK_TYPE_BOOL, ARG_REV_COLOR_GRAD);
  add_arg ("color_gradient", GTK_TYPE_POINTER, ARG_COLOR_GRAD);
  add_arg ("allow_size1_gradient", GTK_TYPE_BOOL, ARG_ALLOW_SIZE1_GRAD);
  add_arg ("reverse_size1_gradient", GTK_TYPE_BOOL, ARG_REV_SIZE1_GRAD);
  add_arg ("size1_range", GTK_TYPE_DOUBLE, ARG_SIZE1_RANGE);
  add_arg ("allow_size2_gradient", GTK_TYPE_BOOL, ARG_ALLOW_SIZE2_GRAD);
  add_arg ("reverse_size2_gradient", GTK_TYPE_BOOL, ARG_REV_SIZE2_GRAD);
  add_arg ("size2_range", GTK_TYPE_DOUBLE, ARG_SIZE2_RANGE);
}

static void
guppi_scatter_style_init (GuppiScatterStyle * obj)
{
  obj->allow_color_grad = TRUE;

  /* A nice default color gradient */
  obj->color_grad = guppi_color_gradient_new ();
  gtk_signal_connect_object (GTK_OBJECT (obj->color_grad),
			     "changed",
			     GTK_SIGNAL_FUNC (guppi_style_changed_delayed),
			     GTK_OBJECT (obj));
  guppi_color_gradient_set_thermal (obj->color_grad, 1.0, 1.0);

  obj->allow_size1_grad = TRUE;
  obj->size1_range = 0.5;

  obj->allow_size2_grad = TRUE;
  obj->size2_range = 0.5;
}

GtkType guppi_scatter_style_get_type (void)
{
  static GtkType guppi_scatter_style_type = 0;
  if (!guppi_scatter_style_type) {
    static const GtkTypeInfo guppi_scatter_style_info = {
      "GuppiScatterStyle",
      sizeof (GuppiScatterStyle),
      sizeof (GuppiScatterStyleClass),
      (GtkClassInitFunc) guppi_scatter_style_class_init,
      (GtkObjectInitFunc) guppi_scatter_style_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_scatter_style_type = gtk_type_unique (GUPPI_TYPE_STYLE,
						&guppi_scatter_style_info);
  }
  return guppi_scatter_style_type;
}

/*****************************************************************************/

GuppiScatterStyle *
guppi_scatter_style_new (void)
{
  return GUPPI_SCATTER_STYLE (guppi_type_new (guppi_scatter_style_get_type ()));
}

GuppiScatterStyle *
guppi_scatter_style_stock (gint i)
{
  GuppiStyle *sty = guppi_style_stock (i, 1);
  GuppiScatterStyle *ss = guppi_scatter_style_new ();

  if (sty != NULL)
    guppi_style_copy (GUPPI_STYLE (ss), sty);

  return ss;
}

/*****************************************************************************/

void
guppi_scatter_style_set_allow_style_gradient (GuppiScatterStyle * ss,
					      gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->allow_style_grad != x) {
    ss->allow_style_grad = x;

    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_reverse_style_gradient (GuppiScatterStyle * ss,
						gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->rev_style_grad != x) {
    ss->rev_style_grad = x;

    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_gradient_style (GuppiScatterStyle * ss,
					GuppiStyle * sty)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));
  g_return_if_fail (sty != NULL);
  g_return_if_fail (GUPPI_IS_STYLE (sty));

  if (ss->grad_style != sty) {

    if (ss->grad_style != NULL) {
      gtk_signal_disconnect_by_func (GTK_OBJECT (ss->grad_style),
				     guppi_style_changed, ss);
      guppi_unref (ss->grad_style);
    }

    ss->grad_style = sty;
    guppi_ref (ss->grad_style);
    gtk_signal_connect_object (GTK_OBJECT (ss->grad_style),
			       "changed",
			       GTK_SIGNAL_FUNC (guppi_style_changed),
			       GTK_OBJECT (ss));

    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_allow_color_gradient (GuppiScatterStyle * ss,
					      gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->allow_color_grad != x) {
    ss->allow_color_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_reverse_color_gradient (GuppiScatterStyle * ss,
						gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->rev_color_grad != x) {
    ss->rev_color_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_color_gradient (GuppiScatterStyle * ss,
					GuppiColorGradient * cg)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->color_grad != cg) {

    if (ss->color_grad) {

      gtk_signal_disconnect_by_func (GTK_OBJECT (ss->color_grad),
				     GTK_SIGNAL_FUNC
				     (guppi_style_changed_delayed), ss);

      guppi_unref (ss->color_grad);
    }

    ss->color_grad = cg;

    if (ss->color_grad) {

      guppi_ref (ss->color_grad);

      gtk_signal_connect_object (GTK_OBJECT (ss->color_grad),
				 "changed",
				 GTK_SIGNAL_FUNC
				 (guppi_style_changed_delayed),
				 GTK_OBJECT (ss));

    }

    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}



void
guppi_scatter_style_set_allow_size1_gradient (GuppiScatterStyle * ss,
					      gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->allow_size1_grad != x) {
    ss->allow_size1_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_reverse_size1_gradient (GuppiScatterStyle * ss,
						gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->rev_size1_grad != x) {
    ss->rev_size1_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_size1_gradient_range (GuppiScatterStyle * ss,
					      double x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));
  g_return_if_fail (0 <= x && x <= 1);

  if (ss->size1_range != x) {
    ss->size1_range = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_allow_size2_gradient (GuppiScatterStyle * ss,
					      gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->allow_size2_grad != x) {
    ss->allow_size2_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_reverse_size2_gradient (GuppiScatterStyle * ss,
						gboolean x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));

  if (ss->rev_size2_grad != x) {
    ss->rev_size2_grad = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}

void
guppi_scatter_style_set_size2_gradient_range (GuppiScatterStyle * ss,
					      double x)
{
  g_return_if_fail (ss != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE (ss));
  g_return_if_fail (0 <= x && x <= 1);

  if (ss->size2_range != x) {
    ss->size2_range = x;
    guppi_style_changed_delayed (GUPPI_STYLE (ss));
  }
}


/****************************************************************************/

guint32
guppi_scatter_style_calc_color_gradient (GuppiScatterStyle * ss, double t)
{
  GuppiColorGradient *cg;

  g_return_val_if_fail (ss != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_SCATTER_STYLE (ss), 0);

  cg = guppi_scatter_style_color_gradient (ss);

  if (t < 0 || t > 1 || !guppi_scatter_style_allow_color_gradient (ss) ||
      cg == NULL || guppi_color_gradient_size (cg) < 1)
    return guppi_style_color (GUPPI_STYLE (ss));

  if (guppi_scatter_style_reverse_color_gradient (ss))
    t = 1 - t;


  return guppi_color_gradient_value (cg, t);
}

double
guppi_scatter_style_calc_size1_gradient (GuppiScatterStyle * ss, double t)
{
  double a, b;

  g_return_val_if_fail (ss != NULL, -1);
  g_return_val_if_fail (GUPPI_IS_SCATTER_STYLE (ss), -1);

  if (t < 0 || t > 1 || !guppi_scatter_style_allow_size1_gradient (ss))
    return guppi_style_marker_size1 (GUPPI_STYLE (ss));

  if (guppi_scatter_style_reverse_size1_gradient (ss))
    t = 1 - t;

  a = guppi_scatter_style_size1_gradient_min (ss);
  b = guppi_scatter_style_size1_gradient_max (ss);

  return a + t * (b - a);
}

double
guppi_scatter_style_calc_size2_gradient (GuppiScatterStyle * ss, double t)
{
  double a, b;

  g_return_val_if_fail (ss != NULL, -1);
  g_return_val_if_fail (GUPPI_IS_SCATTER_STYLE (ss), -1);

  if (t < 0 || t > 1 || !guppi_scatter_style_allow_size2_gradient (ss))
    return guppi_style_marker_size2 (GUPPI_STYLE (ss));

  if (guppi_scatter_style_reverse_size2_gradient (ss))
    t = 1 - t;

  a = guppi_scatter_style_size2_gradient_min (ss);
  b = guppi_scatter_style_size2_gradient_max (ss);

  return a + t * (b - a);
}

/* $Id: guppi-scatter-style.c,v 1.17 2000/12/14 20:22:59 trow Exp $ */
