/* $Id: guppi-scatter-state.c,v 1.20 2000/12/17 05:24:54 trow Exp $ */

/*
 * guppi-scatter-state.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <glade/glade.h>
#include <gtk/gtkmarshal.h>
#include <guppi-convenient.h>
#include <guppi-useful.h>
#include <guppi-data-select.h>
#include <guppi-seq-integer.h>
#include "guppi-scatter-state.h"
#include "guppi-scatter-view.h"

enum {
  ARG_0,
  ARG_X_DATA,
  ARG_Y_DATA,
  ARG_SIZE1_DATA,
  ARG_SIZE2_DATA
};

enum {
  CHANGED_STYLE,
  LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;
static guint gss_signals[LAST_SIGNAL] = { 0 };

static void
guppi_scatter_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiScatterState *state = GUPPI_SCATTER_STATE (obj);

  switch (arg_id) {

  case ARG_X_DATA:
    GTK_VALUE_POINTER (*arg) = guppi_scatter_state_x_data (state);
    break;

  case ARG_Y_DATA:
    GTK_VALUE_POINTER (*arg) = guppi_scatter_state_y_data (state);
    break;

  case ARG_SIZE1_DATA:
    GTK_VALUE_POINTER (*arg) =
      guppi_scatter_state_size1_gradient_data (state);
    break;

  case ARG_SIZE2_DATA:
    GTK_VALUE_POINTER (*arg) =
      guppi_scatter_state_size1_gradient_data (state);
    break;

  default:
    break;
  }
}

static void
guppi_scatter_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiScatterState *state = GUPPI_SCATTER_STATE (obj);
  GuppiSeqScalar *data;

  switch (arg_id) {

  case ARG_X_DATA:
    data = GUPPI_SEQ_SCALAR0 (GTK_VALUE_POINTER (*arg));
    guppi_scatter_state_set_x_data (state, data);
    break;

  case ARG_Y_DATA:
    data = GUPPI_SEQ_SCALAR0 (GTK_VALUE_POINTER (*arg));
    guppi_scatter_state_set_y_data (state, data);
    break;

  case ARG_SIZE1_DATA:
    data = GUPPI_SEQ_SCALAR0 (GTK_VALUE_POINTER (*arg));
    guppi_scatter_state_set_size1_gradient_data (state, data);
    break;

  case ARG_SIZE2_DATA:
    data = GUPPI_SEQ_SCALAR0 (GTK_VALUE_POINTER (*arg));
    guppi_scatter_state_set_size2_gradient_data (state, data);
    break;

  default:
    break;
  }
}

static void
guppi_scatter_state_destroy (GtkObject * obj)
{
  GuppiScatterState *gss = GUPPI_SCATTER_STATE (obj);
  gint i;

  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {
    guppi_unref (gss->style[i]);
  }

  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_scatter_state_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/****************************************************************************/

static GuppiElementView *
make_view (GuppiElementState * state)
{
  return GUPPI_ELEMENT_VIEW (guppi_type_new (GUPPI_TYPE_SCATTER_VIEW));
}

/****************************************************************************/

static void
push_state_to_widget (GuppiScatterState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "x_data");
  guppi_data_select_set_selected_data (GUPPI_DATA_SELECT (w),
				       GUPPI_DATA0 (guppi_scatter_state_x_data
						    (state)));


  w = glade_xml_get_widget (xml, "y_data");
  guppi_data_select_set_selected_data (GUPPI_DATA_SELECT (w),
				       GUPPI_DATA0 (guppi_scatter_state_y_data
						    (state)));
}

static void
x_data_cb (GuppiDataSelect * picker, GuppiData * data,
	   GuppiScatterState * state)
{
  guppi_scatter_state_set_x_data (state, GUPPI_SEQ_SCALAR0 (data));
}

static void
y_data_cb (GuppiDataSelect * picker, GuppiData * data,
	   GuppiScatterState * state)
{
  guppi_scatter_state_set_y_data (state, GUPPI_SEQ_SCALAR0 (data));
}

static void
sz1_data_cb (GuppiDataSelect * picker, GuppiData * data,
	     GuppiScatterState * state)
{
  guppi_scatter_state_set_size1_gradient_data (state,
					       GUPPI_SEQ_SCALAR0 (data));
}

static void
sz2_data_cb (GuppiDataSelect * picker, GuppiData * data,
	     GuppiScatterState * state)
{
  guppi_scatter_state_set_size2_gradient_data (state,
					       GUPPI_SEQ_SCALAR0 (data));
}

static void
c_data_cb (GuppiDataSelect * picker, GuppiData * data,
	   GuppiScatterState * state)
{
  guppi_scatter_state_set_color_gradient_data (state,
					       GUPPI_SEQ_SCALAR0 (data));
}

static void
connect_signals (GuppiScatterState * state, GladeXML * xml)
{
  GtkWidget *w;

  w = glade_xml_get_widget (xml, "x_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (x_data_cb), state);

  w = glade_xml_get_widget (xml, "y_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (y_data_cb), state);

  w = glade_xml_get_widget (xml, "sz1_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (sz1_data_cb), state);

  w = glade_xml_get_widget (xml, "sz2_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (sz2_data_cb), state);

  w = glade_xml_get_widget (xml, "color_data");
  gtk_signal_connect (GTK_OBJECT (w), "selected_data",
		      GTK_SIGNAL_FUNC (c_data_cb), state);
}

static void
destroy_handler (GtkWidget * w, GladeXML * xml)
{
  GuppiScatterState *state;

  state = GUPPI_SCATTER_STATE (gtk_object_get_user_data (GTK_OBJECT (w)));

  gtk_signal_disconnect_by_func (GTK_OBJECT (state),
				 push_state_to_widget, xml);
}

/****************************************************************************/

static GtkWidget *
config_cb (gpointer user_data)
{
  GuppiScatterState *state = GUPPI_SCATTER_STATE (user_data);
  const gchar *xml_path;
  GladeXML *xml;
  GtkWidget *empty_box;
  GtkWidget *w;

  xml_path = guppi_glade_path ("guppi-scatter-state.glade");
  if (xml_path == NULL) {
    g_warning ("File guppi-scatter-scatter.glade not found.");
    return NULL;
  }

  xml = glade_xml_new (xml_path, "scatter_state");
  if (xml == NULL) {
    g_warning ("Glade could not process guppi-scatter-state.glade.");
    return NULL;
  }

  /* Put our style state widgets inside of the empty box */
  empty_box = glade_xml_get_widget (xml, "empty_box");
  if (empty_box != NULL) {
    GtkNotebook *notebook = GTK_NOTEBOOK (gtk_notebook_new ());
    gchar *s;
    gint i;

    for (i = 0; i < MIN (4, GUPPI_SCATTER_STATE_STYLE_COUNT); ++i) {
      GuppiScatterStyle *sty = guppi_scatter_state_get_style (state, i);
      GtkWidget *sew = guppi_style_edit_widget (GUPPI_STYLE (sty));

      s = guppi_strdup_printf (_("Style #%d"), i + 1);
      gtk_notebook_append_page (notebook, sew, gtk_label_new (s));
      guppi_free (s);
    }

    gtk_container_add (GTK_CONTAINER (empty_box), GTK_WIDGET (notebook));
    gtk_widget_show_all (GTK_WIDGET (notebook));
  }

  push_state_to_widget (state, xml);	/* Initialize */

  connect_signals (state, xml);

  w = glade_xml_get_widget (xml, "scatter_state");

  gtk_signal_connect (GTK_OBJECT (state), "changed",
		      GTK_SIGNAL_FUNC (push_state_to_widget), xml);

  gtk_object_set_user_data (GTK_OBJECT (w), state);

  gtk_signal_connect (GTK_OBJECT (w), "destroy",
		      GTK_SIGNAL_FUNC (destroy_handler), xml);

  return w;


}

static GuppiConfigItem *
config_tree (GuppiElementState * state)
{
  GuppiConfigItem *top_node;

  top_node = guppi_config_item_new (_("Configure Scatter Plot"),
				    _("Configure"), config_cb, state);
  return top_node;
}

/****************************************************************************/

static void
guppi_scatter_state_class_init (GuppiScatterStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  gss_signals[CHANGED_STYLE] =
    gtk_signal_new ("changed_style",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiScatterStateClass, changed_style),
		    gtk_marshal_NONE__INT, GTK_TYPE_NONE, 1, GTK_TYPE_INT);

  gtk_object_class_add_signals (object_class, gss_signals, LAST_SIGNAL);

  object_class->set_arg = guppi_scatter_state_set_arg;
  object_class->get_arg = guppi_scatter_state_get_arg;
  object_class->destroy = guppi_scatter_state_destroy;
  object_class->finalize = guppi_scatter_state_finalize;

  state_class->name = _("Scatter Plot");

  state_class->make_view = make_view;
  state_class->config_tree = config_tree;

  gtk_object_add_arg_type ("GuppiScatterState::x_data",
			   GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_X_DATA);
  gtk_object_add_arg_type ("GuppiScatterState::y_data",
			   GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_Y_DATA);
  gtk_object_add_arg_type ("GuppiScatterState::size1_data",
			   GTK_TYPE_POINTER, GTK_ARG_READWRITE,
			   ARG_SIZE1_DATA);
  gtk_object_add_arg_type ("GuppiScatterState::size2_data", GTK_TYPE_POINTER,
			   GTK_ARG_READWRITE, ARG_SIZE2_DATA);
}

static void
data_changed (GuppiShared * sh, GuppiScatterState * ss)
{
  static void grow_style_data (GuppiScatterState *, GuppiSeq *);
  GuppiSeq *seq = GUPPI_SEQ0 (guppi_shared_get (sh));

  grow_style_data (ss, seq);
}


static void
guppi_scatter_state_init (GuppiScatterState * ss)
{
  GuppiElementState *state = GUPPI_ELEMENT_STATE (ss);
  GuppiShared *xsh;
  GuppiShared *ysh;
  gint i;

  guppi_element_state_add_shared (state, SHARED_X_DATA,
				  xsh = guppi_shared_data ());
  guppi_element_state_add_shared (state, SHARED_Y_DATA,
				  ysh = guppi_shared_data ());

  gtk_signal_connect_while_alive (GTK_OBJECT (xsh),
				  "changed",
				  GTK_SIGNAL_FUNC (data_changed),
				  ss, GTK_OBJECT (ss));
  gtk_signal_connect_while_alive (GTK_OBJECT (ysh),
				  "changed",
				  GTK_SIGNAL_FUNC (data_changed),
				  ss, GTK_OBJECT (ss));

  guppi_element_state_add_shared (state, SHARED_STYLE_DATA,
				  guppi_shared_data ());

  guppi_element_state_add_shared (state, SHARED_COLOR_DATA,
				  guppi_shared_data ());

  guppi_element_state_add_shared (state, SHARED_SIZE1_DATA,
				  guppi_shared_data ());
  guppi_element_state_add_shared (state, SHARED_SIZE2_DATA,
				  guppi_shared_data ());


  for (i = 0; i < GUPPI_SCATTER_STATE_STYLE_COUNT; ++i) {
    ss->style[i] = guppi_scatter_style_stock (i);
    guppi_ref (ss->style[i]);
    guppi_sink (ss->style[i]);

    gtk_signal_connect_object (GTK_OBJECT (ss->style[i]),
			       "changed",
			       GTK_SIGNAL_FUNC
			       (guppi_element_state_changed_delayed),
			       GTK_OBJECT (ss));
  }
}

GtkType guppi_scatter_state_get_type (void)
{
  static GtkType gss_type = 0;

  if (!gss_type) {
    static const GtkTypeInfo gss_info = {
      "GuppiScatterState",
      sizeof (GuppiScatterState),
      sizeof (GuppiScatterStateClass),
      (GtkClassInitFunc) guppi_scatter_state_class_init,
      (GtkObjectInitFunc) guppi_scatter_state_init,
      NULL, NULL,
      (GtkClassInitFunc) NULL
    };
    gss_type = gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &gss_info);
  }

  return gss_type;
}

GuppiElementState *
guppi_scatter_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_scatter_state_get_type ()));
}

/**************************************************************************/

/* Convenience functions */

GuppiSeqScalar *
guppi_scatter_state_x_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_X_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

GuppiSeqScalar *
guppi_scatter_state_y_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_Y_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

GuppiSeqInteger *
guppi_scatter_state_style_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_STYLE_DATA);
  return GUPPI_SEQ_INTEGER0 (obj);
}

GuppiSeqScalar *
guppi_scatter_state_color_gradient_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_COLOR_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

GuppiSeqScalar *
guppi_scatter_state_size1_gradient_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_SIZE1_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

GuppiSeqScalar *
guppi_scatter_state_size2_gradient_data (GuppiScatterState * ss)
{
  GtkObject *obj;
  g_return_val_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss), NULL);
  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (ss),
					SHARED_SIZE2_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

static void
grow_style_data (GuppiScatterState * ss, GuppiSeq * seq)
{
  GuppiSeqInteger *style_data;
  gint i, i0, i1, j0, j1;

  if (seq == NULL || guppi_seq_size (seq) == 0)
    return;

  guppi_seq_indices (GUPPI_SEQ (seq), &j0, &j1);

  style_data = guppi_scatter_state_style_data (ss);
  if (style_data == NULL) {
    style_data = GUPPI_SEQ_INTEGER (guppi_seq_integer_new ());
    guppi_scatter_state_set_style_data (ss, style_data);
    guppi_unref (style_data);
    guppi_seq_integer_append (style_data, 0);
  }

  guppi_seq_indices (GUPPI_SEQ (style_data), &i0, &i1);


  /* This is very hacky and *very* inefficient */
  for (i = 0; i < j1 - i1; ++i)
    guppi_seq_integer_append (style_data, 0);
  for (i = 0; i < i0 - j0; ++i) {
    guppi_seq_integer_prepend (style_data, 0);
    guppi_seq_set_min_index (GUPPI_SEQ (style_data), j0);
  }
}

void
guppi_scatter_state_set_x_data (GuppiScatterState * ss, GuppiSeqScalar * sd)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_X_DATA, sd);

  grow_style_data (ss, GUPPI_SEQ0 (sd));
}

void
guppi_scatter_state_set_y_data (GuppiScatterState * ss, GuppiSeqScalar * sd)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_Y_DATA, sd);

  grow_style_data (ss, GUPPI_SEQ0 (sd));
}

void
guppi_scatter_state_set_style_data (GuppiScatterState * ss,
				    GuppiSeqInteger * id)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_STYLE_DATA, id);
}

void
guppi_scatter_state_set_color_gradient_data (GuppiScatterState * ss,
					     GuppiSeqScalar * sd)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_COLOR_DATA, sd);
}

void
guppi_scatter_state_set_size1_gradient_data (GuppiScatterState * ss,
					     GuppiSeqScalar * sd)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_SIZE1_DATA, sd);
}

void
guppi_scatter_state_set_size2_gradient_data (GuppiScatterState * ss,
					     GuppiSeqScalar * sd)
{
  g_return_if_fail (ss != NULL && GUPPI_IS_SCATTER_STATE (ss));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (ss),
				  SHARED_SIZE2_DATA, sd);
}


static void
style_change_cb (GuppiScatterStyle * style, GuppiScatterState * state)
{
  /* a bad hack */
  gtk_signal_emit (GTK_OBJECT (state), gss_signals[CHANGED_STYLE], 0);

  guppi_element_state_changed (GUPPI_ELEMENT_STATE (state));
}

GuppiScatterStyle *
guppi_scatter_state_get_style (GuppiScatterState * ss, gint i)
{
  g_return_val_if_fail (ss != NULL, NULL);
  g_return_val_if_fail (0 <= i && i < GUPPI_SCATTER_STATE_STYLE_COUNT, NULL);

  if (ss->style[i] == NULL) {
    ss->style[i] = GUPPI_SCATTER_STYLE (guppi_scatter_style_stock (i));
    gtk_signal_connect (GTK_OBJECT (ss->style[i]),
			"changed", GTK_SIGNAL_FUNC (style_change_cb), ss);
  }

  return ss->style[i];
}

static void
guppi_scatter_state_brush_from_boolean_data (GuppiScatterState * ss,
					     gint brush_number,
					     GuppiSeqBoolean * bd)
{
  GuppiSeqInteger *style_data;
  gint i, imax;

  g_return_if_fail (ss);
  g_return_if_fail (bd);

  g_return_if_fail (brush_number >= 0);
  g_return_if_fail (brush_number < GUPPI_SCATTER_STATE_STYLE_COUNT);

  style_data = guppi_scatter_state_style_data (ss);
  g_assert (style_data != NULL);

  guppi_seq_common_bounds (GUPPI_SEQ (bd), GUPPI_SEQ (style_data), &i, &imax);

  if (!guppi_seq_boolean_get (bd, i))
    i = guppi_seq_boolean_next_true (bd, i);

  while (i <= imax) {
    guppi_seq_integer_set (style_data, i, brush_number);
    i = guppi_seq_boolean_next_true (bd, i);
  }
}

gboolean
guppi_scatter_state_closest_point (GuppiScatterState * ss,
				   double x, double y,
				   double r,
				   double x_scale, double y_scale,
				   gint * index)
{
  GuppiSeqBoolean *bd = NULL;
  gint hits;
  gint i, imax, min_i = 0;
  double xd, yd, dist, min_dist = 1e+12;
  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;

  g_return_val_if_fail (index != NULL, FALSE);
  g_return_val_if_fail (ss != NULL, FALSE);
  g_return_val_if_fail (r >= 0, FALSE);

  x_data = guppi_scatter_state_x_data (ss);
  y_data = guppi_scatter_state_y_data (ss);

  g_return_val_if_fail (x_data != NULL && y_data != NULL, FALSE);

  bd = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new_aligned (GUPPI_SEQ (x_data)));
  hits = guppi_seq_scalar_in_place_range_query (x_data, bd,
						x - r * x_scale,
						x + r * x_scale);

  if (hits == 0) {
    guppi_unref (bd);
    return FALSE;
  }

  hits = guppi_seq_scalar_bitwise_and_range_query (y_data, bd,
						   y - r * y_scale,
						   y + r * y_scale);

  if (hits == 0) {
    guppi_unref (bd);
    return FALSE;
  }

  imax = guppi_seq_max_index (GUPPI_SEQ (bd));
  for (i = guppi_seq_boolean_first_true (bd); i <= imax;
       i = guppi_seq_boolean_next_true (bd, i)) {
    xd = (guppi_seq_scalar_get (x_data, i) - x) / x_scale;
    yd = (guppi_seq_scalar_get (y_data, i) - y) / y_scale;
    dist = xd * xd + yd * yd;
    if (dist < min_dist) {
      min_dist = dist;
      min_i = i;
    }
  }

  *index = min_i;

  guppi_unref (bd);
  return TRUE;

}

void
guppi_scatter_state_brush_rectangle (GuppiScatterState * ss,
				     gint brush_number,
				     double x0, double y0,
				     double x1, double y1)
{
  GuppiSeqBoolean *bd = NULL;
  gint hits;
  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;

  g_return_if_fail (ss != NULL);

  x_data = guppi_scatter_state_x_data (ss);
  y_data = guppi_scatter_state_y_data (ss);

  if (x_data == NULL || y_data == NULL)
    return;

  bd = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new_aligned (GUPPI_SEQ (x_data)));
  hits = guppi_seq_scalar_in_place_range_query (x_data, bd, x0, x1);

  if (hits) {
    hits = guppi_seq_scalar_bitwise_and_range_query (y_data, bd, y0, y1);
  }

  if (hits) {
    guppi_scatter_state_brush_from_boolean_data (ss, brush_number, bd);
  }

  guppi_unref (bd);
}

/* Mostly code dup from above */
void
guppi_scatter_state_brush_circle (GuppiScatterState * ss,
				  gint brush_number,
				  double x, double y, double r,
				  double x_scale, double y_scale)
{
  GuppiSeqBoolean *bd = NULL;
  gint hits;
  gint i, imax;
  double xd, yd, rr;
  GuppiSeqScalar *x_data;
  GuppiSeqScalar *y_data;

  g_return_if_fail (ss != NULL);

  x_data = guppi_scatter_state_x_data (ss);
  y_data = guppi_scatter_state_y_data (ss);

  if (x_data == NULL || y_data == NULL)
    return;

  r = abs (r);
  rr = r * r;

  bd = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new_aligned (GUPPI_SEQ (x_data)));
  hits = guppi_seq_scalar_in_place_range_query (x_data,
						bd,
						x - r * x_scale,
						x + r * x_scale);

  if (hits) {
    hits = guppi_seq_scalar_bitwise_and_range_query (y_data,
						     bd,
						     y - r * y_scale,
						     y + r * y_scale);
  }

  if (hits) {

    /* blow away anything outside of the radius */
    imax = guppi_seq_max_index (GUPPI_SEQ (bd));
    for (i = guppi_seq_boolean_first_true (bd); i <= imax;
	 i = guppi_seq_boolean_next_true (bd, i)) {
      xd = (guppi_seq_scalar_get (x_data, i) - x) / x_scale;
      yd = (guppi_seq_scalar_get (y_data, i) - y) / y_scale;
      if (xd * xd + yd * yd > rr)
	guppi_seq_boolean_set (bd, i, FALSE);
    }

    guppi_scatter_state_brush_from_boolean_data (ss, brush_number, bd);
  }

  guppi_unref (bd);
}




/* $Id: guppi-scatter-state.c,v 1.20 2000/12/17 05:24:54 trow Exp $ */
