/* This is -*- C -*- */
/* $Id: func.c,v 1.2 2000/12/14 20:22:50 trow Exp $ */

/*
 * guppi-curve-func-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <guppi-useful.h>
#include <guppi-curve.h>
#include <guppi-data-impl-plug-in.h>
#include "func.h"
#include "scm-curve-func.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_CURVE,
  ARG_REPARAMETERIZE,
  ARG_T0,
  ARG_T1,
  ARG_T_FUNC,
  ARG_X_FUNC,
  ARG_Y_FUNC,
  ARG_X_BINARY_FUNC,
  ARG_Y_BINARY_FUNC,
  ARG_USE_2x2_MATRIX,
  ARG_MATRIX_A,
  ARG_MATRIX_B,
  ARG_MATRIX_C,
  ARG_MATRIX_D
};

static void
guppi_curve_func_impl_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_curve_func_impl_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiCurveFuncImpl *func = GUPPI_CURVE_FUNC_IMPL (obj);
  GuppiCurve *c;
  gboolean rp;
  double t;
  GuppiFnWrapper *fw;

  switch (arg_id) {

  case ARG_CURVE:
    c = GUPPI_CURVE (GTK_VALUE_POINTER (*arg));
    if (func->base_curve != c) {
      guppi_refcounting_assign (func->base_curve, c);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    };
    break;

  case ARG_REPARAMETERIZE:
    rp = GTK_VALUE_BOOL (*arg);
    if (func->reparameterize != rp) {
      func->reparameterize = rp;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_T0:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->t0 != t || !func->reparameterize) {
      func->t0 = t;
      func->reparameterize = TRUE;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_T1:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->t1 != t || !func->reparameterize) {
      func->t1 = t;
      func->reparameterize = TRUE;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_T_FUNC:
    fw = GUPPI_FN_WRAPPER (GTK_VALUE_POINTER (*arg));
    if (func->t_fn_wrapper != fw || !func->reparameterize) {
      func->reparameterize = TRUE;
      guppi_refcounting_assign (func->t_fn_wrapper, fw);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_X_FUNC:
    fw = GUPPI_FN_WRAPPER (GTK_VALUE_POINTER (*arg));
    if (func->x_fn_wrapper != fw) {
      guppi_refcounting_assign (func->x_fn_wrapper, fw);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_Y_FUNC:
    fw = GUPPI_FN_WRAPPER (GTK_VALUE_POINTER (*arg));
    if (func->y_fn_wrapper != fw) {
      guppi_refcounting_assign (func->y_fn_wrapper, fw);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_X_BINARY_FUNC:
    fw = GUPPI_FN_WRAPPER (GTK_VALUE_POINTER (*arg));
    if (func->x_binary_fn_wrapper != fw) {
      guppi_refcounting_assign (func->x_binary_fn_wrapper, fw);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_Y_BINARY_FUNC:
    fw = GUPPI_FN_WRAPPER (GTK_VALUE_POINTER (*arg));
    if (func->y_binary_fn_wrapper != fw) {
      guppi_refcounting_assign (func->y_binary_fn_wrapper, fw);
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_USE_2x2_MATRIX:
    rp = GTK_VALUE_BOOL (*arg);
    if (func->use_2x2_matrix != rp) {
      func->use_2x2_matrix = rp;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_MATRIX_A:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->a != t) {
      func->a = t;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_MATRIX_B:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->b != t) {
      func->b = t;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_MATRIX_C:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->c != t) {
      func->c = t;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;

  case ARG_MATRIX_D:
    t = GTK_VALUE_DOUBLE (*arg);
    if (func->d != t) {
      func->d = t;
      guppi_data_impl_changed (GUPPI_DATA_IMPL (obj));
    }
    break;


  default:
    break;
  };
}

static void
guppi_curve_func_impl_finalize (GtkObject * obj)
{
  GuppiCurveFuncImpl *func = GUPPI_CURVE_FUNC_IMPL (obj);

  guppi_unref0 (func->t_fn_wrapper);
  guppi_unref0 (func->x_fn_wrapper);
  guppi_unref0 (func->y_fn_wrapper);
  guppi_unref0 (func->x_binary_fn_wrapper);
  guppi_unref0 (func->y_binary_fn_wrapper);
  guppi_unref0 (func->base_curve);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/****************************************************************************/

static void
v_curve_bounds (GuppiCurveImpl * impl, double *a, double *b)
{
  GuppiCurveFuncImpl *cfi = GUPPI_CURVE_FUNC_IMPL (impl);

  if (cfi->reparameterize) {

  if (a)
    *a = cfi->t0;

  if (b)
    *b = cfi->t1;

  } else {
    guppi_curve_parameter_bounds (cfi->base_curve, a, b);
  }
}

static void
v_curve_get (GuppiCurveImpl * impl, double t, double *x, double *y)
{
  GuppiCurveFuncImpl *cfi = GUPPI_CURVE_FUNC_IMPL (impl);
  double xx, yy;

  if (cfi->reparameterize) {
    if (cfi->t_fn_wrapper) {
      t = guppi_fn_wrapper_eval_d__d (cfi->t_fn_wrapper, t);
      t = guppi_curve_clamp_parameter (cfi->base_curve, t);
    } else {
      t = guppi_curve_parameter_01 (cfi->base_curve, 
				    (t - cfi->t0)/(cfi->t1 - cfi->t0));
    }
  }

  guppi_curve_get (cfi->base_curve, t, &xx, &yy);

  if (cfi->use_2x2_matrix) {
    if (x) *x = cfi->a * xx + cfi->b * yy;
    if (y) *y = cfi->c * xx + cfi->d * yy;
    return;
  }

  if (x) {
    if (cfi->x_binary_fn_wrapper)
      *x = guppi_fn_wrapper_eval_d__d_d (cfi->x_binary_fn_wrapper, xx, yy);
    else if (cfi->x_fn_wrapper)
      *x = guppi_fn_wrapper_eval_d__d (cfi->x_fn_wrapper, xx);
    else /* default is identity fn. */
      *x = xx;
  }

  if (y) {
    if (cfi->y_binary_fn_wrapper)
      *y = guppi_fn_wrapper_eval_d__d_d (cfi->y_binary_fn_wrapper, xx, yy);
    else if (cfi->y_fn_wrapper)
      *y = guppi_fn_wrapper_eval_d__d (cfi->y_fn_wrapper, yy);
    else /* default is identity fn. */
      *y = yy;
  }
}

/****************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiCurveFuncImpl::" str, t, GTK_ARG_WRITABLE, symb)

static void
guppi_curve_func_impl_class_init (GuppiCurveFuncImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiCurveImplClass *curve_impl_class = GUPPI_CURVE_IMPL_CLASS (klass);
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_CURVE_IMPL);

  object_class->get_arg = guppi_curve_func_impl_get_arg;
  object_class->set_arg = guppi_curve_func_impl_set_arg;
  object_class->finalize = guppi_curve_func_impl_finalize;

  data_class->impl_name = _("Curve (Function)");

  curve_impl_class->bounds = v_curve_bounds;
  curve_impl_class->get = v_curve_get;

  add_arg ("curve", GTK_TYPE_POINTER, ARG_CURVE);
  add_arg ("reparameterize", GTK_TYPE_BOOL, ARG_REPARAMETERIZE);
  add_arg ("t0", GTK_TYPE_DOUBLE, ARG_T0);
  add_arg ("t1", GTK_TYPE_DOUBLE, ARG_T1);
  add_arg ("t_function", GTK_TYPE_POINTER, ARG_T_FUNC);
  add_arg ("x_function", GTK_TYPE_POINTER, ARG_X_FUNC);
  add_arg ("y_function", GTK_TYPE_POINTER, ARG_Y_FUNC);
  add_arg ("x_binary_function", GTK_TYPE_POINTER, ARG_X_BINARY_FUNC);
  add_arg ("y_binary_function", GTK_TYPE_POINTER, ARG_Y_BINARY_FUNC);
  add_arg ("use_2x2_matrix", GTK_TYPE_BOOL, ARG_USE_2x2_MATRIX);
  add_arg ("matrix_a", GTK_TYPE_DOUBLE, ARG_MATRIX_A);
  add_arg ("matrix_b", GTK_TYPE_DOUBLE, ARG_MATRIX_B);
  add_arg ("matrix_c", GTK_TYPE_DOUBLE, ARG_MATRIX_C);
  add_arg ("matrix_d", GTK_TYPE_DOUBLE, ARG_MATRIX_D);

}

static void
guppi_curve_func_impl_init (GuppiCurveFuncImpl * obj)
{
  obj->t0 = 0;
  obj->t1 = 1;
}

GtkType guppi_curve_func_impl_get_type (void)
{
  static GtkType guppi_curve_func_impl_type = 0;
  if (!guppi_curve_func_impl_type) {
    static const GtkTypeInfo guppi_curve_func_impl_info = {
      "GuppiCurveFuncImpl",
      sizeof (GuppiCurveFuncImpl),
      sizeof (GuppiCurveFuncImplClass),
      (GtkClassInitFunc) guppi_curve_func_impl_class_init,
      (GtkObjectInitFunc) guppi_curve_func_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_curve_func_impl_type =
      gtk_type_unique (GUPPI_TYPE_CURVE_IMPL, &guppi_curve_func_impl_info);
  }
  return guppi_curve_func_impl_type;
}

/****************************************************************************/

GuppiData *
guppi_curve_new_reparameterize_linearly (double t0, double t1,
					 GuppiCurve *curve)
{
  g_return_val_if_fail (curve && GUPPI_IS_CURVE (curve), NULL);

  return guppi_data_new_by_type (GUPPI_TYPE_CURVE,
				 GUPPI_TYPE_CURVE_FUNC_IMPL,
				 "curve", curve,
				 "t0", t0,
				 "t1", t1,
				 NULL);
}

GuppiData *
guppi_curve_new_reparameterize (double t0, double t1,
				GuppiFnWrapper *t_fn,
				GuppiCurve *curve)
{
  g_return_val_if_fail (curve && GUPPI_IS_CURVE (curve), NULL);
  g_return_val_if_fail (t_fn && GUPPI_IS_FN_WRAPPER (t_fn), NULL);

  return guppi_data_new_by_type (GUPPI_TYPE_CURVE,
				 GUPPI_TYPE_CURVE_FUNC_IMPL,
				 "curve", curve,
				 "t0", t0,
				 "t1", t1,
				 "t_function", t_fn,
				 NULL);
}

/****************************************************************************/

static GuppiDataImpl *
make_impl (void)
{
  return GUPPI_DATA_IMPL (guppi_type_new (guppi_curve_func_impl_get_type ()));
}

GuppiPlugIn *guppi_plug_in (void);

GuppiPlugIn *
guppi_plug_in (void)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;

  pi = guppi_data_impl_plug_in_new ();
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  pi->magic_number = GUPPI_PLUG_IN_MAGIC_NUMBER;
  dimpi->impl_constructor = make_impl;

  scm_curve_func_init ();

  return pi;
}



/* $Id: func.c,v 1.2 2000/12/14 20:22:50 trow Exp $ */
