/*
 * Seahorse
 *
 * Copyright (C) 2003 Jacob Perkins
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome.h>
#include <eel/eel.h>

#include "seahorse-context.h"
#include "seahorse-marshal.h"
#include "seahorse-preferences.h"
#include "seahorse-key-dialogs.h"

struct _SeahorseContextPrivate
{
	GList *key_pairs;
	GList *single_keys;
};

enum {
	ADD,
	PROGRESS,
	LAST_SIGNAL
};

static void	seahorse_context_class_init	(SeahorseContextClass	*klass);
static void	seahorse_context_init		(SeahorseContext	*sctx);
static void	seahorse_context_finalize	(GObject		*gobject);
/* Key signals */
static void	seahorse_context_key_destroyed	(GtkObject		*object,
						 SeahorseContext	*sctx);
static void	seahorse_context_key_changed	(SeahorseKey		*skey,
						 SeahorseKeyChange	change,
						 SeahorseContext	*sctx);
/* gpgme functions */
static void	gpgme_progress			(gpointer		data,
						 const gchar		*what,
						 gint			type,
						 gint			current,
						 gint			total);
static void	set_gpgme_signer		(SeahorseContext	*sctx,
						 const gchar		*id);
/* gconf notification */
static void	gconf_notification		(GConfClient		*gclient,
						 guint			id,
						 GConfEntry		*entry,
						 SeahorseContext	*sctx);

static GtkObjectClass	*parent_class			= NULL;
static guint		context_signals[LAST_SIGNAL]	= { 0 };

GType
seahorse_context_get_type (void)
{
	static GType context_type = 0;
	
	if (!context_type) {
		static const GTypeInfo context_info =
		{
			sizeof (SeahorseContextClass),
			NULL, NULL,
			(GClassInitFunc) seahorse_context_class_init,
			NULL, NULL,
			sizeof (SeahorseContext),
			0,
			(GInstanceInitFunc) seahorse_context_init
		};
		
		context_type = g_type_register_static (GTK_TYPE_OBJECT, "SeahorseContext", &context_info, 0);
	}
	
	return context_type;
}

static void
seahorse_context_class_init (SeahorseContextClass *klass)
{
	GObjectClass *gobject_class;
	
	parent_class = g_type_class_peek_parent (klass);
	gobject_class = G_OBJECT_CLASS (klass);
	
	gobject_class->finalize = seahorse_context_finalize;
	
	klass->add = NULL;
	klass->progress = NULL;
	
	context_signals[ADD] = g_signal_new ("add", G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET (SeahorseContextClass, add),
		NULL, NULL, g_cclosure_marshal_VOID__OBJECT, G_TYPE_NONE, 1, SEAHORSE_TYPE_KEY);
	context_signals[PROGRESS] = g_signal_new ("progress", G_OBJECT_CLASS_TYPE (gobject_class),
		G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET (SeahorseContextClass, progress),
		NULL, NULL, seahorse_marshal_VOID__STRING_DOUBLE, G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_DOUBLE);
}

/* init context, private vars, set prefs, connect signals */
static void
seahorse_context_init (SeahorseContext *sctx)
{
	GpgmeProtocol proto = GPGME_PROTOCOL_OpenPGP;
	/* init context */
	g_return_if_fail (gpgme_engine_check_version (proto) == GPGME_No_Error);
	g_return_if_fail (gpgme_new (&sctx->ctx) == GPGME_No_Error);
	g_return_if_fail (gpgme_set_protocol (sctx->ctx, proto) == GPGME_No_Error);
	/* init private vars */
	sctx->priv = g_new0 (SeahorseContextPrivate, 1);
	sctx->priv->single_keys = NULL;
	sctx->priv->key_pairs = NULL;
	/* init listing & signer */
	gpgme_set_keylist_mode (sctx->ctx, GPGME_KEYLIST_MODE_LOCAL);
	set_gpgme_signer (sctx, eel_gconf_get_string (DEFAULT_KEY));
	/* set prefs */
	gpgme_set_armor (sctx->ctx, eel_gconf_get_boolean (ASCII_ARMOR));
	gpgme_set_textmode (sctx->ctx, eel_gconf_get_boolean (TEXT_MODE));
	/* do callbacks */
	gpgme_set_passphrase_cb (sctx->ctx, (GpgmePassphraseCb)seahorse_passphrase_get, sctx);
	gpgme_set_progress_cb (sctx->ctx, gpgme_progress, sctx);
	eel_gconf_notification_add (PREFERENCES, (GConfClientNotifyFunc)gconf_notification, sctx);
	eel_gconf_monitor_add (PREFERENCES);
}

/* destroy all keys, free private vars */
static void
seahorse_context_finalize (GObject *gobject)
{
	SeahorseContext *sctx;
	SeahorseKey *skey;
	GList *list = NULL;
	
	sctx = SEAHORSE_CONTEXT (gobject);
	list = g_list_concat (g_list_copy (sctx->priv->key_pairs), sctx->priv->single_keys);
	/* destroy all keys */
	while (list != NULL && (skey = list->data) != NULL) {
		seahorse_key_destroy (skey);
		list = g_list_next (list);
	}
	
	g_list_free (sctx->priv->key_pairs);
	g_list_free (sctx->priv->single_keys);
	g_free (sctx->priv);
	gpgme_release (sctx->ctx);
	
	G_OBJECT_CLASS (parent_class)->finalize (gobject);
}

/* removes key from list, disconnects signals, unrefs */
static void
seahorse_context_key_destroyed (GtkObject *object, SeahorseContext *sctx)
{
	SeahorseKey *skey;
	
	skey = SEAHORSE_KEY (object);
	/* remove from list */
	if (SEAHORSE_IS_KEY_PAIR (skey))
		sctx->priv->key_pairs = g_list_remove (sctx->priv->key_pairs, skey);
	else
		sctx->priv->single_keys = g_list_remove (sctx->priv->single_keys, skey);
	/* disconnect signals */
	g_signal_handlers_disconnect_by_func (GTK_OBJECT (skey),
		seahorse_context_key_destroyed, sctx);
	g_signal_handlers_disconnect_by_func (skey,
		seahorse_context_key_changed, sctx);
	
	g_object_unref (skey);
}

/* refresh key's GpgmeKey(s) */
static void
seahorse_context_key_changed (SeahorseKey *skey, SeahorseKeyChange change, SeahorseContext *sctx)
{
	GpgmeKey key;
	SeahorseKeyPair *skpair;
	
	/* get key */
	g_return_if_fail (gpgme_op_keylist_start (sctx->ctx,
		seahorse_key_get_id (skey->key), FALSE) == GPGME_No_Error);
	g_return_if_fail (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error);
	gpgme_op_keylist_end (sctx->ctx);
	/* set key */
	gpgme_key_unref (skey->key);
	skey->key = key;
	/* if is a pair */
	if (SEAHORSE_IS_KEY_PAIR (skey)) {
		skpair = SEAHORSE_KEY_PAIR (skey);
		/* get key */
		g_return_if_fail (gpgme_op_keylist_start (sctx->ctx,
			seahorse_key_get_id (skey->key), TRUE) == GPGME_No_Error);
		g_return_if_fail (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error);
		gpgme_op_keylist_end (sctx->ctx);
		/* set key */
		gpgme_key_unref (skpair->secret);
		skpair->secret = key;
	}
}

/* Calc fraction, call ::show_progress() */
static void
gpgme_progress (gpointer data, const gchar *what, gint type, gint current, gint total)
{
	gdouble fract;
	/* do fract */
	if (total == 0)
		fract = 0;
	else if (current == 100 || total == current)
		fract = -1;
	else
		fract = (gdouble)current / (gdouble)total;
	
	seahorse_context_show_progress (SEAHORSE_CONTEXT (data), what, fract);
}

/* sets key with id as signer */
static void
set_gpgme_signer (SeahorseContext *sctx, const gchar *id)
{
	GpgmeKey key;
	/* clear signers, get key, add key to signers */
	gpgme_signers_clear (sctx->ctx);
	g_return_if_fail (gpgme_op_keylist_start (sctx->ctx, id, TRUE) == GPGME_No_Error);
	if (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error)
		gpgme_signers_add (sctx->ctx, key);
	gpgme_op_keylist_end (sctx->ctx);
	gpgme_key_unref (key);
}

/* change context if prefs changed */
static void
gconf_notification (GConfClient *gclient, guint id, GConfEntry *entry, SeahorseContext *sctx)
{
	const gchar *key = gconf_entry_get_key (entry);
	GConfValue *value = gconf_entry_get_value (entry);
	
	if (g_str_equal (key, ASCII_ARMOR))
		gpgme_set_armor (sctx->ctx, gconf_value_get_bool (value));
	else if (g_str_equal (key, TEXT_MODE))
		gpgme_set_textmode (sctx->ctx, gconf_value_get_bool (value));
	else if (g_str_equal (key, DEFAULT_KEY))
		set_gpgme_signer (sctx, gconf_value_get_string (value));
}

/**
 * seahorse_context_new:
 *
 * Creates a new #SeahorseContext for managing the key ring and preferences.
 *
 * Returns: The new #SeahorseContext
 **/
SeahorseContext*
seahorse_context_new (void)
{
	return g_object_new (SEAHORSE_TYPE_CONTEXT, NULL);
}

/**
 * seahorse_context_destroy:
 * @sctx: #SeahorseContext to destroy
 *
 * Emits the destroy signal for @sctx.
 **/
void
seahorse_context_destroy (SeahorseContext *sctx)
{
	g_return_if_fail (GTK_IS_OBJECT (sctx));
	
	gtk_object_destroy (GTK_OBJECT (sctx));
}

/* refs key and connects signals */
static void
add_key (SeahorseContext *sctx, SeahorseKey *skey)
{
	g_object_ref (skey);
	gtk_object_sink (GTK_OBJECT (skey));
	
	g_signal_connect_after (GTK_OBJECT (skey), "destroy",
		G_CALLBACK (seahorse_context_key_destroyed), sctx);
	g_signal_connect (skey, "changed",
		G_CALLBACK (seahorse_context_key_changed), sctx);
	
	g_signal_emit (G_OBJECT (sctx), context_signals[ADD], 0, skey);
}

/* creates and adds a new key */
static void
add_single_key (SeahorseContext *sctx, GpgmeKey key)
{
	SeahorseKey *skey;
	
	skey = seahorse_key_new (key);
	gpgme_key_unref (key);
	sctx->priv->single_keys = g_list_append (sctx->priv->single_keys, skey);
	add_key (sctx, skey);
}

/* creates and adds a new key pair */
static void
add_key_pair (SeahorseContext *sctx, GpgmeKey key, GpgmeKey secret)
{
	SeahorseKey *skey;
	
	skey = seahorse_key_pair_new (key, secret);
	gpgme_key_unref (key);
	gpgme_key_unref (secret);
	sctx->priv->key_pairs = g_list_append (sctx->priv->key_pairs, skey);
	add_key (sctx, skey);
}

/* loads key pairs */
static void
do_key_pairs (SeahorseContext *sctx)
{
	static gboolean did_pairs = FALSE;
	GpgmeKey key;
	GList *keys = NULL;
	SeahorseKey *skey;
	guint count = 1;
	gint progress_update;
	gdouble length;
	
	if (did_pairs)
		return;
	
	progress_update = eel_gconf_get_integer (PROGRESS_UPDATE);
	
	/* get secret keys */
	g_return_if_fail (gpgme_op_keylist_start (sctx->ctx, NULL, TRUE) == GPGME_No_Error);
	while (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error) {
		if (progress_update > 0 && !(count % progress_update))
			seahorse_context_show_progress (sctx, g_strdup_printf (
				_("Loading key pair %d"), count), 0);
		
		keys = g_list_append (keys, key);
		count++;
	}
	
	count = 1;
	length = g_list_length (keys);
	
	/* get corresponding public keys */
	while (keys != NULL) {
		if (progress_update > 0 && !(count % progress_update))
			seahorse_context_show_progress (sctx, g_strdup_printf (
				_("Processing key pair %d"), count), count/length);
		
		/* create new key pair */
		if ((gpgme_op_keylist_start (sctx->ctx, seahorse_key_get_id (
		keys->data), FALSE) == GPGME_No_Error) &&
		(gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error)) {
			add_key_pair (sctx, key, keys->data);
			gpgme_op_keylist_end (sctx->ctx);
		}
		/* will only happen if secret exists but public doesn't */
		else
			gpgme_key_unref (keys->data);
		
		
		keys = g_list_next (keys);
		count++;
	}
	
	did_pairs = TRUE;
	seahorse_context_show_progress (sctx, g_strdup_printf (
		_("Loaded %d key pairs"), count), -1);
}

/* loads single keys, doing pairs first */
static void
do_key_list (SeahorseContext *sctx)
{
	static gboolean did_keys = FALSE;
	GpgmeKey key;
	GList *keys = NULL;
	SeahorseKey *skey;
	guint count = 1, secret_count;
	gint progress_update;
	gdouble length;
	
	if (did_keys)
		return;
	
	if (sctx->priv->key_pairs == NULL)
		do_key_pairs (sctx);
	
	secret_count = g_list_length (sctx->priv->key_pairs);
	progress_update = eel_gconf_get_integer (PROGRESS_UPDATE);
	
	/* get public keys */
	g_return_if_fail (gpgme_op_keylist_start (sctx->ctx, NULL, FALSE) == GPGME_No_Error);
	while (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error) {
		/* show progress */
		if (progress_update > 0 && !(count % progress_update))
			seahorse_context_show_progress (sctx, g_strdup_printf (
				_("Loading key %d"), count), 0);
			
		keys = g_list_append (keys, key);
		count++;
	}
	
	count = 1;
	length = g_list_length (keys);
	
	/* add keys to list */
	while (keys != NULL) {
		if (progress_update > 0 && !(count % progress_update))
			seahorse_context_show_progress (sctx, g_strdup_printf (
				_("Processing key %d"), count), count/length);
		
		/* if there are more secret keys left and key has a secret part */
		if (secret_count && (gpgme_op_keylist_start (sctx->ctx,
		seahorse_key_get_id (keys->data), TRUE) == GPGME_No_Error) &&
		(gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error)) {
			gpgme_key_unref (key);
			gpgme_key_unref (keys->data);
			secret_count--;
			gpgme_op_keylist_end (sctx->ctx);
		}
		/* otherwise add to list */
		else
			add_single_key (sctx, keys->data);
		
		keys = g_list_next (keys);
		count++;
	}
	
	did_keys = TRUE;
	seahorse_context_show_progress (sctx, g_strdup_printf (
		_("Loading %d keys"), count), -1);
}

/**
 * seahorse_context_get_keys:
 * @sctx: #SeahorseContext
 *
 * Gets the complete key ring.
 *
 * Returns: A #GList of #SeahorseKey
 **/
GList*
seahorse_context_get_keys (SeahorseContext *sctx)
{
	static gboolean init_list = FALSE;
	
	if (!init_list) {
		do_key_list (sctx);
		init_list = TRUE;
	}
	
	/* copy key_pairs since will append single_keys to list & key_pairs is small */
	return g_list_concat (g_list_copy (sctx->priv->key_pairs), sctx->priv->single_keys);
}

/**
 * seahorse_context_get_key_pairs:
 * @sctx: #SeahorseContext
 *
 * Gets the secret key ring
 *
 * Returns: A #GList of #SeahorseKeyPair
 **/
GList*
seahorse_context_get_key_pairs (SeahorseContext *sctx)
{
	static gboolean init_list = FALSE;
	
	if (!init_list) {
		do_key_pairs (sctx);
		init_list = TRUE;
	}
	
	return sctx->priv->key_pairs;
}

/**
 * seahorse_context_get_key:
 * @sctx: Current #SeahorseContext
 * @key: GpgmeKey corresponding to the desired #SeahorseKey
 *
 * Gets the #SeahorseKey in the key ring corresponding to @key.
 *
 * Returns: The found #SeahorseKey, or NULL
 **/
SeahorseKey*
seahorse_context_get_key (SeahorseContext *sctx, GpgmeKey key)
{
	GList *list;
	SeahorseKey *skey = NULL;
	const gchar *id1, *id2 = "";
	
	g_return_val_if_fail (sctx != NULL && SEAHORSE_IS_CONTEXT (sctx), NULL);
	g_return_val_if_fail (key != NULL, NULL);
	
	list = seahorse_context_get_keys (sctx);
	id1 = seahorse_key_get_id (key);
	
	/* Search by keyid */
	while (list != NULL && (skey = list->data) != NULL) {
		id2 = seahorse_key_get_id (skey->key);
		
		if (g_str_equal (id1, id2))
			break;
		
		list = g_list_next (list);
	}
	
	gpgme_key_unref (key);
	
	if (!g_str_equal (id1, id2))
		return NULL;
	else
		return skey;
}

/**
 * seahorse_context_show_status:
 * @sctx: Current #SeahorseContext
 * @op: Operation performed
 * @success: Whether @op was successful
 *
 * Generates the status message, the calls seahorse_context_show_progress()
 * with the message and -1.
 **/
void
seahorse_context_show_status (SeahorseContext *sctx, const gchar *op, gboolean success)
{
	gchar *status;
	
	g_return_if_fail (SEAHORSE_IS_CONTEXT (sctx));
	g_return_if_fail (op != NULL);
	
	/* Construct status message */
	if (success)
		status = g_strdup_printf (_("%s Successful"), op);
	else
		status = g_strdup_printf (_("%s Failed"), op);
	g_print ("%s\n", status);
	
	seahorse_context_show_progress (sctx, status, -1);
	
	g_free (status);
}

/**
 * seahorse_context_show_progress:
 * @sctx: #SeahorseContext
 * @op: Operation message to display
 * @fract: Fract should have one of 3 values: 0 < fract <= 1 will show
 * the progress as a fraction of the total progress; fract == -1 signifies
 * that the operation is complete; any other value will pulse the progress.
 *
 * Emits the progress signal for @sctx.
 **/
void
seahorse_context_show_progress (SeahorseContext *sctx, const gchar *op, gdouble fract)
{
	g_return_if_fail (sctx != NULL && SEAHORSE_IS_CONTEXT (sctx));
	
	g_signal_emit (G_OBJECT (sctx), context_signals[PROGRESS], 0, op, fract);
}

/**
 * seahorse_context_key_added
 * @sctx: Current #SeahorseContext
 *
 * Causes @sctx to add any new #SeahorseKey to the key ring.
 * Emits add signal for @sctx.
 **/
void
seahorse_context_key_added (SeahorseContext *sctx)
{
	GpgmeKey key;
	GList *list, *keys = NULL;
	SeahorseKey *skey;
	gint progress_update;
	guint count = 1;
	
	g_return_if_fail (sctx != NULL && SEAHORSE_IS_CONTEXT (sctx));
        g_return_if_fail (gpgme_op_keylist_start (sctx->ctx, NULL, FALSE) == GPGME_No_Error);	
	
	list = seahorse_context_get_keys (sctx);
	progress_update = eel_gconf_get_integer (PROGRESS_UPDATE);
	
	/* go to end of list, then build list containing new keys */
	while (gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error) {
		if (progress_update > 0 && !(count % progress_update))
			seahorse_context_show_progress (sctx, _("Processing new key"), 0);
		if (list == NULL)
			keys = g_list_append (keys, key);
		else
			list = g_list_next (list);
		count++;
	}
	
	/* add any new keys */
	while (keys != NULL) {
		/* if has secret part */
		if ((gpgme_op_keylist_start (sctx->ctx, seahorse_key_get_id (
		keys->data), TRUE) == GPGME_No_Error) &&
		gpgme_op_keylist_next (sctx->ctx, &key) == GPGME_No_Error) {
			add_key_pair (sctx, keys->data, key);
			gpgme_op_keylist_end (sctx->ctx);
		}
		else
			add_single_key (sctx, keys->data);
		
		keys = g_list_next (keys);
	}
}

/**
 * seahorse_context_set_signer:
 * @sctx: Current #SeahorseContext
 * @signer: New signer #SeahorseKey
 *
 * Sets signer attribute of @sctx to @signer and saves in gconf.
 **/
void
seahorse_context_set_default_key (SeahorseContext *sctx, SeahorseKeyPair *skpair)
{
	g_return_if_fail (sctx != NULL && SEAHORSE_IS_CONTEXT (sctx));
	g_return_if_fail (skpair != NULL && SEAHORSE_IS_KEY_PAIR (skpair));
	
	eel_gconf_set_string (DEFAULT_KEY, seahorse_key_get_id (skpair->secret));
}

/**
 * seahorse_context_get_last_signer:
 * @sctx: Current #SeahorseContext
 *
 * Gets the signing #SeahorseKey for @sctx
 *
 * Returns: The default signing #SeahorseKey
 **/
SeahorseKeyPair*
seahorse_context_get_default_key (SeahorseContext *sctx)
{
	GpgmeKey key;
	
	key = gpgme_signers_enum (sctx->ctx, 0);
	return SEAHORSE_KEY_PAIR (seahorse_context_get_key (sctx, key));
}
