/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.glib;

import org.gnu.javagnome.Handle;
import org.gnu.javagnome.Struct;

/**
 * A mechanism to wrap opaque structures registered by the type system.
 */
public class Boxed extends Struct {

    /**
     * This class is only instantiable via subclasses.
     */
    protected Boxed() {
        // nothing to do
    }
	
    protected Boxed(Handle handle) {
        super( handle );
    }

    /**
     * Gets a <tt>Boxed</tt> instance for the given <tt>Handle</tt>.  If
     * no Java object currently exists for the given <tt>Handle</tt>, this
     * method will return <tt>null</tt>.  You should then instantiate the
     * required Java class using the class' handle-constructor.  For example:
     *
     * <pre>
     * // Get a Handle from somewhere (typically as a parameter to a method
     * // used as a callback and invoked from the C JNI side).
     * SomeGtkClass finalobj = null;
     * Boxed obj = Boxed.getBoxedFromHandle( handle );
     * if ( obj == null ) {
     *     finalobj = new SomeGtkClass( handle );
     * } else {
     *     finalobj = (SomeGtkClass)obj;
     * }
     * </pre>
     *
     * NOTE: This is for internal use only and should never need to be used
     * in application code.
     */
    public static Boxed getBoxedFromHandle( Handle hndl ) {
        return (Boxed) hndl.getProxiedObject();
    }
    
    /**
     * Check if two objects refer to the same (native) object.
     *
     * @param other the reference object with which to compare.
     * @return true if both objects refer to the same object.
     */
    public boolean equals( Object other ){
        return 
            other instanceof Boxed && 
            getHandle().equals( ((Boxed)other).getHandle() );
    }
	
    /**
     * Returns a hash code value for the object. This allows for
     * using Boxed objects as keys in hashmaps.
     *
     * @return a hash code value for the object.
     */
    public int hashCode() {
        return getHandle().hashCode();
    }

    protected void finalize() throws Throwable {
        System.out.println( "Boxed:finalize() of: " + this.getClass().getName() );
        System.out.println( "  " + getHandle().toString() );
        nativeFinalize(getHandle());
        System.out.println( "Boxed:finalize() completed." );
    }

    native static final private void nativeFinalize(Handle handle);
}
