package Color::Library::Dictionary::NBS_ISCC::SC;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::SC;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::SC - (NBS/ISCC SC) Soil Color Charts

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (SC) Soil Color Charts

    U.S. Department of Agriculture
    Soil Color Charts 

This combinatorial field-work dictionary's 50 names mape to 176 colors, using 76 of 267 centroids. Again, its focus on soil colors limits its general usefulness.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#SC>

=head1 COLORS

	black                   black                #3e322c

	black                   black                #28201c

	black                   black                #3b3121

	black                   black                #363527

	black                   black                #25241d

	black                   black                #222222

	brown                   brown                #a67b5b

	brown                   brown                #6f4e37

	brown                   brown                #958070

	brown                   brown                #635147

	brown                   brown                #826644

	brown                   brown                #7e6d5a

	brownish yellow         brownishyellow       #be8a3d

	brownish yellow         brownishyellow       #c19a6b

	dark brown              darkbrown            #6f4e37

	dark brown              darkbrown            #635147

	dark brown              darkbrown            #826644

	dark brown              darkbrown            #4b3621

	dark brown              darkbrown            #7e6d5a

	dark gray               darkgray             #5b504f

	dark gray               darkgray             #57554c

	dark gray               darkgray             #555555

	dark grayish brown      darkgrayishbrown     #7e6d5a

	dark grayish brown      darkgrayishbrown     #6c541e

	dark olive              darkolive            #665d1e

	dark olive              darkolive            #403d21

	dark olive              darkolive            #363527

	dark olive-gray         darkolivegray        #5b5842

	dark red                darkred              #722f37

	dark red                darkred              #882d17

	dark red                darkred              #56070c

	dark red                darkred              #79443b

	dark red                darkred              #80461b

	dark reddish brown      darkreddishbrown     #79443b

	dark reddish brown      darkreddishbrown     #3e1d1e

	dark reddish brown      darkreddishbrown     #674c47

	dark reddish brown      darkreddishbrown     #43302e

	dark reddish brown      darkreddishbrown     #6f4e37

	dark reddish brown      darkreddishbrown     #422518

	dark reddish brown      darkreddishbrown     #635147

	dark reddish brown      darkreddishbrown     #3e322c

	dark reddish gray       darkreddishgray      #5c504f

	dark reddish gray       darkreddishgray      #635147

	dark reddish gray       darkreddishgray      #5b504f

	dark yellowish brown    darkyellowishbrown   #996515

	dark yellowish brown    darkyellowishbrown   #826644

	dark yellowish brown    darkyellowishbrown   #4b3621

	dusky red               duskyred             #79443b

	dusky red               duskyred             #674c47

	gray                    gray                 #8e8279

	gray                    gray                 #8a8776

	gray                    gray                 #848482

	grayish brown           grayishbrown         #7e6d5a

	grayish brown           grayishbrown         #967117

	light brown             lightbrown           #a67b5b

	light brownish gray     lightbrownishgray    #ae9b82

	light brownish gray     lightbrownishgray    #967117

	light gray              lightgray            #c1b6b3

	light gray              lightgray            #8e8279

	light gray              lightgray            #ae9b82

	light gray              lightgray            #c2b280

	light gray              lightgray            #bfb8a5

	light gray              lightgray            #8a8776

	light gray              lightgray            #b9b8b5

	light gray              lightgray            #848482

	light olive brown       lightolivebrown      #967117

	light olive-gray        lightolivegray       #8c8767

	light olive-gray        lightolivegray       #8a8776

	light red               lightred             #e66761

	light red               lightred             #c48379

	light red               lightred             #cb6d51

	light red               lightred             #b4745e

	light red               lightred             #d99058

	light reddish brown     lightreddishbrown    #a87c6d

	light reddish brown     lightreddishbrown    #977f73

	light reddish brown     lightreddishbrown    #a67b5b

	light yellowish brown   lightyellowishbrown  #c19a6b

	light yellowish brown   lightyellowishbrown  #a18f60

	olive                   olive                #867e36

	olive                   olive                #665d1e

	olive                   olive                #8c8767

	olive                   olive                #5b5842

	olive brown             olivebrown           #6c541e

	olive brown             olivebrown           #3b3121

	olive gray              olivegray            #8c8767

	olive gray              olivegray            #5b5842

	olive gray              olivegray            #8a8776

	olive yellow            oliveyellow          #ab9144

	pale brown              palebrown            #c19a6b

	pale brown              palebrown            #ae9b82

	pale olive              paleolive            #a18f60

	pale olive              paleolive            #8c8767

	pale red                palered              #ad8884

	pale red                palered              #a87c6d

	pale red                palered              #977f73

	pale yellow             paleyellow           #f3e5ab

	pale yellow             paleyellow           #c2b280

	pink                    pink                 #f4c2c2

	pink                    pink                 #d9a6a9

	pink                    pink                 #ecd5c5

	pink                    pink                 #c7ada3

	pink                    pink                 #c2ac99

	pink                    pink                 #fad6a5

	pink                    pink                 #c19a6b

	pinkish gray            pinkishgray          #c7ada3

	pinkish gray            pinkishgray          #c2ac99

	pinkish gray            pinkishgray          #977f73

	pinkish gray            pinkishgray          #958070

	pinkish white           pinkishwhite         #ecd5c5

	pinkish white           pinkishwhite         #c7ada3

	pinkish white           pinkishwhite         #c2ac99

	red                     red                  #ab4e52

	red                     red                  #905d5d

	red                     red                  #cb6d51

	red                     red                  #9e4732

	red                     red                  #b4745e

	red                     red                  #79443b

	red                     red                  #ae6938

	red                     red                  #80461b

	reddish black           reddishblack         #282022

	reddish black           reddishblack         #3e1d1e

	reddish black           reddishblack         #43302e

	reddish black           reddishblack         #3e322c

	reddish black           reddishblack         #28201c

	reddish brown           reddishbrown         #a87c6d

	reddish brown           reddishbrown         #79443b

	reddish brown           reddishbrown         #977f73

	reddish brown           reddishbrown         #674c47

	reddish brown           reddishbrown         #a67b5b

	reddish brown           reddishbrown         #6f4e37

	reddish gray            reddishgray          #8f817f

	reddish gray            reddishgray          #977f73

	reddish gray            reddishgray          #958070

	reddish gray            reddishgray          #8e8279

	reddish yellow          reddishyellow        #fab57f

	reddish yellow          reddishyellow        #d99058

	reddish yellow          reddishyellow        #fbc97f

	reddish yellow          reddishyellow        #e3a857

	reddish yellow          reddishyellow        #be8a3d

	strong brown            strongbrown          #80461b

	strong brown            strongbrown          #996515

	very dark brown         verydarkbrown        #422518

	very dark brown         verydarkbrown        #3e322c

	very dark brown         verydarkbrown        #4b3621

	very dark brown         verydarkbrown        #483c32

	very dark gray          verydarkgray         #5b504f

	very dark gray          verydarkgray         #57554c

	very dark gray          verydarkgray         #555555

	very dark grayish brown verydarkgrayishbrown #483c32

	very dark grayish brown verydarkgrayishbrown #6c541e

	very dusky red          veryduskyred         #3e1d1e

	very dusky red          veryduskyred         #43302e

	very pale brown         verypalebrown        #c2ac99

	very pale brown         verypalebrown        #fad6a5

	very pale brown         verypalebrown        #c19a6b

	very pale brown         verypalebrown        #ae9b82

	weak red                weakred              #905d5d

	weak red                weakred              #a87c6d

	weak red                weakred              #79443b

	weak red                weakred              #977f73

	weak red                weakred              #674c47

	white                   white                #eae3e1

	white                   white                #c1b6b3

	white                   white                #fad6a5

	white                   white                #f0ead6

	white                   white                #bfb8a5

	white                   white                #f2f3f4

	white                   white                #b9b8b5

	yellow                  yellow               #fbc97f

	yellow                  yellow               #e3a857

	yellow                  yellow               #f8de7e

	yellow                  yellow               #c9ae5d

	yellowish brown         yellowishbrown       #996515

	yellowish brown         yellowishbrown       #826644

	yellow red              yellowred            #ae6938

	yellow red              yellowred            #80461b


=cut

sub _load_color_list() {
    return [
['nbs-iscc-sc:black.62','black','black',[62,50,44],'3e322c',4076076],
['nbs-iscc-sc:black.65','black','black',[40,32,28],'28201c',2629660],
['nbs-iscc-sc:black.96','black','black',[59,49,33],'3b3121',3879201],
['nbs-iscc-sc:black.111','black','black',[54,53,39],'363527',3552551],
['nbs-iscc-sc:black.114','black','black',[37,36,29],'25241d',2434077],
['nbs-iscc-sc:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-sc:brown.57','brown','brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-sc:brown.58','brown','brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-sc:brown.60','brown','brown',[149,128,112],'958070',9797744],
['nbs-iscc-sc:brown.61','brown','brown',[99,81,71],'635147',6508871],
['nbs-iscc-sc:brown.77','brown','brown',[130,102,68],'826644',8545860],
['nbs-iscc-sc:brown.80','brown','brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-sc:brownishyellow.72','brownishyellow','brownish yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-sc:brownishyellow.76','brownishyellow','brownish yellow',[193,154,107],'c19a6b',12687979],
['nbs-iscc-sc:darkbrown.58','darkbrown','dark brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-sc:darkbrown.61','darkbrown','dark brown',[99,81,71],'635147',6508871],
['nbs-iscc-sc:darkbrown.77','darkbrown','dark brown',[130,102,68],'826644',8545860],
['nbs-iscc-sc:darkbrown.78','darkbrown','dark brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-sc:darkbrown.80','darkbrown','dark brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-sc:darkgray.64','darkgray','dark gray',[91,80,79],'5b504f',5984335],
['nbs-iscc-sc:darkgray.113','darkgray','dark gray',[87,85,76],'57554c',5723468],
['nbs-iscc-sc:darkgray.266','darkgray','dark gray',[85,85,85],'555555',5592405],
['nbs-iscc-sc:darkgrayishbrown.80','darkgrayishbrown','dark grayish brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-sc:darkgrayishbrown.95','darkgrayishbrown','dark grayish brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-sc:darkolive.107','darkolive','dark olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-sc:darkolive.108','darkolive','dark olive',[64,61,33],'403d21',4209953],
['nbs-iscc-sc:darkolive.111','darkolive','dark olive',[54,53,39],'363527',3552551],
['nbs-iscc-sc:darkolivegray.110','darkolivegray','dark olive-gray',[91,88,66],'5b5842',5986370],
['nbs-iscc-sc:darkred.16','darkred','dark red',[114,47,55],'722f37',7483191],
['nbs-iscc-sc:darkred.40','darkred','dark red',[136,45,23],'882d17',8924439],
['nbs-iscc-sc:darkred.41','darkred','dark red',[86,7,12],'56070c',5637900],
['nbs-iscc-sc:darkred.43','darkred','dark red',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:darkred.55','darkred','dark red',[128,70,27],'80461b',8406555],
['nbs-iscc-sc:darkreddishbrown.43','darkreddishbrown','dark reddish brown',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:darkreddishbrown.44','darkreddishbrown','dark reddish brown',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-sc:darkreddishbrown.46','darkreddishbrown','dark reddish brown',[103,76,71],'674c47',6769735],
['nbs-iscc-sc:darkreddishbrown.47','darkreddishbrown','dark reddish brown',[67,48,46],'43302e',4403246],
['nbs-iscc-sc:darkreddishbrown.58','darkreddishbrown','dark reddish brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-sc:darkreddishbrown.59','darkreddishbrown','dark reddish brown',[66,37,24],'422518',4334872],
['nbs-iscc-sc:darkreddishbrown.61','darkreddishbrown','dark reddish brown',[99,81,71],'635147',6508871],
['nbs-iscc-sc:darkreddishbrown.62','darkreddishbrown','dark reddish brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-sc:darkreddishgray.23','darkreddishgray','dark reddish gray',[92,80,79],'5c504f',6049871],
['nbs-iscc-sc:darkreddishgray.61','darkreddishgray','dark reddish gray',[99,81,71],'635147',6508871],
['nbs-iscc-sc:darkreddishgray.64','darkreddishgray','dark reddish gray',[91,80,79],'5b504f',5984335],
['nbs-iscc-sc:darkyellowishbrown.74','darkyellowishbrown','dark yellowish brown',[153,101,21],'996515',10052885],
['nbs-iscc-sc:darkyellowishbrown.77','darkyellowishbrown','dark yellowish brown',[130,102,68],'826644',8545860],
['nbs-iscc-sc:darkyellowishbrown.78','darkyellowishbrown','dark yellowish brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-sc:duskyred.43','duskyred','dusky red',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:duskyred.46','duskyred','dusky red',[103,76,71],'674c47',6769735],
['nbs-iscc-sc:gray.63','gray','gray',[142,130,121],'8e8279',9339513],
['nbs-iscc-sc:gray.112','gray','gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-sc:gray.265','gray','gray',[132,132,130],'848482',8684674],
['nbs-iscc-sc:grayishbrown.80','grayishbrown','grayish brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-sc:grayishbrown.94','grayishbrown','grayish brown',[150,113,23],'967117',9859351],
['nbs-iscc-sc:lightbrown.57','lightbrown','light brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-sc:lightbrownishgray.79','lightbrownishgray','light brownish gray',[174,155,130],'ae9b82',11443074],
['nbs-iscc-sc:lightbrownishgray.94','lightbrownishgray','light brownish gray',[150,113,23],'967117',9859351],
['nbs-iscc-sc:lightgray.10','lightgray','light gray',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-sc:lightgray.63','lightgray','light gray',[142,130,121],'8e8279',9339513],
['nbs-iscc-sc:lightgray.79','lightgray','light gray',[174,155,130],'ae9b82',11443074],
['nbs-iscc-sc:lightgray.90','lightgray','light gray',[194,178,128],'c2b280',12759680],
['nbs-iscc-sc:lightgray.93','lightgray','light gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-sc:lightgray.112','lightgray','light gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-sc:lightgray.264','lightgray','light gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-sc:lightgray.265','lightgray','light gray',[132,132,130],'848482',8684674],
['nbs-iscc-sc:lightolivebrown.94','lightolivebrown','light olive brown',[150,113,23],'967117',9859351],
['nbs-iscc-sc:lightolivegray.109','lightolivegray','light olive-gray',[140,135,103],'8c8767',9209703],
['nbs-iscc-sc:lightolivegray.112','lightolivegray','light olive-gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-sc:lightred.27','lightred','light red',[230,103,97],'e66761',15099745],
['nbs-iscc-sc:lightred.30','lightred','light red',[196,131,121],'c48379',12878713],
['nbs-iscc-sc:lightred.37','lightred','light red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-sc:lightred.39','lightred','light red',[180,116,94],'b4745e',11826270],
['nbs-iscc-sc:lightred.53','lightred','light red',[217,144,88],'d99058',14258264],
['nbs-iscc-sc:lightreddishbrown.42','lightreddishbrown','light reddish brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-sc:lightreddishbrown.45','lightreddishbrown','light reddish brown',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:lightreddishbrown.57','lightreddishbrown','light reddish brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-sc:lightyellowishbrown.76','lightyellowishbrown','light yellowish brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-sc:lightyellowishbrown.91','lightyellowishbrown','light yellowish brown',[161,143,96],'a18f60',10588000],
['nbs-iscc-sc:olive.106','olive','olive',[134,126,54],'867e36',8814134],
['nbs-iscc-sc:olive.107','olive','olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-sc:olive.109','olive','olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-sc:olive.110','olive','olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-sc:olivebrown.95','olivebrown','olive brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-sc:olivebrown.96','olivebrown','olive brown',[59,49,33],'3b3121',3879201],
['nbs-iscc-sc:olivegray.109','olivegray','olive gray',[140,135,103],'8c8767',9209703],
['nbs-iscc-sc:olivegray.110','olivegray','olive gray',[91,88,66],'5b5842',5986370],
['nbs-iscc-sc:olivegray.112','olivegray','olive gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-sc:oliveyellow.88','oliveyellow','olive yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-sc:palebrown.76','palebrown','pale brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-sc:palebrown.79','palebrown','pale brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-sc:paleolive.91','paleolive','pale olive',[161,143,96],'a18f60',10588000],
['nbs-iscc-sc:paleolive.109','paleolive','pale olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-sc:palered.18','palered','pale red',[173,136,132],'ad8884',11372676],
['nbs-iscc-sc:palered.42','palered','pale red',[168,124,109],'a87c6d',11041901],
['nbs-iscc-sc:palered.45','palered','pale red',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:paleyellow.89','paleyellow','pale yellow ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-sc:paleyellow.90','paleyellow','pale yellow ',[194,178,128],'c2b280',12759680],
['nbs-iscc-sc:pink.28','pink','pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-sc:pink.29','pink','pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-sc:pink.31','pink','pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-sc:pink.32','pink','pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-sc:pink.33','pink','pink',[194,172,153],'c2ac99',12758169],
['nbs-iscc-sc:pink.73','pink','pink',[250,214,165],'fad6a5',16438949],
['nbs-iscc-sc:pink.76','pink','pink',[193,154,107],'c19a6b',12687979],
['nbs-iscc-sc:pinkishgray.32','pinkishgray','pinkish gray',[199,173,163],'c7ada3',13086115],
['nbs-iscc-sc:pinkishgray.33','pinkishgray','pinkish gray',[194,172,153],'c2ac99',12758169],
['nbs-iscc-sc:pinkishgray.45','pinkishgray','pinkish gray',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:pinkishgray.60','pinkishgray','pinkish gray',[149,128,112],'958070',9797744],
['nbs-iscc-sc:pinkishwhite.31','pinkishwhite','pinkish white',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-sc:pinkishwhite.32','pinkishwhite','pinkish white',[199,173,163],'c7ada3',13086115],
['nbs-iscc-sc:pinkishwhite.33','pinkishwhite','pinkish white',[194,172,153],'c2ac99',12758169],
['nbs-iscc-sc:red.15','red','red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-sc:red.19','red','red',[144,93,93],'905d5d',9461085],
['nbs-iscc-sc:red.37','red','red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-sc:red.38','red','red',[158,71,50],'9e4732',10372914],
['nbs-iscc-sc:red.39','red','red',[180,116,94],'b4745e',11826270],
['nbs-iscc-sc:red.43','red','red',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:red.54','red','red',[174,105,56],'ae6938',11430200],
['nbs-iscc-sc:red.55','red','red',[128,70,27],'80461b',8406555],
['nbs-iscc-sc:reddishblack.24','reddishblack','reddish black',[40,32,34],'282022',2629666],
['nbs-iscc-sc:reddishblack.44','reddishblack','reddish black',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-sc:reddishblack.47','reddishblack','reddish black',[67,48,46],'43302e',4403246],
['nbs-iscc-sc:reddishblack.62','reddishblack','reddish black',[62,50,44],'3e322c',4076076],
['nbs-iscc-sc:reddishblack.65','reddishblack','reddish black',[40,32,28],'28201c',2629660],
['nbs-iscc-sc:reddishbrown.42','reddishbrown','reddish brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-sc:reddishbrown.43','reddishbrown','reddish brown',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:reddishbrown.45','reddishbrown','reddish brown',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:reddishbrown.46','reddishbrown','reddish brown',[103,76,71],'674c47',6769735],
['nbs-iscc-sc:reddishbrown.57','reddishbrown','reddish brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-sc:reddishbrown.58','reddishbrown','reddish brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-sc:reddishgray.22','reddishgray','reddish gray',[143,129,127],'8f817f',9404799],
['nbs-iscc-sc:reddishgray.45','reddishgray','reddish gray',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:reddishgray.60','reddishgray','reddish gray',[149,128,112],'958070',9797744],
['nbs-iscc-sc:reddishgray.63','reddishgray','reddish gray',[142,130,121],'8e8279',9339513],
['nbs-iscc-sc:reddishyellow.52','reddishyellow','reddish yellow',[250,181,127],'fab57f',16430463],
['nbs-iscc-sc:reddishyellow.53','reddishyellow','reddish yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-sc:reddishyellow.70','reddishyellow','reddish yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-sc:reddishyellow.71','reddishyellow','reddish yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-sc:reddishyellow.72','reddishyellow','reddish yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-sc:strongbrown.55','strongbrown','strong brown',[128,70,27],'80461b',8406555],
['nbs-iscc-sc:strongbrown.74','strongbrown','strong brown',[153,101,21],'996515',10052885],
['nbs-iscc-sc:verydarkbrown.59','verydarkbrown','very dark brown',[66,37,24],'422518',4334872],
['nbs-iscc-sc:verydarkbrown.62','verydarkbrown','very dark brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-sc:verydarkbrown.78','verydarkbrown','very dark brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-sc:verydarkbrown.81','verydarkbrown','very dark brown',[72,60,50],'483c32',4734002],
['nbs-iscc-sc:verydarkgray.64','verydarkgray','very dark gray ',[91,80,79],'5b504f',5984335],
['nbs-iscc-sc:verydarkgray.113','verydarkgray','very dark gray ',[87,85,76],'57554c',5723468],
['nbs-iscc-sc:verydarkgray.266','verydarkgray','very dark gray ',[85,85,85],'555555',5592405],
['nbs-iscc-sc:verydarkgrayishbrown.81','verydarkgrayishbrown','very dark grayish brown',[72,60,50],'483c32',4734002],
['nbs-iscc-sc:verydarkgrayishbrown.95','verydarkgrayishbrown','very dark grayish brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-sc:veryduskyred.44','veryduskyred','very dusky red',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-sc:veryduskyred.47','veryduskyred','very dusky red',[67,48,46],'43302e',4403246],
['nbs-iscc-sc:verypalebrown.33','verypalebrown','very pale brown',[194,172,153],'c2ac99',12758169],
['nbs-iscc-sc:verypalebrown.73','verypalebrown','very pale brown',[250,214,165],'fad6a5',16438949],
['nbs-iscc-sc:verypalebrown.76','verypalebrown','very pale brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-sc:verypalebrown.79','verypalebrown','very pale brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-sc:weakred.19','weakred','weak red',[144,93,93],'905d5d',9461085],
['nbs-iscc-sc:weakred.42','weakred','weak red',[168,124,109],'a87c6d',11041901],
['nbs-iscc-sc:weakred.43','weakred','weak red',[121,68,59],'79443b',7947323],
['nbs-iscc-sc:weakred.45','weakred','weak red',[151,127,115],'977f73',9928563],
['nbs-iscc-sc:weakred.46','weakred','weak red',[103,76,71],'674c47',6769735],
['nbs-iscc-sc:white.9','white','white',[234,227,225],'eae3e1',15393761],
['nbs-iscc-sc:white.10','white','white',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-sc:white.73','white','white',[250,214,165],'fad6a5',16438949],
['nbs-iscc-sc:white.92','white','white',[240,234,214],'f0ead6',15788758],
['nbs-iscc-sc:white.93','white','white',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-sc:white.263','white','white',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-sc:white.264','white','white',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-sc:yellow.70','yellow','yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-sc:yellow.71','yellow','yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-sc:yellow.86','yellow','yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-sc:yellow.87','yellow','yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-sc:yellowishbrown.74','yellowishbrown','yellowish brown',[153,101,21],'996515',10052885],
['nbs-iscc-sc:yellowishbrown.77','yellowishbrown','yellowish brown',[130,102,68],'826644',8545860],
['nbs-iscc-sc:yellowred.54','yellowred','yellow red',[174,105,56],'ae6938',11430200],
['nbs-iscc-sc:yellowred.55','yellowred','yellow red',[128,70,27],'80461b',8406555]
    ];
}

sub _description {
    return {
          'subtitle' => 'Soil Color Charts',
          'title' => 'NBS/ISCC SC',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (SC) Soil Color Charts

    U.S. Department of Agriculture
    Soil Color Charts 

This combinatorial field-work dictionary\'s 50 names mape to 176 colors, using 76 of 267 centroids. Again, its focus on soil colors limits its general usefulness.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#SC]
'
        }

}

1;
