// copyall.cpp
//
// Syntax:  copyall source-file dest-file

#include "CopyFile.h"

#include <storage/Node.h>
#include <storage/Entry.h>
#include <storage/File.h>
#include <kernel/fs_attr.h>
#include <kernel/fs_info.h>
#include <kernel/fs_index.h>
#include <math.h>
#include <errno.h>
#include <unistd.h>
#include <new>

// simple min() function
inline off_t min(off_t a, off_t b) { return (a < b) ? a : b; }

// handy temporary-buffer class:  it's assigned an array pointer in its constructor,
// and assumes responsbility for deleting that array in its destructor.  This makes
// it vastly easier to handle dynamic arrays in code that can be exited in a variety
// of ways, such as miscellaneous return statements or exceptions.
class auto_buffer
{
public:
	explicit auto_buffer(char* p) : mPtr(p) { }
	~auto_buffer() { delete [] mPtr; }

	char* pointer() const { return mPtr; }

private:
	char* mPtr;
};

// class to automatically manage locking & unlocking a node with the same sort
// of exit-point unconcern that auto_buffer provides.
class NodeLocker
{
public:
	explicit NodeLocker(BNode& node) : mNode(node) { mErr = node.Lock(); }
	~NodeLocker() { mNode.Unlock(); }

	status_t LockCheck() const { return mErr; }		// did the lock succeed?

private:
	BNode& mNode;
	status_t mErr;
};

// routine to copy a file's attributes
static status_t
copy_attributes(const entry_ref& source, const entry_ref& dest, void *buffer, size_t bufferSize, bool createIndices)
{
	status_t err;

	// set up a BNode for the source file
	BNode srcNode(&source);
	if ((err = srcNode.InitCheck()) != B_NO_ERROR) return err;

	// likewise the destination file
	BNode destNode(&dest);
	if ((err = destNode.InitCheck()) != B_NO_ERROR) return err;

	// set up a copy buffer
	char* bufPtr = NULL;
	size_t bufSize;
	if (!buffer)		// no client-supplied copy buffer
	{
		// Allocate a very large buffer, for efficiency.  If we can't, try reducing the
		// allocation size until we either succeed or we reach zero size.
		bufSize = 512L * 1024L;
		while ((buffer = new (nothrow) char[bufSize]) == NULL)
		{
			bufSize -= 8L * 1024L;
			if (bufSize <= 0) return B_NO_MEMORY;
		}
	}
	else					// the client supplied a copy buffer
	{
		if (!bufferSize) return B_BAD_VALUE;		// can't specify a zero-size buffer
		bufSize = bufferSize;
	}

	// Arrange to have the buffer deleted automatically when this function returns.
	// If we were passed a buffer, bufPtr will be NULL so this will have no effect.
	auto_buffer buf(bufPtr);

	// if we're creating indices, we need more information about the source and
	// destination volumes
	fs_info srcInfo, destInfo;
	if (createIndices)
	{
		if (fs_stat_dev(source.device, &srcInfo) != 0) return errno;
		if (fs_stat_dev(dest.device, &destInfo) != 0) return errno;
	}

	// Lock the node so that nobody can add or delete attributes while we're
	// trying to copy them.  If we can't lock the node, then we return the
	// appropriate failure code.
	NodeLocker lock(srcNode);
	if ((err = lock.LockCheck()) != B_NO_ERROR) return err;

	// Iterate over the source file's attributes, copying them to the destination
	// as we go
	char attrName[B_ATTR_NAME_LENGTH];
	srcNode.RewindAttrs();		// rewind after locking to make sure we get all attrs
	while ((err = srcNode.GetNextAttrName(attrName)) == B_NO_ERROR)
	{
		attr_info info;

		// get the attribute's size and type
		if ((err = srcNode.GetAttrInfo(attrName, &info)) != B_NO_ERROR) return err;

		// if the attribute is indexed, ensure that the index exists on the destination volume
		if (createIndices)
		{
			struct index_info indexInfo;
			if (!fs_stat_index(srcInfo.dev, attrName, &indexInfo))
			{
				// this is allowed to fail, e.g. if the index already exists
				fs_create_index(destInfo.dev, attrName, indexInfo.type, 0);
			}
		}

		// read chunks from the attribute and write them to the destination file.
		// Negative return values from ReadAttr() or WriteAttr() are error codes,
		// so we abort the copy and return if we see one.
		ssize_t nRead;
		off_t offset = 0;
		while ((offset < info.size) && ((nRead = srcNode.ReadAttr(attrName, info.type, offset, buffer, bufSize)) > 0))
		{
			err = destNode.WriteAttr(attrName, info.type, offset, buffer, nRead);
			if (err < 0)
			{
				return err;
			}

			offset += nRead;
		}
	}

	// B_ENTRY_NOT_FOUND means that we've finished traversing
	// all of the attributes, so that's what we look for to indicate successful
	// completion.
	return (err == B_ENTRY_NOT_FOUND) ? B_NO_ERROR : err;
}

static status_t
copy_data(const entry_ref& source, const entry_ref& dest, void* buffer, size_t bufferSize)
{
	status_t err = B_NO_ERROR;
	char* bufPtr = NULL;
	size_t bufSize;

	if (!buffer)		// no client-supplied copy buffer
	{
		// Allocate a very large buffer, for efficiency.  If we can't, try reducing the
		// allocation size until we either succeed or we reach zero size.
		bufSize = 512L * 1024L;
		while ((buffer = new (nothrow) char[bufSize]) == NULL)
		{
			bufSize -= 8L * 1024L;
			if (bufSize <= 0) return B_NO_MEMORY;
		}
	}
	else					// the client supplied a copy buffer
	{
		if (!bufferSize) return B_BAD_VALUE;		// can't specify a zero-size buffer
		bufSize = bufferSize;
	}

	// Arrange to have the buffer deleted automatically when this function returns.
	// If we were passed a buffer, bufPtr will be NULL so this will have no effect.
	auto_buffer buf(bufPtr);

	// Open the source file for reading
	BFile srcFile(&source, O_RDONLY);
	if ((err = srcFile.InitCheck()) != B_NO_ERROR) return err;

	// Create & open the destination file for writing
	BFile destFile(&dest, O_WRONLY | O_CREAT | O_TRUNC);
	if ((err = destFile.InitCheck()) != B_NO_ERROR) return err;

	// Set the size of the destination file, to try to optimize contiguity of data
	// on the disk, then seek back to the beginning of the file for writing
	struct stat info;
	if ((err = srcFile.GetStat(&info)) != B_NO_ERROR) return err;
	if ((err = destFile.SetSize(info.st_size)) != B_NO_ERROR) return err;
	if ((err = destFile.Seek(0, SEEK_SET)) != B_NO_ERROR) return err;

	// Copy the file data in big chunks
	ssize_t nRead;
	while ((nRead = srcFile.Read(buffer, bufSize)) > 0)
	{
		err = destFile.Write(buffer, nRead);
		if (err < 0) return err;
	}

	return (nRead == 0) ? B_NO_ERROR : nRead;
}

// calculate the number of disk blocks required to store a file on the specified device
status_t preflight_file_size(const entry_ref& fileRef, const fs_info& srcInfo, const fs_info& destInfo, off_t* outBlocksNeeded)
{
	status_t err;
	off_t diskBlockSize = destInfo.block_size;

	// check the file's validity
	BFile file(&fileRef, O_RDONLY);
	if ((err = file.InitCheck()) != B_NO_ERROR) return err;

	// okay, the file is valid. First we calculate the number of blocks needed for
	// the file's ordinary data:  one for the file's inode, plus an appropriate number
	// to hold the actual file data.
	struct stat fileInfo;
	if ((err = file.GetStat(&fileInfo)) != B_NO_ERROR) return err;
	off_t blocksNeeded = 1 + (off_t) ceil(double(fileInfo.st_size) / double(diskBlockSize));

	// Now the interesting part:  how many disk blocks will the file's attributes consume
	//
	// attributes:
	//
	// check whether the attr. fits in the fast area (~780 bytes, but we estimate conservatively):
	//		- space used = attr. data size + name length + [constant overhead] bytes + [index size] + [attr dir overhead]
	//	not in fast area:
	//		- space used = 1 block + attr. data blocks + [index size] + [attr dir overhead]
	//
	// index size:
	//		- space = 0 if no index for the attribute (include test for creating index on destination volume!)
	//		- space = [index data size] + [index dir overhead]
	//		-- dominic's heuristic:  2x the data size for all indexed attributes + 3 blocks if > 512 bytes of indexed data
	//
	// attr dir overhead:
	//		- space = 

	if (destInfo.flags & B_FS_HAS_ATTR)
	{
		// lock the file's node while iterating over its attributes
		NodeLocker lock(file);
		if ((err = lock.LockCheck()) != B_NO_ERROR) return err;

		// other file systems may support attributes (e.g. NTFS) but only bfs
		// has this particular "fast" attribute area implementation
		bool fastSpaceAvailable = strcmp(destInfo.fsh_name, "bfs") ? false : true;
		bool hasSlowAttributes = false;

		off_t fastSpaceUsed = 0;
		off_t indexedData = 0;
		char attrName[B_ATTR_NAME_LENGTH];

		file.RewindAttrs();
		while ((err = file.GetNextAttrName(attrName)) == B_NO_ERROR)
		{
			attr_info info;
			if ((err = file.GetAttrInfo(attrName, &info)) != B_NO_ERROR) return err;

			// If the attribute will fit in the fast area when copied, record its presence there and go on to
			// the next area.  The size of an attribute in the fast area is:
			//
			//		4 bytes for the attribute's type
			//		2 bytes for a name-length indicator
			//		2 bytes for the attribute's data length indicator
			//		strlen(name) + 1 bytes for storing the attribute's name itself
			//		info.size bytes for the attribute data itself
			//
			// this works out to 9 + strlen(name) + info.size
			const size_t FAST_ATTR_OVERHEAD = 9;
			off_t fastLength = info.size + strlen(attrName) + FAST_ATTR_OVERHEAD;

			// the actual inode data is 232 bytes, so the remainder of the block is available
			// for fast attribute data
			const off_t  INODE_SIZE = 232;
			if (fastSpaceAvailable && (fastSpaceUsed + fastLength < diskBlockSize - INODE_SIZE))
			{
				fastSpaceUsed += fastLength;
			}
			else
			{
				// this attribute doesn't fit in the fast area, so it needs separate per-attribute storage.  Each
				// attribute will consume however many blocks it takes to store the attribute's data, plus
				// an inode block.
				blocksNeeded += 1 + off_t(ceil(double(info.size) / double(diskBlockSize)));
				hasSlowAttributes = true;
			}

			// if attributes are indexed, we may require additional disk space for their index entries.
			struct index_info indexInfo;
			if (!fs_stat_index(srcInfo.dev, attrName, &indexInfo))
			{
				indexedData += min(info.size, 256);
			}
		}

		// if there are non-fast attributes, we'll estimate a couple of blocks for the attribute directory
		if (hasSlowAttributes)
		{
			blocksNeeded += 2;
		}

		// account for indexed data
		blocksNeeded += 2 * indexedData / diskBlockSize + 2;
	}

	*outBlocksNeeded = blocksNeeded;
	return B_NO_ERROR;
}

status_t 
CopyFile(const entry_ref &source, const entry_ref &dest, void *buffer, size_t bufferSize, bool createIndices, bool preflight)
{
	status_t err;

	// check to see whether the destination volume has enough space for the file
	if (preflight)
	{
		fs_info srcInfo, destInfo;
		if (fs_stat_dev(source.device, &srcInfo) != 0) return errno;
		if (fs_stat_dev(dest.device, &destInfo) != 0) return errno;

		off_t blocksNeeded;
		if ((err = preflight_file_size(source, srcInfo, destInfo, &blocksNeeded)) != B_NO_ERROR) return err;

		// bfs needs a certain number of free blocks at all times to ensure that
		// transient file system demands can always be met; we account for that
		// free-block demand (currently 128 blocks) when checking to see whether
		// we have enough free blocks on the output volume.
		if (blocksNeeded > destInfo.free_blocks - 128) return B_DEVICE_FULL;
	}

	// preflight succeeded, so now copy the file
	err = copy_data(source, dest, buffer, bufferSize);
	if (!err) err = copy_attributes(source, dest, buffer, bufferSize, createIndices);
	return err;
}
