
#include "ToolLib.h"
#include "BitmapDocument.h"
#include "BitmapEditor.h"
#include "PaintApp.h"

/***********************************************/

void BRectBasedTool::DrawIcon(BView *drawInto, BRect location)
{
	int32 w = location.Width();
	int32 h = location.Height();
	w /= 8; h /= 8;
	location.InsetBy(w,h);
	DrawPrimitive(drawInto,location);
};

BRectBasedTool::BRectBasedTool(bool fill) : BTool()
{
	m_fill = fill;
};

void BRectBasedTool::StartUsing()
{
	m_theRect.Set(
		m_oldPoint.x,m_oldPoint.y,
		m_oldPoint.x,m_oldPoint.y);
		
	m_document->Lock();
		m_document->View()->SetDrawingMode(B_OP_INVERT);
		DrawPrimitive(m_document->View(),m_theRect);
	m_document->Unlock();
};

void BRectBasedTool::PointerMoved(BPoint newPoint)
{
	BRegion r;
	BRect oldRect = m_theRect;

	m_document->Lock();
		DrawPrimitive(m_document->View(),m_theRect);
		if (newPoint.y < m_invokePoint.y) {
			m_theRect.top = newPoint.y;
			m_theRect.bottom = m_invokePoint.y;
		} else {
			m_theRect.top = m_invokePoint.y;
			m_theRect.bottom = newPoint.y;
		};
		if (newPoint.x < m_invokePoint.x) {
			m_theRect.left = newPoint.x;
			m_theRect.right = m_invokePoint.x;
		} else {
			m_theRect.left = m_invokePoint.x;
			m_theRect.right = newPoint.x;
		};
		DrawPrimitive(m_document->View(),m_theRect);
		m_document->View()->Sync();
	m_document->Unlock();

	GetDeltaInval(&r,oldRect,m_theRect);
	m_document->SetDirty(&r);
};

void BRectBasedTool::StopUsing()
{
	BRegion r;

	m_document->Lock();
		DrawPrimitive(m_document->View(),m_theRect);
		m_document->View()->SetDrawingMode(B_OP_COPY);
		DrawPrimitive(m_document->View(),m_theRect);
		m_document->View()->Sync();
	m_document->Unlock();

	InvalPrimitive(&r,m_theRect);
	m_document->SetDirty(&r);	
};

/***********************************************/

void BRectTool::DrawPrimitive(BView *drawInto, BRect r)
{
	if (m_fill) drawInto->FillRect(r);
	else		drawInto->StrokeRect(r);
};

void BRectTool::GetDeltaInval(BRegion *inval, BRect oldRect, BRect newRect)
{
	BRegion tmpReg;
	if (m_fill)	{
		tmpReg.Set(oldRect);
		tmpReg.Exclude(newRect);
		inval->Set(newRect);
		inval->Exclude(oldRect);
		inval->Include(&tmpReg);
	} else {
		tmpReg.Set(oldRect);
		oldRect.InsetBy(1,1);
		tmpReg.Exclude(oldRect);
		oldRect.InsetBy(-1,-1);
		inval->Set(newRect);
		newRect.InsetBy(1,1);
		inval->Exclude(newRect);
		newRect.InsetBy(-1,-1);
		inval->Include(&tmpReg);
	};
};

void BRectTool::InvalPrimitive(BRegion *inval, BRect r)
{
	inval->Set(r);
	if (!m_fill) {
		r.InsetBy(1,1);
		if (r.IsValid()) inval->Exclude(r);
	};
};

/***********************************************/

void BEllipseTool::DrawPrimitive(BView *drawInto, BRect r)
{
	if (m_fill) drawInto->FillEllipse(r);
	else		drawInto->StrokeEllipse(r);
};

void BEllipseTool::GetDeltaInval(BRegion *inval, BRect oldRect, BRect newRect)
{
	inval->Set(oldRect);
	inval->Include(newRect);
};

void BEllipseTool::InvalPrimitive(BRegion *inval, BRect r)
{
	inval->Set(r);
};

/***********************************************/

BPoint iconPoints[4] = 
{
	BPoint(0.2,0.2),
	BPoint(0.1,1.3),
	BPoint(0.9,-0.3),
	BPoint(0.8,0.8),
};

void BPenTool::DrawIcon(BView *drawInto, BRect location)
{
	BPoint l_iconPoints[4];
	float width = location.Width();
	float height = location.Height();
	for (int32 i=0;i<4;i++) {
		l_iconPoints[i].x = location.left + iconPoints[i].x * width;
		l_iconPoints[i].y = location.top + iconPoints[i].y * height;
	};
	drawInto->SetPenSize(3.0);
	drawInto->StrokeBezier(l_iconPoints);
	drawInto->SetPenSize(1.0);
};

void BPenTool::StartUsing()
{
	m_document->Lock();
		m_document->View()->SetLineMode(B_ROUND_CAP,B_ROUND_JOIN,10);
		m_fudge = m_document->View()->PenSize() * 0.5 + 1.0;
	m_document->Unlock();
};

void BPenTool::PointerMoved(BPoint newPoint)
{
	BRegion reg;
	
	m_document->Lock();
		m_document->View()->StrokeLine(m_oldPoint,newPoint);
		m_document->View()->Sync();
	m_document->Unlock();

	BRect r(	min(newPoint.x-m_fudge,m_oldPoint.x-m_fudge),
				min(newPoint.y-m_fudge,m_oldPoint.y-m_fudge),
				max(newPoint.x+m_fudge,m_oldPoint.x+m_fudge),
				max(newPoint.y+m_fudge,m_oldPoint.y+m_fudge));
	reg.Set(r);

	m_document->SetDirty(&reg);
};

