/*
 * Copyright (C) 1996 Be, Inc.  All Rights Reserved
 */
#include "generic.h"

#define BUTTON_HEIGHT 18

static const rgb_color BACKGROUND_COLOR = { 216, 216, 216, 0 };
static const rgb_color WHITE_COLOR = { 255, 255, 255, 0 };
static const rgb_color BLACK_COLOR = { 0, 0, 0, 0 };

enum control_codes {
	OKHIT = 0x1000,
	CANCELHIT
};

static BRect WINRECT(		140,	140,	380,		300);

static BRect TITLEFRAME(	10,		10,		200,		25);
static BRect IPFRAME(		10,		30,		200,		55);
static BRect NMFRAME(		10,		70,		200,		85);

static BRect CANCELFRAME(	10,		100,	90,			115 + BUTTON_HEIGHT);
static BRect OKFRAME(		120,	100,	200,		115 + BUTTON_HEIGHT);

class GenericCWindow : public BWindow {
	const char *device;
	net_settings *ncw;
	BCallbackHandler *callback;
public:
	GenericCWindow(BRect frame,
			const char *title, 
			const char *device,
			net_settings *ncw,
			BCallbackHandler *callback,
			window_type type,
			ulong flags,
			ulong workspace = B_CURRENT_WORKSPACE);

	void Finish(status_t status) {
		callback->Done(status);
		Lock();
		Quit();
	}
	void MessageReceived(BMessage *msg) {
		switch (msg->what) {
		case OKHIT:
			Finish(B_NO_ERROR);
			break;
		case CANCELHIT:
			Finish(B_ERROR);
			break;
		default:
			BWindow::MessageReceived(msg);
			break;
		}
	}
	friend filter_result kd_filter(BMessage *, BHandler **, BMessageFilter *);
};


filter_result kd_filter(BMessage *msg, BHandler **target, BMessageFilter *f)
{
	GenericCWindow		*w = (GenericCWindow *) f->Looper();
	uchar		ch = 0;

	if (msg->FindInt8("byte", (int8 *)&ch) == B_NO_ERROR) {
		if (ch == B_ESCAPE) {
			w->Finish(B_ERROR);
		}
	}

	return B_DISPATCH_MESSAGE;
}

/*
 * compute a text rect, given a frame
 */
static BRect 
textrect(BRect frame)
{
 	frame.InsetBy(2, 2);
	frame.OffsetTo(0, 0);
	return (frame);
}

static void
textcontrol_init(
				 BTextControl *tc
				 )
{
	BTextView *tv;

	tv = (BTextView *)tc->ChildAt(0);
	tv->SetViewColor(WHITE_COLOR);
	tv->SetHighColor(BLACK_COLOR);
	tv->SetLowColor(WHITE_COLOR);
}

static void
title_init(
		   BTextView *view,
		   const char *text
		   )
{
	
	view->MakeEditable(FALSE);
	view->MakeSelectable(FALSE);
	view->SetViewColor(BACKGROUND_COLOR);
	view->SetHighColor(BLACK_COLOR);
	view->SetLowColor(BACKGROUND_COLOR);
	view->SetText(text);
}


static void
save_settings(
			  const char *device,
			  net_settings *ncw,
			  const char *config,
			  const char *link
			  )
{
	set_net_setting(ncw, device, NULL, NULL);
	set_net_setting(ncw, device, "DEVICECONFIG", config);
	set_net_setting(ncw, device, "DEVICELINK", link);
	set_net_setting(ncw, device, "DEVICETYPE", "ETHERNET");
}
			   

status_t
GenericConfig::Config(
					  const char *device,
					  net_settings *ncw,
					  BCallbackHandler *callback,
					  bool autoconf
					  )
{
	GenericCWindow *win;

	if (autoconf) {
		save_settings(device, ncw, config, link);
		return (B_NO_ERROR);
	}
	win = new GenericCWindow(WINRECT, title, device, ncw, callback, B_MODAL_WINDOW, 
					  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_NOT_CLOSABLE);

	win->Show();
	return (B_NO_ERROR);
}


int
GenericDevice::Start(const char *device)
{
	char devicelink[NC_MAXVALLEN+1];
	ether_init_params_t params;

	if (!netconfig_find(device, "DEVICELINK", devicelink,
						sizeof(devicelink))) {
		return (B_ERROR);
	}
	return (Init(devicelink, (char *)&params));
}


GenericCWindow::GenericCWindow(BRect frame,
			const char *wtitle, 
			const char *device,
			net_settings *ncw,
			BCallbackHandler *callback,
			window_type type,
			ulong flags,
			ulong workspace)
	: BWindow(frame, wtitle, type,  flags, workspace) 
{
	BButton *okbut;
	BButton *cancelbut;
	BView *background;
	BMenuItem *mitem;
	rgb_color hi;
	rgb_color lo;
	BTextView *titleview;
	int status;

	this->device = device;
	this->ncw = ncw;
	this->callback = callback;

	Lock();
	BMessageFilter *filter = new BMessageFilter(B_KEY_DOWN, kd_filter);
	AddCommonFilter(filter);

	callback = callback;
	titleview = new BTextView(TITLEFRAME, "title", textrect(TITLEFRAME), 
						  B_FOLLOW_NONE, B_WILL_DRAW);

	okbut = new BButton(OKFRAME, "", "Set", new BMessage(OKHIT));
	cancelbut = new BButton(CANCELFRAME, "", "Cancel", new BMessage(CANCELHIT));


	background = new BView(Bounds(), "", B_FOLLOW_ALL, B_WILL_DRAW);

	AddChild(background);
	background->SetViewColor(BACKGROUND_COLOR);


	background->AddChild(titleview);
	background->AddChild(cancelbut);
	background->AddChild(okbut);
	okbut->MakeDefault(TRUE);

	title_init(titleview, wtitle);
	
	Unlock();
}


