// Copyright 1992-98, Be Incorporated, All Rights Reserved.
//
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <Debug.h>
#include <Directory.h>
#include <Path.h>

#include "scsi.h"
#include "scsiprobe_driver.h"
#include "CDEngine.h"

PeriodicWatcher::PeriodicWatcher(const char *const device)
	:	Notifier(),
		device(device)
{
}

 
PeriodicWatcher::PeriodicWatcher(BMessage *)
	:	device(0)
{
	// not implemented yet
	PRINT(("under construction"));
}

BHandler *
PeriodicWatcher::RecipientHandler() const
{
	 return engine;
}

void
PeriodicWatcher::DoPulse()
{
	// control the period here
	if (UpdateState())
		Notify();
}

void 
PeriodicWatcher::UpdateNow()
{
	UpdateState();
}


PlayState::PlayState(const char *const device)
	:	PeriodicWatcher(device),
		oldState(kNoCD)
{
}

bool
PlayState::UpdateState()
{
	// check the current CD play state and force a notification to
	// be sent if it changed from last time
	scsi_position pos;
	int32 dev = open(device, 0);
	if (dev <= 0)
		return CurrentState(kNoCD);

	status_t result = ioctl(dev, B_SCSI_GET_POSITION, &pos);
	close(dev);

	if (result != B_NO_ERROR)
		return CurrentState(kNoCD);
	else if ((!pos.position[1]) || (pos.position[1] >= 0x13) ||
	   ((pos.position[1] == 0x12) && (!pos.position[6])))
		return CurrentState(kStopped);
	else if (pos.position[1] == 0x11)
		return CurrentState(kPlaying);
	else
		return CurrentState(kPaused);
}

CDState
PlayState::GetState() const
{
	return oldState;
}

bool
PlayState::CurrentState(CDState newState)
{
	if (newState != oldState) {
		oldState = newState;
		return true;
	}
	return false;
}

TrackState::TrackState(const char *const device)
	:	PeriodicWatcher(device),
		currentTrack(0)
{
}

int32
TrackState::GetTrack() const
{
	return currentTrack;
}

bool
TrackState::UpdateState()
{
	// check the current CD track number and force a notification to
	// be sent if it changed from last time
	scsi_position pos;
	int32 dev = open(device, 0);
	if (dev <= 0)
		return CurrentState(-1);

	status_t result = ioctl(dev, B_SCSI_GET_POSITION, &pos);
	close(dev);

	if (result != B_NO_ERROR)
		return CurrentState(-1);
		
	if (!pos.position[1] || pos.position[1] >= 0x13
		|| (pos.position[1] == 0x12 && !pos.position[6]))
		return CurrentState(0);
	else
		return CurrentState(pos.position[6]);
}

int32 
TrackState::GetNumTracks() const
{
	// get the number of tracks on the current CD
	int32 dev = open(device, 0);
	if (dev <= 0)
		return 0;

	scsi_toc toc;

	status_t result = ioctl(dev, B_SCSI_GET_TOC, &toc);
	close(dev);

	if (result != B_NO_ERROR)
		return 0;
	
	return toc.toc_data[3];
}

bool
TrackState::CurrentState(int32 track)
{
	if (track != currentTrack) {
		currentTrack = track;
		return true;
	}
	return false;
}

bool 
TimeState::UpdateState()
{
	// check the current CD time and force a notification to
	// be sent if it changed from last time
	// currently only supports global time
	scsi_position pos;
	int32 dev = open(device, 0);
	if (dev <= 0)
		return CurrentState(-1, -1);

	status_t result = ioctl(dev, B_SCSI_GET_POSITION, &pos);
	close(dev);

	if (result != B_NO_ERROR)
		return CurrentState(-1, -1);
	else if ((!pos.position[1]) || (pos.position[1] >= 0x13) ||
	   ((pos.position[1] == 0x12) && (!pos.position[6])))
		return CurrentState(0, 0);
	else
		return CurrentState(pos.position[9], pos.position[10]);
}

bool 
TimeState::CurrentState(int32 minutes, int32 seconds)
{
	if (minutes == oldMinutes && seconds == oldSeconds)
		return false;
	oldMinutes = minutes;
	oldSeconds = seconds;
	return true;
}

void 
TimeState::GetTime(int32 &minutes, int32 &seconds) const
{
	minutes = oldMinutes;
	seconds = oldSeconds;
}

CDEngine::CDEngine(const char *device)
	:	BHandler("CDEngine"),
		playState(device),
		trackState(device),
		timeState(device),
		volumeState(device),
		device(device)
{
}

CDEngine::CDEngine(BMessage *message)
	:	BHandler(message),
		playState(message),
		trackState(message),
		timeState(message),
		volumeState(message),
		device(0)
{
}

CDEngine::~CDEngine()
{
}

void 
CDEngine::AttachedToLooper(BLooper *looper)
{
	looper->AddHandler(this);
	playState.AttachedToLooper(this);
	trackState.AttachedToLooper(this);
	timeState.AttachedToLooper(this);
	volumeState.AttachedToLooper(this);
}

void
CDEngine::PlayOrPause()
{
	// if paused, or stopped, plays, if playing, pauses
	playState.UpdateNow();
	switch (playState.GetState()) {
		case kNoCD:
			return;
		case kStopped:
			Play();
			break;
		case kPaused:
			PlayContinue();
			break;
		case kPlaying:
			Pause();
			break;
	}
}

void 
CDEngine::Pause()
{
	// pause the CD
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error opening %s\n", device));
		return;
	}
	status_t result = ioctl(dev, B_SCSI_PAUSE_AUDIO);
	if (result != B_NO_ERROR) {
		PRINT(("error %s pausing, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

void 
CDEngine::Play()
{
	// play the CD
	if (playState.GetState() == kNoCD)
		// no CD available, bail out
		return;

	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error opening %s\n", device));
		return;
	}

	scsi_play_track track;

	track.start_track = 1;
	track.start_index = 1;
	track.end_track = 99;
	track.end_index = 1;

	status_t result = ioctl(dev, B_SCSI_PLAY_TRACK, &track);
	if (result != B_NO_ERROR) {
		PRINT(("error %s playing track, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

void
CDEngine::PlayContinue()
{
	// continue after a pause
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error opening %s\n", device));
		return;
	}
	status_t result = ioctl(dev, B_SCSI_RESUME_AUDIO);
	if (result != B_NO_ERROR) {
		PRINT(("error %s resuming, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

void 
CDEngine::Stop()
{
	// stop a playing CD
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error opening %s\n", device));
		return;
	}
	status_t result = ioctl(dev, B_SCSI_STOP_AUDIO);
	if (result != B_NO_ERROR) {
		PRINT(("error %s stoping, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

void 
CDEngine::Eject()
{
	// eject a CD
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error ejecting %s\n", device));
		return;
	}
	status_t result = ioctl(dev, B_SCSI_EJECT);
	if (result != B_NO_ERROR) {
		PRINT(("error %s ejecting, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

void 
CDEngine::SkipOneForward()
{
	// skip forward by one track
	CDState state = playState.GetState();
	
	if (state == kNoCD)
		return;

	bool wasPaused = state == kPaused
		|| state == kStopped;

	SelectTrack(trackState.GetTrack() + 1);
	if (wasPaused)
		// make sure we don't start playing if we were paused before
		Pause();
}

void 
CDEngine::SkipOneBackward()
{
	// skip backward by one track
	CDState state = playState.GetState();
	
	if (state == kNoCD)
		return;

	bool wasPaused = state == kPaused
		|| state == kStopped;

	int32 track = trackState.GetTrack();
	
	if (track > 1)
		track--;
	
	SelectTrack(track);

	if (wasPaused)
		// make sure we don't start playing if we were paused before
		Pause();
}

void 
CDEngine::StartSkippingBackward()
{
	// start skipping
	CDState state = playState.GetState();
	
	if (state == kNoCD)
		return;

	scsi_scan scan;
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error skipping backward %s\n", device));
		return;
	}
	scan.direction = -1;
	scan.speed = 1;
	status_t result = ioctl(dev, B_SCSI_SCAN, &scan);
	if (result != B_NO_ERROR) {
		PRINT(("error %s skipping backward, device %s\n",
			strerror(result), device));
		return;
	}
	close(dev);

}

void 
CDEngine::StartSkippingForward()
{
	// start skipping
	CDState state = playState.GetState();
	
	if (state == kNoCD)
		return;

	scsi_scan scan;
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error skipping forward %s\n", device));
		return;
	}
	scan.direction = 1;
	scan.speed = 1;
	status_t result = ioctl(dev, B_SCSI_SCAN, &scan);
	if (result != B_NO_ERROR) {
		PRINT(("error %s skipping forward, device %s\n",
			strerror(result), device));
		return;
	}
	close(dev);
}

void 
CDEngine::StopSkipping()
{
	// stop skipping
	CDState state = playState.GetState();
	
	if (state == kNoCD)
		return;

	scsi_scan scan;
	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error in stop skipping %s\n", device));
		return;
	}
	scan.direction = 0;
	scan.speed = 1;
	status_t result = ioctl(dev, B_SCSI_SCAN, &scan);
	if (result != B_NO_ERROR) {
		PRINT(("error %s in stop skipping, device %s\n",
			strerror(result), device));
		return;
	}

	result = ioctl(dev, B_SCSI_RESUME_AUDIO);
	if (result != B_NO_ERROR) {
		PRINT(("error %s resuming, device %s\n", strerror(result), device));
		return;
	}

	close(dev);
}

void 
CDEngine::SelectTrack(int32 trackNumber)
{
	// go to a selected track
	if (playState.GetState() == kNoCD)
		return;

	int32 dev = open(device, 0);
	if (dev <= 0) {
		PRINT(("error opening %s\n", device));
		return;
	}

	scsi_play_track track;

	track.start_track = trackNumber;
	track.start_index = 1;
	track.end_track = 99;
	track.end_index = 1;

	status_t result = ioctl(dev, B_SCSI_PLAY_TRACK, &track);
	if (result != B_NO_ERROR) {
		PRINT(("error %s playing track, device %s\n", strerror(result), device));
		return;
	}
	close(dev);
}

const bigtime_t kPulseRate = 500000;

void
CDEngine::DoPulse()
{
	// this is the CDEngine's hearbeat; Since it is a Notifier, it checks if
	// any values changed since the last hearbeat and sends notices to observers

	bigtime_t time = system_time();
	if (time > lastPulse && time < lastPulse + kPulseRate)
		return;
	
	// every pulse rate have all the different state watchers check the
	// curent state and send notifications if anything changed
	
	lastPulse = time;

	playState.DoPulse();
	trackState.DoPulse();
	timeState.DoPulse();
	volumeState.DoPulse();
}

void 
CDEngine::MessageReceived(BMessage *message)
{
	if (message->what == 'slTk') {
		// handle message from menu selection
		int32 track;
		if (message->FindInt32("track", &track) == B_OK)
			SelectTrack(track);
			
	} else
		// handle observing
		if (!Notifier::HandleObservingMessages(message)
		&& !CDEngineFunctorFactory::DispatchIfFunctionObject(message))
		BHandler::MessageReceived(message);
}

// here is some funky code used to detect an ATAPI or SCSI CDROM

struct ide_ctrl_info {
    bool	ide_0_present;	/* wether the controller is present or not */
    bool	ide_0_master_present;
    bool	ide_0_slave_present;
    int		ide_0_master_type;/* type of device: B_DISK, B_CD etc... */
    int		ide_0_slave_type;	/* same for the slave */
    bool	ide_1_present;	/* wether the controller is present or not */
    bool	ide_1_master_present;
    bool	ide_1_slave_present;
    int		ide_1_master_type;/* type of device: B_DISK, B_CD etc... */
    int		ide_1_slave_type;	/* same for the slave */
};

enum {
	IDE_GET_DEVICES_INFO = B_DEVICE_OP_CODES_END + 50,
	IDE_BYTE_SWAP = B_DEVICE_OP_CODES_END + 150, 
	IDE_NO_BYTE_SWAP
};

static bool
try_dir(char *result, const char *directory)
{
	// sniff out any cd players out there, return the first one you find
	// if we supported multiple CDPlayers, we would have to get more creative here
	bool add;
	const char *name;
	int fd;

	BDirectory dir;
	dir.SetTo(directory);
	if (dir.InitCheck() == B_NO_ERROR) {
		dir.Rewind();

		BEntry entry;
		while (dir.GetNextEntry(&entry) >= 0) {
			BPath path;
			entry.GetPath(&path);
			name = path.Path();
			if (entry.IsDirectory())
				try_dir(result, name);
			else if (strstr(name, "/raw")) {
				add = false;
				if (strstr(name, "/scsi")) {
					fd = open("/dev/scsiprobe", 0);
					if (fd >= 0) {
						scsiprobe_inquiry inquiry;
						inquiry.path = name[15] - '0';
						inquiry.id = name[16] - '0';
						inquiry.lun = name[17] - '0';
						inquiry.len = 36;
						if (ioctl(fd, B_SCSIPROBE_INQUIRY, &inquiry) == B_NO_ERROR)
							add = ((inquiry.data[0] & 0x1f) == 5);
						close(fd);
					}
				} else if (strstr(name, "/ide")) {
					ide_ctrl_info ide_info;
					fd = open("/dev/disk/ide/rescan", 0);
					if (fd >= 0) {
						if (ioctl(fd, IDE_GET_DEVICES_INFO, &ide_info) == B_NO_ERROR) {
							if (name[14] == '0') {
								if (strstr(name, "master")
									&& ide_info.ide_0_master_type == B_CD)
									add = true;
								else if (strstr(name, "slave")
									&& ide_info.ide_0_slave_type == B_CD)
									add = true;
							}
							else {
								if (strstr(name, "master")
									&& ide_info.ide_1_master_type == B_CD)
									add = true;
								else if (strstr(name, "slave")
									&& ide_info.ide_1_slave_type == B_CD)
									add = true;
							}
						}
						close(fd);
					}
				}
				if (add) {
					strcpy(result, name);
					return true;
				}
			}
		}
	}
	return false;
}


bool
CDEngine::FindCDPlayer(char *result)
{
	return try_dir(result, "/dev/disk");
}

