// Copyright 1992-98, Be Incorporated, All Rights Reserved.
//
// This defines an engine that tracks the state of the CD player
// and supports the CD player control and status calls

#ifndef __CD_ENGINE__
#define __CD_ENGINE__

#include <Looper.h>
#include <View.h>

#include "Observer.h"
#include "FunctionObjectMessage.h"

class CDEngine;
class PeriodicWatcher : public Notifier {
// watcher sits somewhere were it can get pulses and makes sure
// notices get sent if state changes
public:
	PeriodicWatcher(const char *const device);
	PeriodicWatcher(BMessage *);
	void DoPulse();
	void UpdateNow();

	void AttachedToLooper(CDEngine *engine)
		{ this->engine = engine; }

	virtual BHandler *RecipientHandler() const;

protected:
	const char *const device;
private:	
	virtual bool UpdateState() = 0;
	CDEngine *engine;
};

enum CDState {
	kNoCD,
	kStopped,
	kPaused,
	kPlaying,
	kSkipping
};

class PlayState : public PeriodicWatcher {
	// this watcher sends notices to observers that are interrested
	// about play state changes
public:
	PlayState(const char *const device);
	PlayState(BMessage *message)
		:	PeriodicWatcher(message)
		{}

	CDState GetState() const;
private:
	bool UpdateState();
	bool CurrentState(CDState);
	CDState oldState;
};

class TrackState : public PeriodicWatcher {
	// this watcher sends notices to observers that are interrested
	// about changes in the current track
public:
	TrackState(const char *const device);
	TrackState(BMessage *message)
		:	PeriodicWatcher(message)
		{}

	int32 GetTrack() const;
	int32 GetNumTracks() const;

private:
	bool UpdateState();
	bool CurrentState(int32);
	int32 currentTrack;
};

class TimeState : public PeriodicWatcher {
	// this watcher sends notices to observers that are interrested
	// about changes in the current time
public:
	TimeState(const char *const device)
		:	PeriodicWatcher(device)
		{ }
	TimeState(BMessage *message)
		:	PeriodicWatcher(message)
		{}

	void GetTime(int32 &minutes, int32 &seconds) const;

private:
	bool UpdateState();
	bool CurrentState(int32 minutes, int32 seconds);
	int32 oldMinutes;
	int32 oldSeconds;
};

class VolumeState : public PeriodicWatcher {
	// this watcher sends notices to observers that are interrested
	// about changes in the current volume
	// currently not used yet
public:
	VolumeState(const char *const device)
		:	PeriodicWatcher(device)
		{ }
	VolumeState(BMessage *message)
		:	PeriodicWatcher(message)
		{}

	bool UpdateState() { return true; }
	virtual void DoPulse() {}
	int32 GetVolume() const;
};

class CDEngine : public BHandler {
	// The CD engine defines all the different CD control calls; also,
	// it hosts the different state watchers and helps them send notices
	// to observers about the CD state changes
public:
	CDEngine(const char *device);
	CDEngine(BMessage *);

	virtual ~CDEngine();

	// observing supprt
	virtual void MessageReceived(BMessage *);
	void AttachedToLooper(BLooper *);
	void DoPulse();

	// control calls
	void PlayOrPause();
	void PlayContinue();
	void Play();
	void Pause();
	void Stop();
	void Eject();
	void SkipOneForward();
	void SkipOneBackward();
	void StartSkippingBackward();
	void StartSkippingForward();
	void StopSkipping();
	void SelectTrack(int32);
	

	TrackState *TrackStateWatcher()
		{ return &trackState; }
		// to find the current Track, you may call the GetTrack function
		// TrackState defines

	PlayState *PlayStateWatcher()
		{ return &playState; }
		// to find the current play state, you may call the GetState function
		// PlayState defines

	TimeState *TimeStateWatcher()
		{ return &timeState; }
		// to find the current location on the CD, you may call the GetTime function
		// TimeState defines

	static bool FindCDPlayer(char *);
	
private:
	const char *const device;

	PlayState playState;
	TrackState trackState;
	TimeState timeState;
	VolumeState volumeState;
	
	bigtime_t lastPulse;
};


// some function object glue
class CDEngineFunctorFactory : public FunctorFactoryCommon {
public:
	static BMessage *NewFunctorMessage(void (CDEngine::*func)(),
		CDEngine *target)
		{ return NewMessage(
			&PlainMemberFunctionObject<void (CDEngine::*)(),
				CDEngine>(func, target)); }

	static BMessage *NewFunctorMessage(void (CDEngine::*func)(ulong),
		CDEngine *target, ulong param)
		{ return NewMessage(
			&SingleParamMemberFunctionObject<void (CDEngine::*)(ulong),
				CDEngine, ulong>(func, target, param)); }
};


#endif