//******************************************************************************
//
//	File:			FindPanel.cpp
//
//	Written by:		Douglas Wright (with code stolen from Tracker;)
//
//	Copyright 1997, Be Incorporated, All Rights Reserved.
//
//******************************************************************************


#include <stdlib.h>
#include <string.h>
#include <parsedate.h>

#include <Debug.h>
#include <Application.h>
#include <Box.h>
#include <Button.h>
#include <CheckBox.h>
#include <View.h>
#include <MenuField.h>
#include <MenuItem.h>
#include <PopUpMenu.h>
#include <FindDirectory.h>
#include <Mime.h>
#include <Path.h>
#include <Query.h>
#include <Directory.h>
#include <Volume.h>
#include <VolumeRoster.h>
#include <NodeInfo.h>
#include <TextControl.h>

#include <fs_attr.h>

#include "Attributes.h"
#include "Commands.h"
#include "ContainerWindow.h"
#include "FindPanel.h"
#include "FSUtils.h"
#include "MimeTypes.h"
#include "Tracker.h"
#include "Utilities.h"

extern int item_compare(const void *i1, const void *i2);

const BRect kInitialRect(100, 100, 530, 210);
const int32 kInitialAttrModeWindowHeight = 140;
const int32 kIncrementPerAttribute = 30;

/* Send Panel */

SendPanel::SendPanel()
	BWindow(kInitialRect, "Send", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
 	BRect initialRect = SendView::InitialViewSize();
	ResizeTo(initialRect.Width(), initialRect.Height());

	Run();
}

~SendPanel::SendPanel()
{
}

void
SendPanel::SetMessenger(BMessenger *msgr)
{
	fMessenger = msgr;
}

void
SendPanel::MessageReceived(BMessage *message)
{
	switch (message->what) {	
		case SEND_BUTTON:
			DoFind();
			break;

		default:
			BView::MessageReceived(message);
			break;
	}
}


/* Send View */
SendView::SendView(BRect frame)
	:	BView(frame, "MainView", B_FOLLOW_ALL, B_WILL_DRAW)
{
}


SendView::~SendView()
{
}

void
SendView::AttachedToWindow()
{
	SetViewColor(view_background_color);
	SetLowColor(view_background_color);
	
	ASSERT(dynamic_cast<TFindPanel *>(Window()));

	BRect bounds(Bounds());
	bounds.InsetBy(15, 30);
	bounds.bottom -= 10;
	AddChild(new BBox(bounds, "Box"));

	// add popup for message types
	fMessageTypeMenu = new BPopUpMenu("MessageTypeMenu");
	AddMessageTypesToMenu();

	BRect rect(bounds);
	rect.top -= 25;
	rect.right = rect.left + 150;
	rect.bottom = rect.top + 15;
	BMenuField* menuField = new BMenuField(rect, "MessageTypeMenu", "", fMessageTypeMenu);
	menuField->SetDivider(0.0);
	AddChild(menuField);

	// add Search button
	rect = Bounds();
	rect.left = rect.right - 80;
	rect.top = rect.bottom - 30;
	rect.right = rect.left + 60;
	rect.bottom = rect.top + 20;
	BButton* button = new BButton(rect, "send", "Send",
		new BMessage(SEND_BUTTON), B_FOLLOW_RIGHT + B_FOLLOW_BOTTOM);
	AddChild(button);
	button->MakeDefault(true);
}


BRect 
SendView::InitialViewSize()
{
	return kInitialRect;
}

static void
PopUpMenuSetTitle(BMenu *menu, const char *title)
{
	// This should really be in BMenuField
	BMenu *bar = menu->Supermenu();

	ASSERT(bar);
	ASSERT(bar->ItemAt(0));
	if (!bar || !bar->ItemAt(0))
		return;

	bar->ItemAt(0)->SetLabel(title);
}

void
SendView::MessageReceived(BMessage* msg)
{
	BView *view;

	switch (msg->what) {
		case _BY_ATTR_:
		case _BY_NAME_:
		case _BY_FORMULA_:
			SwitchMode(msg->what);
			break;

		case _ADD_:
			AddAttrView();
			break;

		case _REMOVE_:
			RemoveAttrView();
			break;

		default:
			BView::MessageReceived(msg);
			break;
	}
}

void
SendView::BuildMessage(BMessage *msg) const
{
	// go through each attrview and add the attr and comparison info
	for (int32 i = 0; i < attrViewList.CountItems(); i++) {

		SpecifierView *view = specViewList.ItemAt(i);
		BTextControl *textControl = dynamic_cast<BTextControl *>
			(view->FindView("TextEntry"));
		if (!textControl)
			return;

		BMenuField *menuField = dynamic_cast<BMenuField *>(view->FindView("MenuField"));
		if (!menuField)
			return;

		BMenuItem *item = menuField->Menu()->FindMarked();
		if (item) {
			BMessage *message = item->Message();
			int32 type;
			if (message->FindInt32("type", &type) == B_NO_ERROR) {
				const char *str;
				if (message->FindString("name", &str) == B_NO_ERROR)
					query->PushAttr(str);
				else
					query->PushAttr(item->Label());
				switch (type) {
					case B_STRING_TYPE:
						query->PushString(textControl->TextView()->Text(), true);
						break;

					case B_OFF_T_TYPE:
						off_t val = StringToScalar(textControl->TextView()->Text());
						query->PushInt64(val);
						break;

					case B_TIME_TYPE:
						time_t a_time = parsedate(textControl->TextView()->Text(), time(NULL));
						query->PushInt32(a_time);
						break;

					case B_INT32_TYPE:
						char *end;
						long a_long = strtol(textControl->TextView()->Text(), &end, 10);
						query->PushInt32(a_long);
						break;
				}
			}

			query_op op;
			if (item->Submenu()->FindMarked()
				&& (message = item->Submenu()->FindMarked()->Message()) != 0) {
				message->FindInt32("operator", (int32*)&op);
				query->PushOp(op);
			} else
				query->PushOp(B_EQ);

			// add logic based on selection in Logic menufield
			if (i > 0) {
				TAttrView *prevView = attrViewList.ItemAt(i - 1);
				menuField = dynamic_cast<BMenuField *>(prevView->FindView("Logic"));
				if (menuField
					&& (item = menuField->Menu()->FindMarked()) != 0) {
					message = item->Message();
					message->FindInt32("combine", (int32*)&op);
					query->PushOp(op);
				} else
					query->PushOp(B_AND);
			}
		}
	}
}


void
SendView::AddMessageTypesToMenu(BPopUpMenu* menu)
{
	BList		list;
	BMessage	msg;
	BMessage*	item_msg;
	int32		i;
	BMimeType	mimetype;
	const char*	str;
	const char*	desc;
	BMessage*	type_list;	

	be_app->Lock();
	type_list = ((TTracker*)be_app)->MimeTypeList();

	i = 0;
	while (type_list->FindString("mimetype", i, &str) == B_NO_ERROR) {
		type_list->FindString("desc", i++, &desc);

		item_msg = new BMessage(_MIMETYPE_ITEM_);
		item_msg->AddString("mimetype", str);
		list.AddItem(new BMenuItem(desc, item_msg));
	}

	be_app->Unlock();

	// add items in sorted order
	qsort((uchar*)list.Items(), list.CountItems(), sizeof(BMenuItem*), item_compare);

	for (i = 0; i < list.CountItems(); i++)
		menu->AddItem((BMenuItem*)list.ItemAt(i));

	menu->SetTargetForItems(this);
}

void
SendView::AddMessageTypesToMenu()
{
	BMessage*	item_msg;

	item_msg = new BMessage(_MIMETYPE_ITEM_);
	item_msg->AddString("mimetype", ALL_MIME_TYPES);
	MimeTypeMenu()->AddItem(new BMenuItem("All files and folders", item_msg));
	MimeTypeMenu()->AddSeparatorItem();
	MimeTypeMenu()->ItemAt(0)->SetMarked(true);

	AddMessageTypesToMenu(fMessageTypeMenu);
}

void
SendView::AddOneSpecifierItem(BBox *box, const BRect rect)
{
	SpecifierView *specView = new SpecifierView(rect);
	specViewList.AddItem(specView);

	box->AddChild(specView);
}

void
SendView::SetUpAddRemoveButtons(BBox *box)
{
	BButton *button = (BButton*)Window()->FindView("remove");
	if (!button) {	
		BRect rect = box->Bounds();
		rect.InsetBy(5, 10);
		rect.top = rect.bottom - 20;
		rect.right = rect.left + 40;
		
		button = new BButton(rect, "add", "Add", new BMessage(_ADD_),
			B_FOLLOW_RIGHT + B_FOLLOW_BOTTOM);
		button->SetTarget(this);
		box->AddChild(button);
	
		rect.OffsetBy(50, 0);
		rect.right = rect.left + 55;
		button = new BButton(rect, "remove", "Remove", new BMessage(_REMOVE_),
			B_FOLLOW_RIGHT + B_FOLLOW_BOTTOM);
		
		button->SetEnabled(false);
		button->SetTarget(this);
		box->AddChild(button);
	}
	// enable remove button as needed
	button->SetEnabled(specViewList.CountItems() > 1);
}

void
SendView::AddSpecifierView()
{
	BBox *box = dynamic_cast<BBox *>(FindView("Box"));
	BRect bounds(Bounds());

	SpecifierView *previous = specViewList.LastItem();
	
	if (previous) 
		Window()->ResizeBy(0, 30);

	bounds = Bounds();
	bounds.InsetBy(15, 30);
	bounds.bottom -= 10;

	if (previous) {
		box->ResizeTo(bounds.Width(), bounds.Height());
		bounds = previous->Frame();
		bounds.OffsetBy(0, 30);
	} else {
		bounds = box->Bounds();
		bounds.InsetBy(5, 5);
		bounds.bottom = bounds.top + 20;
	}
	AddOneAttributeItem(box, bounds);

	// add logic to previous attrview
	if (previous) 
		previous->AddLogicMenu();

	SetUpAddRemoveButtons(box);
}

void
SendView::RemoveSpecifierView()
{
	if (specViewList.CountItems() < 2)
		return;

	BBox *box = (BBox*)FindView("Box");
	SpecifierView *specView = specViewList.LastItem();
	if (!box || !specView)
		return;
	
	Window()->ResizeBy(0, -30);
	BRect bounds(Bounds());
	bounds.InsetBy(15, 30);
	bounds.bottom -= 10;
	box->ResizeTo(bounds.Width(), bounds.Height());

	specViewList.RemoveItem(specView);
	specView->RemoveSelf();
	delete specView;

	specView = specViewList.ItemAt(specViewList.CountItems() - 1);
	specView->RemoveLogicMenu();

	if (specViewList.CountItems() != 1)
		return;
		
	BButton *button = (BButton*)Window()->FindView("remove");
	if (button)
		button->SetEnabled(false);
}


/* SpecifierView */
SpecView::SpecView(BRect frame)
	:	BView(frame, "SpecView", B_FOLLOW_NONE, B_WILL_DRAW)
{
	BMessage*		msg;
	BRect			bounds;
	BMenuItem*		item;

	SetViewColor(view_background_color);
	SetLowColor(view_background_color);

	BPopUpMenu *menu = new BPopUpMenu("PopUp");

	// add NAME attribute to popup
	BMenu *submenu = new BMenu("Name");
	submenu->SetRadioMode(true);
	submenu->SetFont(be_plain_font);
	msg = new BMessage(_ATTR_MAIN_);
	msg->AddString("name", "name");
	msg->AddInt32("type", B_STRING_TYPE);
	item = new BMenuItem(submenu, msg);
	menu->AddItem(item);

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_CONTAINS);
	submenu->AddItem(new BMenuItem("contains", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_EQ);
	submenu->AddItem(new BMenuItem("is", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_NE);
	submenu->AddItem(new BMenuItem("is not", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_BEGINS_WITH);
	submenu->AddItem(new BMenuItem("starts with", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_ENDS_WITH);
	submenu->AddItem(new BMenuItem("ends with", msg));

	// mark first item
	menu->ItemAt(0)->SetMarked(true);
	submenu->ItemAt(0)->SetMarked(true);

	// add SIZE attribute
	submenu = new BMenu("Size");
	submenu->SetRadioMode(true);
	submenu->SetFont(be_plain_font);
	msg = new BMessage(_ATTR_MAIN_);
	msg->AddString("name", "size");
	msg->AddInt32("type", B_OFF_T_TYPE);
	item = new BMenuItem(submenu, msg);
	menu->AddItem(item);

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_GE);
	submenu->AddItem(new BMenuItem("greater than", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_LE);
	submenu->AddItem(new BMenuItem("less than", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_EQ);
	submenu->AddItem(new BMenuItem("is", msg));

	// add "modified" field
	submenu = new BMenu("Modified");
	submenu->SetRadioMode(true);
	submenu->SetFont(be_plain_font);
	msg = new BMessage(_ATTR_MAIN_);
	msg->AddString("name", "last_modified");
	msg->AddInt32("type", B_TIME_TYPE);
	item = new BMenuItem(submenu, msg);
	menu->AddItem(item);

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_LE);
	submenu->AddItem(new BMenuItem("before", msg));

	msg = new BMessage(_ATTRIBUTE_ITEM_);
	msg->AddInt32("operator", B_GE);
	submenu->AddItem(new BMenuItem("after", msg));


	bounds = Bounds();
	bounds.right = bounds.left + 100;
	bounds.bottom = bounds.top + 15;
	menuField = new BMenuField(bounds, "MenuField", "", menu);
	menuField->SetDivider(0.0);

	// add text entry box
	bounds = Bounds();
	bounds.left += bounds.right - 180;
	bounds.top += 2;
	bounds.right -= 42;
	textControl = new BTextControl(bounds, "TextEntry", "", "", NULL);
	textControl->SetDivider(0.0);
}

void 
TAttrView::AttachedToWindow()
{
	AddChild(textControl);
	textControl->MakeFocus();

	AddChild(menuField);
	BMenu *menu = menuField->Menu();
	// add attributes from currently selected mimetype
	AddMimeTypeAttrs(menu);

	// target everything
	menu->SetTargetForItems(this);

	for (int32 index = menu->CountItems() - 1; index >= 0; index--)
		menu->SubmenuAt(index)->SetTargetForItems(this);
}


SpecView::~SpecView()
{
}

void
SpecView::Draw(BRect)
{
	BMenuItem *item = menuField->Menu()->FindMarked();
	if (!item)
		return;
	
	if (item->Submenu()->FindMarked()) {
		float width = StringWidth(item->Submenu()->FindMarked()->Label());
		BRect bounds(textControl->Frame());

		bounds.left -= (width + 10);
		bounds.bottom -= 6;
		MovePenTo(bounds.LeftBottom());
		DrawString(item->Submenu()->FindMarked()->Label());
	}
}

void
SpecView::MessageReceived(BMessage* msg)
{
	BMenuItem*	item;

	switch (msg->what) {
		case _ATTRIBUTE_ITEM_:
			if (msg->FindPointer("source", &item) != B_NO_ERROR)
				return;

			item->Menu()->Superitem()->SetMarked(true);
			Invalidate();
			break;
		
		// in case someone selected just and attribute without the
		// comparator
		case _ATTR_MAIN_:
			if (msg->FindPointer("source", &item) != B_NO_ERROR)
				return;

			item->Submenu()->ItemAt(0)->SetMarked(true);
			Invalidate();
			break;

		default:
			BView::MessageReceived(msg);
			break;
	}
}
