// =============================================================================
//     BTSNetMsgUtils.cpp
// ============================================================================= 
/*	
	Routines for sending and receiving messages via 
	a socket
*/

#include "BTSNetMsgUtils.h"
#include <support/Debug.h>
#include <errno.h>
#include <string.h>
#include <NetKit.h>

const char* SOURCE_SOCKET = "Source Socket";

// =============================================================================
//     SendNetMessage
// ============================================================================= 
long SendNetMessage(const BTSSocket& socket, BMessage* inMessage)
{
	// Converts a BMessage into a buffer and sends it over the specified socket.

	long 	numBytes = inMessage->FlattenedSize();
	char* 	buf = (char*)malloc(numBytes);
	long 	result = B_NO_ERROR;
	
	PRINT(("SendNetMessage - ENTER\n"));
	inMessage->Flatten(buf, numBytes);

	if (numBytes > 0 && buf != NULL)
	{ 
		result = SendNetMessageData(socket, numBytes, buf);
	}
	if (buf != NULL)	free(buf);
	
	PRINT(("SendNetMessage - EXIT\n"));
	return result;
}

// =============================================================================
//     SendNetMessageData
// ============================================================================= 
long SendNetMessageData(const BTSSocket& socket, const long numBytes, 
						const char* buf)
{
	// Sends the data for a BMessage over a socket, preceded by the message's
	// size.
	
	long 	result = B_NO_ERROR;
	long 	id	= MSG_IDENTIFIER;
	
	PRINT(( "SendNetMessageData - ENTER\n"));
	socket.SendLock();
	result = socket.Send((char*)&id, sizeof(long));
	if (result >= B_NO_ERROR)
	{
		result = socket.Send((char*)&numBytes, sizeof(long));
		if (result >= B_NO_ERROR)
		{
			result = socket.Send(buf, numBytes);
		}
	}
	socket.SendUnlock();
	PRINT( ("SendNetMessageData - EXIT\n"));
	return result;
}

// =============================================================================
//     ReceiveNetMessage
// ============================================================================= 
long ReceiveNetMessage(const BTSSocket& socket, BMessage& outMessage)
{
	// Reads data from a socket, expects to find a long value at the beginnning
	// of the stream that represents the message size. If the long value is 
	// valid, reads the buffer, safely unflattens it back into a BMessage, and
	// returns it to the caller.
	
	char*				buf;				// Message data buffer
	long				msgSize;			// Message size
	long 				result;				// Result of socket calls
	long				msgID = -1;
	
	PRINT(("ReceiveNetMessage - ENTER\n"));
	// Get the header identifying a message.
	socket.RecvLock();
	result = socket.Recv((char*)&msgID, sizeof(long));
	if (result == B_NO_ERROR && msgID == MSG_IDENTIFIER)
	{
		// Get the message size.
		result = socket.Recv((char*)&msgSize, sizeof(long));
		
		msgSize = ntohl(msgSize);	// Convert from network to native format.
	
		if (msgSize >= 0 && result == B_NO_ERROR)
		{
			buf = (char*)malloc(msgSize);	
			if (buf != NULL)
			{
				// Get the message data.
				result = socket.Recv(buf, msgSize);
				if (result == B_NO_ERROR) 
				{
					// Convert data back into a BMessage.
					if (outMessage.Unflatten(buf) == B_NO_ERROR)
					{
						// Add an identifier of where it came from.
						outMessage.AddPointer(SOURCE_SOCKET, (void*)&socket);
					}
					else result = B_ERROR;
				}
				free(buf);
			}
		}
		else if (msgSize > 0) result = B_ERROR;
	}
	socket.RecvUnlock();
	
	PRINT(("ReceiveNetMessage - EXIT\n"));
	return result;
}

// =============================================================================
//     SafeUnflatten
// ============================================================================= 
BMessage* SafeUnflatten(const char* buf)
{
	// Safely unflattens a buffer back into a BMessage. Basically, just
	// check for proper message data header ('PARC') before unflattening.
	
	BMessage* newMessage = new BMessage();
	
	PRINT(("SafeUnflatten - ENTER\n"));
	if (buf == NULL) return NULL;
	if (newMessage == NULL) return NULL;
	if (!(strcmp(buf, "PARC")))
	{		
		if (newMessage->Unflatten(buf) != B_NO_ERROR)
		{
			delete newMessage;
			newMessage = NULL;
		}
	}
	else if (newMessage != NULL)
	{
		delete newMessage;
		newMessage = NULL;
	}
	PRINT(("SafeUnflatten - EXIT\n"));
	return newMessage;
}

