//#define DEBUG 1
// =============================================================================
//     BTSNetMsgServer.cpp
// =============================================================================
/*	Defines a "message server" that communicates with clients by converting
	BMessages to and from network data. The server acts by taking messages that
	are posted to it and either repeating them to all connected clients
	or to a specified target client. The server also has an associated
	BLooper that is the message receiver. Any messages received from clients
	are posted to this message receiver.
*/
#include <stdio.h>
#include <string.h>
#include <net/netdb.h>
#include <app/Application.h>
#include <malloc.h>
#include <support/Debug.h>
#include <signal.h>

#include "BTSNetMsgServer.h"
#include "BTSNetMsgUtils.h"

// Server- related thread names
const char* kConnectionRequestHandlerName = "Server Request Handler";
const char* kClientListenerName = "Client Listener";

// Put this in a message to target the message at a specific socket.
// Otherwise messages go to all clients.
extern const char* TARGET_SOCKET = "Target Socket";

// =============================================================================
//     BTSNetMsgServer
// =============================================================================
BTSNetMsgServer::BTSNetMsgServer(const unsigned short port, 
						BLooper* messageReceiver,
						const long priority, 
						const int maxConnections, 
						const unsigned long address, 
						const int family, 
						const int type, 
						const int protocol  ) :
						fSocket(port, maxConnections, address,
								type,protocol, family),
						fPriority(priority),
						fMaxConnections(maxConnections),
						fNewClient(NULL)
{
	PRINT(("BTSNetMsgServer::BTSNetMsgServer - ENTER\n"));
	//_setDebugFlag(TRUE);
	if (messageReceiver == NULL)
	{
		messageReceiver = be_app;
	}
	fIsExiting = FALSE;
	fNewClientSem = ::create_sem(1, "Client Addition Sem");
	fSocketListSem = ::create_sem(1, "Client Socket List Sem");
	::acquire_sem(fSocketListSem);
	fMessageReceiver = messageReceiver;
	PRINT(("BTSNetMsgServer::BTSNetMsgServer - EXIT\n"));
	return;
}

// =============================================================================
//     ~BTSNetMsgServer
// =============================================================================
BTSNetMsgServer::~BTSNetMsgServer()
{
	fSocket.Close();
	fIsExiting = TRUE;
	// Close all client connections.
	while(!(fClientSocketList.IsEmpty()))
	{
		BTSSocket* socket = (BTSSocket*)fClientSocketList.ItemAt(0);
		RemoveClient(socket);
	}
	return;
}

// =============================================================================
//     Run
// =============================================================================
thread_id
BTSNetMsgServer::Run()
{
	thread_id 	theID = -1;		// To return thread id of this thread
	int 		result = 0;		// Results of socket function calls
	
	//Start the main server thread.
	theID = BLooper::Run();
	if (theID > B_NO_ERROR)		// Did server thread start?
	{
		// Start separate thread to handle conn. requests.
		fConnectionRequestHandlerID = 
		::spawn_thread(HandleConnectionRequests, 
					kConnectionRequestHandlerName, 
					fPriority, 
					(void*)this);
		if (fConnectionRequestHandlerID > B_NO_ERROR)
		{
			PRINT(("BTSNetMsgServer - Starting connect handler\n"));
			::resume_thread(fConnectionRequestHandlerID);
		}
	}
	else
	{
		// Error
	}
	PRINT(( "Message Server now running...\n"));
	return theID;
}


// =============================================================================
//     SendToClients
// =============================================================================
long 
BTSNetMsgServer::SendToClients(BMessage* message)
{
	BTSSocket* 	sourceSocket;
	int			sourceID = -1;
	long 		numBytes = message->FlattenedSize();
	char* 		buf = (char*)malloc(numBytes); 
	int			i;
	BTSSocket* 	socket;
	long 		result = B_NO_ERROR;
	
	message->Flatten(buf, numBytes);
	
	// Check to see if it came from a client, so we
	// don't send it back to the same client.
	if (message->FindPointer(SOURCE_SOCKET, &sourceSocket) == B_NO_ERROR)
	{
		sourceID = sourceSocket->ID();
	}
	if (buf != NULL && numBytes > 0)
	{
		// Make a copy of the client list in case it changes.
		::acquire_sem(fSocketListSem);
		BList socketList(fClientSocketList);
		::release_sem(fSocketListSem);
		
		// Loop and send data to all clients.
		for (i = 0; i < socketList.CountItems(); i++)
		{
			socket = (BTSSocket*)socketList.ItemAt(i);
			if (socket->ID() != sourceID)
			{
				PRINT(("BTSNetMsgServer - Send to client %ld\n", socket->ID()));
				result = SendNetMessageData(*socket, numBytes, buf);
			}
			// Remove client if send fails.
			if (result != B_NO_ERROR) 
			{
				socket->Close();
			}
		}
		free(buf);
	}
	return result;
}


// =============================================================================
//     MessageReceived
// =============================================================================
void 
BTSNetMsgServer::MessageReceived(BMessage* inMessage)
{
	// Messages that are not control messages are repeated either to the 
	// specified target client, or to all clients if no target is specified.
	
	PRINT(("BTSNetMsgServer::MessageReceived: ENTER\n"));
	switch (inMessage->what)
	{
		case DEAD_CONNECTION_MSG:
			// A connection has died or been closed. Delete it. This is 
			// a synchronous message, we don't want anyone to do anything
			// until we get rid of the dead socket. Ick.
			BTSSocket* socket;
			if (inMessage->FindPointer(SOURCE_SOCKET, &socket) == B_NO_ERROR)
			{	
				if (RemoveClient(socket))
				{
					BMessenger 	messenger(fMessageReceiver);
					BMessage		reply;
					BMessage		message(inMessage);
					PRINT(("Client removed.\n"));
					messenger.SendMessage(&message, &reply);
				}
			}
		break;
		
		case NEW_CLIENT_MSG:
			// Register the client's socket in our list of sockets.
			BTSSocket* newSocket;
			if (inMessage->FindPointer(SOURCE_SOCKET, &newSocket) == B_NO_ERROR)
			{
				AddClient(newSocket);
				fMessageReceiver->PostMessage(inMessage);
			}
		break;
		
		default:
			BTSSocket* msgSocket; 
			if (inMessage->FindPointer(TARGET_SOCKET, &msgSocket) == B_NO_ERROR)
			{
				SendNetMessage(*msgSocket, inMessage);
			}	
			else if (!fClientSocketList.IsEmpty())
			{
				long result = SendToClients(inMessage);
			}
		break;
	}
	
	PRINT(("BTSNetMsgServer::MessageReceived: EXIT\n"));
	return;
}

// =============================================================================
//     AddClient
// =============================================================================
void
BTSNetMsgServer::AddClient(BTSSocket* socket)
{
		
	// Start separate thread to listen for incoming client data.
	// Adds a client's listener thread id to the list. When the 
	// socket list is empty, the semaphore is alread acquired 
	// (see removeclient, constructor)
	::acquire_sem(fNewClientSem);
	fNewClient = socket;
	thread_id listenThreadID = ::spawn_thread(ListenToClient, 
								kClientListenerName,
								fPriority,
								(void*)this);
	if (listenThreadID > B_NO_ERROR)
	{
		PRINT(("BTSNetMsgServer - Starting client listener\n"));
		::resume_thread(listenThreadID);
		if (!(fClientSocketList.IsEmpty()))
		{
			::acquire_sem(fSocketListSem);
		}
		fClientSocketList.AddItem((void*)socket);
		::release_sem(fSocketListSem);	
	}
	return;
}


// =============================================================================
//     RemoveClient
// =============================================================================
bool
BTSNetMsgServer::RemoveClient(BTSSocket* socket)
{
	// Removes 
	bool	removed = FALSE;	// Reports whether remove is successful.
	
	PRINT(( "Removing client listening on socket %d\n", socket->ID()));
	if (!(fClientSocketList.IsEmpty()))
	{
		::acquire_sem(fSocketListSem);
		if (socket->InitCheck()) socket->Close();
		
		removed = fClientSocketList.RemoveItem((void*)socket);
		if (!(fClientSocketList.IsEmpty()))
		{
			::release_sem(fSocketListSem);
		}
	}
	return removed;
}

// =============================================================================
//     ListenToClient
// =============================================================================
long
BTSNetMsgServer::ListenToClient(void* arg)
{
	BTSNetMsgServer*	thisServer = (BTSNetMsgServer*)arg;
	BLooper*			messageReceiver = NULL;
	long				result = B_NO_ERROR;
	struct timeval 		tv;
	struct fd_set		readBits;
	struct fd_set		writeBits;
	BTSSocket* 			socket = NULL;
	sem_id				newClientSem = 0;

	if (thisServer != NULL)
	{
	 	messageReceiver = thisServer->MessageReceiver();
		socket = thisServer->NewClient();
	 	newClientSem = thisServer->NewClientSem();
	}
	
	if (newClientSem > 0 && socket != NULL && messageReceiver != NULL)
	{
		// Set delay to wait for client data before returning.
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		
		FD_ZERO(&writeBits);
		PRINT(("BTSNetMsgServer::ListenToClients - THREAD ENTER\n"));
		::release_sem(newClientSem);
		
		while (socket->InitCheck())
		{
			// Set the select bits for the known sockets
			FD_ZERO(&readBits);
			FD_SET(socket->ID(), &readBits);
			PRINT(("BTSNetMsgServer::ListenToClient - running on socket %d\n",
						socket->ID()));
			PRINT(("Calling select...\n"));
	
			// Blocks here until data arrives on any socket.
			if (::select(32, &readBits, &writeBits, NULL, &tv) > 0)
			{			
				BMessage newMessage;
				PRINT(("Server received data\n"));
	
				result = ReceiveNetMessage(*socket, newMessage);
					
				if ( result == B_NO_ERROR )
				{	
					// Post it to the message receiver.
					PRINT(("BTSNetMsgServer::ListenToClient, socket %d - posting message\n",
							socket->ID()));
					messageReceiver->PostMessage(&newMessage);
				}								
				else if (result == ECONNABORTED)
				{
					// Inform the server that a client went away.
					BMessage	deadMessage(DEAD_CONNECTION_MSG);
					BMessenger 	messenger(thisServer);
					BMessage 	reply;
					PRINT(("Connection aborted\n"));
					PRINT(("Closing client socket %d\n", socket));
					
					// Close the socket.
					result = socket->Close();
					PRINT(( "Client socket %d closed, result is %d\n", 
							socket, result));
					
					if (deadMessage.AddPointer(SOURCE_SOCKET, socket) == B_NO_ERROR)		
					{
						PRINT(("Sending abort message to server"));
						messenger.SendMessage(&deadMessage, &reply);
					}
				}
				result = B_NO_ERROR;
			}	
	
			// See if exit flag has been set.
			if (thisServer->IsExiting()) break;
		}
	}
	#if DEBUG
	else
	{
		PRINT(("BTSNetMsgServer::ListenToClients - bad data!\n"));
	}	
	#endif
	PRINT(("BTSNetMsgServer::ListenToClients - THREAD EXIT\n"));
	exit_thread(result);
	return result;
}

// =============================================================================
//     HandleConnectionRequests
// =============================================================================
long
BTSNetMsgServer::HandleConnectionRequests(void* arg)
{
	BTSNetMsgServer* 		thisServer = (BTSNetMsgServer*)arg;
	BTSServerSocket*		acceptSocket = thisServer->Socket();
	BTSSocket* 				newClient;
	sockaddr_in 			clientInterface;
	int						clientIntfSize = sizeof(sockaddr_in);
	long					result = 0;
	PRINT( ("HandleConnectionRequests - THREAD ENTER, accept socket is %d\n", 
				acceptSocket->ID()));
	// Thread blocks here, on accept().
	while ((newClient  = acceptSocket->Accept((struct sockaddr*)&clientInterface, 
							&clientIntfSize)) != NULL)
	{		
		PRINT(( "Connection request, new socket is %d\n", newClient->ID()));
		// A client has requested a connection. Make a handler for the
		// client socket.

		BMessage aMessage(NEW_CLIENT_MSG);
		if  (aMessage.AddPointer(SOURCE_SOCKET, (void*)newClient) == B_NO_ERROR)
		{
				thisServer->PostMessage(&aMessage);
		}
		else break;
		clientIntfSize = sizeof(sockaddr_in);
	}
	PRINT( ("HandleConnectionRequests - THREAD EXIT\n"));
	exit_thread(result);
	return result;
}

