//#define DEBUG 1
// =============================================================================
//     BTSSocket.cpp
// =============================================================================
#include "BTSSocket.h"
#include <stdio.h>
#include <support/Debug.h>
#include <errno.h>
#include <string.h>

// =============================================================================
//     BTSSocket
// =============================================================================
BTSSocket::BTSSocket() : fFamily(AF_INET), fType(SOCK_STREAM), 
						fProtocol(IPPROTO_TCP), fID(-1)
{
	//result = SetOption(SOL_SOCKET, SO_NONBLOCK, (char*)&ones, sizeof(int));
	
	Init();
	return;
}

// =============================================================================
//     BTSSocket
// =============================================================================

BTSSocket::BTSSocket(const int type, const int protocol, const int family) :
			fFamily(family), fID(-1)
{
	if (protocol == 0) fProtocol = IPPROTO_TCP;
	else
		fProtocol = protocol;

	if (type == 0) fType  = SOCK_STREAM; 
	else
		fType = type;
		
	//result = SetOption(SOL_SOCKET, SO_NONBLOCK, (char*)&ones, sizeof(int));
	Init();
	return;	
}

// =============================================================================
//     BTSSocket
// =============================================================================
BTSSocket::BTSSocket(const int socketID) : fFamily(-1), fType(-1), fProtocol(-1)
{
	fID = socketID;
	Init();
	return;
}

// =============================================================================
//     Init
// =============================================================================
void
BTSSocket::Init()
{
	char 	sendSemName[30];
	char	recvSemName[30];
	::sprintf(sendSemName, "Send Sem %d", fID);
	::sprintf(recvSemName, "Recv Sem %d", fID);
	fSendSem = ::create_sem(1, sendSemName);
	fRecvSem = ::create_sem(1, recvSemName);
	
	if (fID < 0)
	{
		fID = ::socket(fFamily,  fType, fProtocol);
	}
	
	fBound = FALSE;
	fListening = FALSE;
	fConnected = FALSE;
	return;
}

// =============================================================================
//     SetOption
// =============================================================================
long
BTSSocket::SetOption(const int level, const int option, char* data, 
					const unsigned int size) const
{
	long result = ::setsockopt(fID, level, option, data, size);	
	PRINT(("Result of socket option setting is %d\n", result));
	return result;
}

// =============================================================================
//     Open
// =============================================================================
long
BTSSocket::Open()
{
	if (fID > -1)
	{
		::closesocket(fID);
	}
	fID = ::socket(fFamily,  fType, fProtocol);
	return 0;
}

// =============================================================================
//     SendLock
// =============================================================================
long
BTSSocket::SendLock() const 
{
	return ::acquire_sem(fSendSem);
}

// =============================================================================
//     SendUnlock
// =============================================================================
long
BTSSocket::SendUnlock() const 
{
	return ::release_sem(fSendSem);
}

// =============================================================================
//     RecvLock
// =============================================================================
long
BTSSocket::RecvLock() const 
{
	return ::acquire_sem(fRecvSem);
}

// =============================================================================
//     RecvUnlock
// =============================================================================
long
BTSSocket::RecvUnlock() const 
{
	return ::release_sem(fRecvSem);
}

// =============================================================================
//     Interrupt
// =============================================================================
void
BTSSocket::Interrupt()
{
	if (fID >= 0)
	{
		struct sockaddr 	addr;
		int 				size;
 		int 				result = ::getsockname(fID, &addr, &size);
	}
	return;
}

// =============================================================================
//     ConnectToAddress
// =============================================================================
long
BTSSocket::ConnectToAddress(const BTSAddress& address)
{
	long 				result;
	const sockaddr_in* 	sockAddr = address.SockAddr();
	
	PRINT(("BTSSocket::ConnectToAddress - ENTER, id= %d port= %d, family= %d rem addr = %ud\n", 
	 	fID, sockAddr->sin_port, sockAddr->sin_family, 
	 	sockAddr->sin_addr.s_addr));
	result = ::connect(fID, (struct sockaddr*)sockAddr, 
						sizeof(sockAddr));
	PRINT(("BTSSocket::ConnectToAddress - EXIT\n"));
	fConnected =  (result == B_NO_ERROR);
	return result;
}

// =============================================================================
//     BindTo
// =============================================================================
long	
BTSSocket::BindTo(const BTSAddress& address)
{
	long	result;
	const 	sockaddr_in* sockAddr  = address.SockAddr();
	PRINT(("BTSSocket::BindTo - ENTER, id= %d port= %d, family= %d rem addr = %ud\n", 
	 	fID, sockAddr->sin_port, sockAddr->sin_family, 
	 	sockAddr->sin_addr.s_addr));
	
	result = ::bind(fID, (struct sockaddr*)sockAddr, 
						sizeof(sockAddr));
	PRINT(("BTSSocket::BindTo - EXIT, result is %s\n", strerror(result)));
	if (result == B_NO_ERROR) fBound = TRUE;
	return result;
}

// =============================================================================
//     Accept
// =============================================================================
BTSSocket*	
BTSSocket::Accept(struct sockaddr *addr, int *size) const
{
	BTSSocket* 	newClient = NULL;
	int			clientSock = -1;
	if (fID >= 0 && fBound && fListening)
	{
		clientSock = ::accept(fID, addr,size);
		if (clientSock >= 0)
		{
			newClient = new BTSSocket(clientSock);
		}
	}
	return newClient;
}

// =============================================================================
//     Listen
// =============================================================================
long	
BTSSocket::Listen(const int maxConnections)
{
	PRINT(("BTSSocket::Listen - ENTER\n"));
	fMaxConnections = maxConnections;
	long result = ::listen(fID, maxConnections);
	PRINT(("BTSSocket::Listen - EXIT, resut is %s\n", strerror (result)));
	if (result == B_NO_ERROR) fListening = TRUE;
	return result;
}

// =============================================================================
//     Close
// =============================================================================
long
BTSSocket::Close()
{
	PRINT(("BTSSocket::Close - ENTER, socket %d\n", fID));
	int id = fID;
	fID = -1;
	Interrupt();
	SendLock();
	RecvLock();	
	if (id > -1)
	{
 		::closesocket(fID);
 		fID = -1;
 	}
 	SendUnlock();
 	RecvUnlock();
	PRINT(("BTSSocket::Close - EXIT, socket %d\n", fID));
 	return 0;
}

// =============================================================================
//     Send
// =============================================================================
long
BTSSocket::Send(const char* buf, const size_t bufSize) const
{
	// Sends the data for a BMessage over a socket, preceded by the message's
	// size.
	
	long 	result = B_NO_ERROR;
	int 	numBytes = -1;
	int		sentBytes = 0;
	PRINT(( "SOCKET SEND - ENTER, %ld bytes on socket %d\n", bufSize, fID));
	if (bufSize > 0 && buf != NULL)
	{
		while (sentBytes < bufSize && (result == B_NO_ERROR || result == EINTR
				&& fID >= 0))
		{
			PRINT(("SOCKET SEND - Sending %ld/%ld bytes on socket %d..\n", 
					bufSize-sentBytes, bufSize, fID));
			errno = 0;
			numBytes = ::send(fID, buf+sentBytes, bufSize-sentBytes, 0);
			if (numBytes < 0) result = errno;
			if (numBytes > 0) sentBytes += numBytes;
		}
	}	
	#if DEBUG
	else
	{
		PRINT(("------------>BTSSocket::Send - Bad Info, socket %d\n", fID));
	}
	if (sentBytes > 0) UpdateSendCount(sentBytes);
	PRINT( ("SOCKET SEND - EXIT, sent %ld bytes on %d result is %s\n", sentBytes, 
				fID, strerror(result)));
	#endif
	return result;
}

// =============================================================================
//     Recv
// =============================================================================
long
BTSSocket::Recv(const char* buf, const size_t bufSize) const
{
	// Receives a network data buffer of a certain size. Does not return until
	// the buffer is full or if the socket returns 0 bytes (meaning it was 
	// closed) or returns an error besides EINTR. (EINTR can be generated when a
	// send() occurs on the same socket.
	
	long result = B_NO_ERROR;	// error value of socket calls
	int  receivedBytes = 0;		
	int  numBytes = 0;
	
	PRINT(("BTSSocket::Receive - ENTER, socket %d\n", fID));
	if (buf != NULL && bufSize > 0)
	{
		while (receivedBytes < bufSize && 
				(result == B_NO_ERROR || result == EINTR) && fID >= 0)
		{
			PRINT(("Receiving %ld/%ld bytes on %d\n", bufSize - receivedBytes, bufSize, 
					fID));
			errno = 0;
			numBytes = ::recv(fID, (char*)(buf+receivedBytes), 
								bufSize - receivedBytes, 0);
			
			if (numBytes == 0)
			{
				PRINT(("BTSSocket::Receive - got 0 bytes on %d at %ld/%ld bytes\n", 
							fID, receivedBytes, bufSize));
				result = ECONNABORTED;
				break;
			}
			else if (numBytes < 0) 
			{
				result = errno;
				PRINT(("BTSSocket::Receive - error when receiving data on socket %d - %s\n", 
							fID, strerror(result)));
			}
			else 
			{
				result = B_NO_ERROR;
				receivedBytes += numBytes;
				#if DEBUG
				UpdateReceiveCount(numBytes);
				#endif
			}
		}
		PRINT(("SOCKET %d RECEIVE - Received %ld bytes result is %s\n", fID, numBytes,
					strerror(result)));
	}
	if (result == EINTR && receivedBytes == bufSize) result = B_NO_ERROR;
	PRINT(("BTSSocket::Receive - EXIT, socket %d\n", fID));
	return result;
}

// =============================================================================
//     UpdateSendCount
// =============================================================================
void BTSSocket::UpdateSendCount(const long numBytes)
{
	static long sendCount = 0;
	sendCount += numBytes;
	PRINT(("Total bytes sent: %ld\n", sendCount));
	return;
}

// =============================================================================
//     UpdateReceiveCount
// =============================================================================
void BTSSocket::UpdateReceiveCount(const long numBytes)
{
	static long receiveCount = 0;
	receiveCount += numBytes;
	PRINT(("Total bytes received: %ld\n", receiveCount));
	return;
}
