#include <stdio.h>
#include <string.h>
#include <Bitmap.h>
#include <Application.h>
#include <Debug.h>
#include <MenuBar.h>
#include <Menu.h>
#include <MenuItem.h>

#include "BTSNetDrawWindow.h"
#include "BTSDrawView.h"
#include "BTSNetDrawClient.h"
#include "BTSNetDrawMsgDefs.h"


// =============================================================================
//     BTSNetDrawWindow
// =============================================================================
BTSNetDrawWindow::BTSNetDrawWindow(const BRect size, char* inName) :
	 BWindow(size, inName, B_TITLED_WINDOW, B_NOT_RESIZABLE)
{
	
	Lock();
	
	// Add menus.
	BRect		mbarRect = Bounds();
	mbarRect.bottom = mbarRect.top + 15;
	
	BMenuBar* 	mbar = new BMenuBar(mbarRect, "mbar");
	BMenu*		drawMenu = new BMenu("Draw");
	BMenuItem* 	aboutItem = new BMenuItem("About DrawClient...", 
						new BMessage(B_ABOUT_REQUESTED));

	aboutItem->SetTarget(be_app);
	drawMenu->AddItem(aboutItem);
	
	drawMenu->AddSeparatorItem();
	
	BMenuItem* clearItem = new BMenuItem("Clear", 
						new BMessage(CLEAR_MSG));
	clearItem->SetTarget(be_app);
	drawMenu->AddItem(clearItem);
	
	BMenuItem* quitItem = new BMenuItem("Quit", 
								new BMessage(B_QUIT_REQUESTED));
	quitItem->SetTarget(be_app);
	
	mbar->AddItem(drawMenu);
	AddChild(mbar);
	
	// Align all the children in the window.
	BRect drawSize(size);
	drawSize.OffsetTo(B_ORIGIN);
	drawSize.top += 15;
	drawSize.bottom += 15;
	
	// Create and add the children.
	fOffView = new BView(drawSize, "OffView", 0,
		B_NOT_RESIZABLE);
	fOffBitMap = new BBitmap(drawSize, B_MONOCHROME_1_BIT, TRUE);

	fDrawView = new BTSDrawView(drawSize, fOffBitMap);
	
	fOffBitMap->AddChild(fOffView);
	AddChild(fDrawView);
	Unlock();
	

	return;
}
	
	
// =============================================================================
//     QuitRequested
// =============================================================================
bool	
BTSNetDrawWindow::QuitRequested()
{
	// Tell the application to quit!
	be_app->PostMessage(B_QUIT_REQUESTED);
	return(TRUE);
}

// =============================================================================
//     MessageReceived
// =============================================================================
void 
BTSNetDrawWindow::MessageReceived(BMessage* inMessage)
{
	PRINT(("BTSNetDrawWindow::MessageReceived - ENTER\n"));
	switch (inMessage->what)
	{
		case NETDRAWLINE_MSG:
		
			PRINT(("NET DRAW\n"));
			// Draw the line in this client's window.
			BPoint start;
			if (inMessage->FindPoint("LineStart", &start) == B_NO_ERROR)
			{
				BPoint end;
				if (inMessage->FindPoint("LineEnd", &end) == B_NO_ERROR)
				{
					PRINT(("Drawing...\n"));
					DrawLine(start, end);
				}
			}
						
			PRINT(("NET DRAW END\n"));		
		break;
		
		case BITMAP_DATA_MSG:
			long bitmapSize;
			void* bitmapData;
			if (inMessage->FindData("Bitmap", B_RAW_TYPE, &bitmapData, &bitmapSize) == B_NO_ERROR)
			{
				SetBitmap(bitmapData, bitmapSize);
			}
		break;
		
		case QUIT_DRAW_WINDOW_MSG:	
			PRINT(("Window quitting...\n"));
			fDrawView->SetExiting(TRUE);
			BWindow::QuitRequested();
		break;
		
		case CLEAR_MSG:
			Clear();
		break;
		
		default:
			BWindow::MessageReceived(inMessage);
		break;				
	}
	PRINT(("BTSNetDrawWindow::MessageReceived - EXIT\n"));
}

// =============================================================================
//     DrawLine
// =============================================================================
void 
BTSNetDrawWindow::DrawLine(const BPoint& start, const BPoint& end)
{
	PRINT(("BTSNetDrawWindow::DrawLine - ENTER\n"));	
	BRect	boundsRect;
	SetRect(&boundsRect, start, end);
	fOffBitMap->Lock();
	fOffView->MovePenTo(start.x, start.y);
	fOffView->StrokeLine(end);	
	fOffView->Sync();
	fOffBitMap->Unlock();
	
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, boundsRect, boundsRect);
	fDrawView->Sync();
	Unlock();
	PRINT(("BTSNetDrawWindow::DrawLine - EXIT\n"));
	return;	
}
	
// =============================================================================
//     Clear
// =============================================================================
void 
BTSNetDrawWindow::Clear()
{
	void* buf = fOffBitMap->Bits();
	long numBytes = fOffBitMap->BitsLength();
	memset(buf, 0, numBytes);
	fOffBitMap->Lock();
	fOffBitMap->SetBits(buf, numBytes, 0, B_MONOCHROME_1_BIT);
	fOffView->Sync();
	fOffBitMap->Unlock();
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, Bounds(), Bounds());
	Unlock();
}

// =============================================================================
//     SetBitmap
// =============================================================================	
void BTSNetDrawWindow::SetBitmap(const void* bitmapData, const long bitmapSize)
{
	fOffBitMap->Lock();
	fOffBitMap->SetBits(bitmapData, bitmapSize, 0, B_MONOCHROME_1_BIT);
	fOffView->Sync();
	fOffBitMap->Unlock();
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, fDrawView->Bounds(), fDrawView->Bounds());
	Unlock();
}

// =============================================================================
//     WindowActivated
// =============================================================================	
void BTSNetDrawWindow::WindowActivated(bool active)
{
	if (active)
	{
		BPoint mousePoint;
		ulong 	buttons;
		fDrawView->GetMouse(&mousePoint, &buttons);
		fDrawView->SetLineStart(mousePoint);
	}
	BWindow::WindowActivated(active);
	return;
}

// =============================================================================
//     SetRect
// =============================================================================	
void
BTSNetDrawWindow::SetRect(BRect* boundsRect, const BPoint& lineStart, 
	const BPoint& lineEnd)
{
	float top, left, bot, right;
	if (lineStart.x < lineEnd.x)
	{
		left = lineStart.x;
		right = lineEnd.x;
	}
	else
	{
		left = lineEnd.x;
		right = lineStart.x;
	}
	if (lineStart.y < lineEnd.y)
	{
		top = lineStart.y;
		bot = lineEnd.y;
	}
	else
	{
		top = lineEnd.y;
		bot = lineStart.y;
	}
	boundsRect->Set(left, top, right, bot);
	return;
}