// =============================================================================
//     BTSSocket.cpp
// =============================================================================
#include "BTSSocket.h"
#include <stdio.h>
#include <support/Debug.h>
#include <errno.h>
#include <string.h>

// =============================================================================
//     BTSSocket
// =============================================================================
BTSSocket::BTSSocket() : fFamily(AF_INET), fType(SOCK_STREAM), 
						fProtocol(IPPROTO_TCP)
{
	int		ones = 0xFFFFFFFF;
	int		result;
	
	fID = ::socket(fFamily,  fType, fProtocol);
	result = SetOption(SOL_SOCKET, SO_NONBLOCK, (char*)&ones, sizeof(int));
	
	Init();
	return;
}

// =============================================================================
//     BTSSocket
// =============================================================================

BTSSocket::BTSSocket(const int type, const int protocol, const int family) :
			fFamily(family)
{
	int		ones = 0xFFFFFFFF;
	int		result;
	
	if (protocol == 0) fProtocol = IPPROTO_TCP;
	else
		fProtocol = protocol;

	if (type == 0) fType  = SOCK_STREAM; 
	else
		fType = type;
		
	fID = ::socket(fFamily,  fType, fProtocol);
	result = SetOption(SOL_SOCKET, SO_NONBLOCK, (char*)&ones, sizeof(int));
	Init();
	return;	
}

// =============================================================================
//     BTSSocket
// =============================================================================
BTSSocket::BTSSocket(const int socketID) : fFamily(-1), fType(-1), fProtocol(-1)
{
	fID = socketID;
	Init();
	return;
}

// =============================================================================
//     Init
// =============================================================================
void
BTSSocket::Init()
{
	char 	sendSemName[30];
	char	recvSemName[30];
	::sprintf(sendSemName, "Send Sem %d", fID);
	::sprintf(recvSemName, "Recv Sem %d", fID);
	fSendSem = create_sem(1, sendSemName);
	fRecvSem = create_sem(1, recvSemName);
	return;
}

// =============================================================================
//     SetOption
// =============================================================================
long
BTSSocket::SetOption(const int level, const int option, char* data, 
					const unsigned int size) const
{
	long result = ::setsockopt(fID, level, option, data, size);	
	PRINT(("Result of socket option setting is %d\n", result));
	return result;
}
// =============================================================================
//     Open
// =============================================================================
long
BTSSocket::Open()
{
	if (fID > -1)
	{
		::closesocket(fID);
	}
	fID = ::socket(fFamily,  fType, fProtocol);
	return 0;
}

// =============================================================================
//     SendLock
// =============================================================================
long
BTSSocket::SendLock() const 
{
	return ::acquire_sem(fSendSem);
}

// =============================================================================
//     SendUnlock
// =============================================================================
long
BTSSocket::SendUnlock() const 
{
	return ::release_sem(fSendSem);
}

// =============================================================================
//     RecvLock
// =============================================================================
long
BTSSocket::RecvLock() const 
{
	return ::acquire_sem(fRecvSem);
}

// =============================================================================
//     RecvUnlock
// =============================================================================
long
BTSSocket::RecvUnlock() const 
{
	return ::release_sem(fRecvSem);
}
// =============================================================================
//     ConnectToAddress
// =============================================================================
long
BTSSocket::ConnectToAddress(const BTSAddress& address)
{
	long 				result;
	const sockaddr_in* 	sockAddr = address.SockAddr();
	result = ::connect(fID, (struct sockaddr*)sockAddr, 
						sizeof(sockAddr));
	return result;
}

// =============================================================================
//     BindTo
// =============================================================================
long	
BTSSocket::BindTo(const BTSAddress& address) const
{
	long	result;
	const 	sockaddr_in* sockAddr  = address.SockAddr();
	
	result = ::bind(fID, (struct sockaddr*)sockAddr, 
						sizeof(sockAddr));
	return result;
}

// =============================================================================
//     Listen
// =============================================================================
long	
BTSSocket::Listen(const int maxConnections)
{
	fMaxConnections = maxConnections;
	return ::listen(fID, maxConnections);
}

// =============================================================================
//     Close
// =============================================================================
long
BTSSocket::Close()
{
	if (fID > -1)
	{
 		::closesocket(fID);
 		fID = -1;
 	}
 	return 0;
}

// =============================================================================
//     Send
// =============================================================================
long
BTSSocket::Send(const char* buf, const long bufSize) const
{
	// Sends the data for a BMessage over a socket, preceded by the message's
	// size.
	
	long 	result = B_NO_ERROR;
	int 	numBytes = -1;
	int		sentBytes = 0;
	PRINT(( "SOCKET SEND - ENTER, %ld bytes on socket %d\n", bufSize, fID));
	if (bufSize > 0 && buf != NULL)
	{
		while (sentBytes < bufSize && result == B_NO_ERROR || result == EINTR)
		{
			PRINT(("SOCKET SEND - Sending data..\n"));
			numBytes = ::send(fID, buf, bufSize, 0);
			if (numBytes < 0) result = errno;
			if (numBytes > 0) sentBytes += numBytes;
			if (sentBytes < numBytes) result = errno;
			#if DEBUG
			else UpdateSendCount(sentBytes);
			#endif
		}
	}	
	PRINT( ("SOCKET SEND - EXIT, sent %ld bytes on %d result is %s\n", sentBytes, 
				fID, strerror(result)));
	return result;
}

// =============================================================================
//     Recv
// =============================================================================
long
BTSSocket::Recv(const char* buf, const long bufSize) const
{
	// Receives a network data buffer of a certain size. Does not return until
	// the buffer is full or if the socket returns 0 bytes (meaning it was 
	// closed) or returns an error besides EINTR. (EINTR can be generated when a
	// send() occurs on the same socket.
	
	long result = B_NO_ERROR;	// error value of socket calls
	int  receivedBytes = 0;		
	int  numBytes = 0;
	
	PRINT(("SOCKET %d RECEIVE: ENTER \n", fID));

	while (receivedBytes < bufSize && (result == B_NO_ERROR || result == EINTR))
	{
		PRINT(("Receiving %ld bytes on %d\n", bufSize- receivedBytes, GetID()));
		numBytes = ::recv(fID, (char*)(buf+receivedBytes), 
							bufSize - receivedBytes, 0);
		
		if (numBytes == 0)
		{
			result = ECONNABORTED;
			break;
		}
		else if (numBytes < 0) 
		{
			PRINT(("error when receiving data!\n"));
			result = errno;
		}
		else 
		{
			receivedBytes += numBytes;
			#if DEBUG
			UpdateReceiveCount(numBytes);
			#endif
		}
	}
	PRINT(("SOCKET %d RECEIVE - Received %ld bytes result is %s\n", fID, numBytes,
				strerror(result)));

	if (result == EINTR && receivedBytes == bufSize) result = B_NO_ERROR;
	PRINT(("SOCKET %d RECEIVE: EXIT\n", fID));
	return result;
}

// =============================================================================
//     UpdateSendCount
// =============================================================================
void BTSSocket::UpdateSendCount(const long numBytes)
{
	static long sendCount = 0;
	sendCount += numBytes;
	PRINT(("Total bytes sent: %ld\n", sendCount));
	return;
}

// =============================================================================
//     UpdateReceiveCount
// =============================================================================
void BTSSocket::UpdateReceiveCount(const long numBytes)
{
	static long receiveCount = 0;
	receiveCount += numBytes;
	PRINT(("Total bytes received: %ld\n", receiveCount));
	return;
}
