/*
	screen

	This program simply exercises some of the YAGS drawing
	primitives.
	
	It gets a hold of the screen buffer, and starts doing
	some drawing.
*/

#include <Application.h>
#include <stdio.h>
#include <stdlib.h>
#include "screen.h"
#include "yagstst.h"
#include "effects.h"

AYAGSScreen *gYAGSScreen;

// Index of the accelerated graphics hooks used by this demo. All of them are need,
// or the demo will not work. Currently, it works only on S3 based cards. 
#define INDEX_LINE            3
#define INDEX_RECT            5
#define INDEX_SYNCHRO         10

// Type definition for the graphics hooks used by this demo. 
typedef long (*LINE)(long,long,long,long,uchar,bool,short,short,short,short);
typedef long (*RECT)(long,long,long,long,uchar);
typedef long (*SYNCHRO)(void);

//=====================================================================
//	Class: ADemoWindow
//=====================================================================

AYAGSScreen::AYAGSScreen()
	: BWindowScreen("YAGS", B_8_BIT_640x480)
{
	gYAGSScreen = this;
}

void 
AYAGSScreen::Quit()
{
	Disconnect();
	
	// Quit the WindowScreen.
	BWindowScreen::Quit();
}

void 
AYAGSScreen::ScreenConnected(bool active)
{
	if (active == TRUE)
	{
		printf("ADemoWindow::ScreenConnected - active\n");
		
		width = CardInfo()->width;
		height = CardInfo()->height;
		pixels = CardInfo()->frame_buffer;
		bytes_per_row = CardInfo()->bytes_per_row;
		
		fScreen.mode = B_COLOR_8_BIT;
		fScreen.pixels = (unsigned char *)pixels;
		fScreen.bytes_per_row = bytes_per_row;
		fScreen.width = width;
		fScreen.height = height;
		
		rgb_color    palette[256];
		palette[0].red = 0;
		palette[0].green = 0;
		palette[0].blue = 0;
		for (int i=1;i<256;i++) 
		{
			palette[i].red = rand();
			palette[i].green = rand();
			palette[i].blue = rand();
		}
		palette[255].red = 255;
		palette[255].green = 255;
		palette[255].blue = 255;
		SetColorList(palette);
	} else
		printf("ADemoWindow::ScreenConnected - inactive\n");
}

void	
AYAGSScreen::MessageReceived(BMessage *message)	
{
	//printf("ADemoWindow::MessageReceived - %d\n", message->what);
	
	switch (message->what) 
	{
		case B_KEY_DOWN:
		{
			long key = message->FindLong("key");
			long aChar = message->FindLong("char");
			switch(aChar)
			{
				case B_ESCAPE:
					be_app->PostMessage(B_QUIT_REQUESTED);
				break;

			}
		}
	
		default:
			// Just forward all messages to the be_app
			be_app->PostMessage(message);	
		break;
	}
	
}




void	
AYAGSScreen::HLine(const U16 x1, const U16 y1, const U16 x2, const U16 y2, const U08 aColor)
{
	LINE      draw_line;
	
	draw_line = (LINE)CardHookAt(INDEX_LINE);
	(draw_line)(x1, x2, y1, y2, aColor, 0, 0, 0, 0, 0);

	//Line(x1, y1, x2, y2, aColor);
}

void	
AYAGSScreen::VLine(const U16 x1, const U16 y1, const U16 x2, const U16 y2, const U08 aColor)
{
	LINE      draw_line;
	
	draw_line = (LINE)CardHookAt(INDEX_LINE);
	(draw_line)(x1, x2, y1, y2, aColor, 0, 0, 0, 0, 0);

	//Line(x1, y1, x2, y2, aColor);
}

void	
AYAGSScreen::Line(const U16 x1, const U16 y1, const U16 x2, const U16 y2, const U08 aColor)
{
	LINE      draw_line;
	
	draw_line = (LINE)CardHookAt(INDEX_LINE);
	(draw_line)(x1, x2, y1, y2, aColor, 0, 0, 0, 0, 0);
}

void	
AYAGSScreen::Lines(const Point2Ds *pts, const unsigned long npts, const U08 aColor)
{
	for (int counter =0; counter < npts; counter+=2)
	{
		Line(pts[counter].x, pts[counter].y, 
			pts[counter+1].x,pts[counter+1].y,
			aColor);
	}	
}

void	
AYAGSScreen::LineStrip(const Point2Ds *pts, const unsigned long npts, const U08 aColor)
{
	for (long counter =0; counter < npts-1; counter++)
	{
		Line(pts[counter].x, pts[counter].y, 
			pts[counter+1].x,pts[counter+1].y,
			aColor);
	}	
}


void	
AYAGSScreen::Triangle(
			const U16 x1, const U16 y1,
			const U16 x2, const U16 y2,
			const U16 x3, const U16 y3,
			const U08 aColor)
{
	Line(x1,y1,x2,y2, aColor); 
	Line(x2,y2,x3,y3, aColor); 
	Line(x3,y3,x1,y1, aColor); 
}

void
AYAGSScreen::TopTriangleFill(
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor)
{
	float dx_right;
	float dx_left;
	float xs;
	float xe;
	
	float X1=x1, X2=x2, X3=x3;
	float Y1=y1, Y2=y2, Y3=y3;
	
	float height;
	int temp_x;
	int temp_y;
	
	if (X2 < X1)
	{
		temp_x = X2;
		X2 = X1;
		X1 = temp_x;
	}
	
	// compute deltas
	height = Y3 - Y1+1;
	
	dx_left = (X3-X1)/height;
	dx_right = (X3-X2)/height;
	
	// set starting points
	xs = (float)X1;
	xe = (float)X2+(float)0.5;
	
#if 0
	// We won't bother with clipping now
	// But here's the code that should go somewhere
	// perform clipping
#endif

	// Start drawing the lines
	for (temp_y = Y1; temp_y <= Y3; temp_y++)
	{
		HLine((U16)xs, (U16)temp_y, (U16)xe, (U16)temp_y, aColor);
		
		xs += dx_left;
		xe += dx_right;
	}
}

void
AYAGSScreen::BottomTriangleFill(
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor)
{
	float dx_right;
	float dx_left;
	float xs;
	float xe;
	
	float X1=x1, X2=x2, X3=x3;
	float Y1=y1, Y2=y2, Y3=y3;
	
	float height;
	int temp_x;
	int temp_y;
	int deltaX;
	int deltaY;
	
	if (X3 < X2)
	{
		temp_x = X2;
		X2 = X3;
		X3 = temp_x;
	}
	
	// compute deltas
	height = Y3 - Y1+1;
	
	dx_left = (X2-X1)/height;
	dx_right = (X3-X1)/height;
	
	// set starting points
	xs = (float)X1;
	xe = (float)X1+(float)0.5;
	
#if 0
	// We won't bother with clipping now
	// But here's the code that should go somewhere
	// perform clipping
#endif

	// Start drawing the lines
	for (temp_y = Y1; temp_y <= Y3; temp_y++)
	{
		deltaX = xe - xs;
		if (deltaX > 0)
			HLine((U16)xs, (U16)temp_y, (U16)xe, (U16)temp_y, aColor);
		
		xs += dx_left;
		xe += dx_right;
	}
}

void	
AYAGSScreen::TriangleFill(
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor)
{
	Y2DTriangleFill(fScreen,x1, y1,x2, y2,x3, y3,aColor);
	return;
	
	int temp_x, temp_y;
	int new_x;
	
	U16 X1=x1, X2=x2, X3=x3;
	U16 Y1=y1, Y2=y2, Y3=y3;
	
	// test for h lines and v lines
	if ((X1==X2) && (X2==X3))
	{
		// Vertical line
		VLine(X1,Y1,X1,Y3,aColor);
		return;
	}
	
	if  ((Y1==Y2) && (Y2==Y3))
   	{
		// Horizontal Line
		HLine(X1,Y1,X3,Y1,aColor);
		return;
	}
	
	// sort p1,p2,p3 in ascending y order
	if (Y2<Y1)
	{
		temp_x = X2;
		temp_y = Y2;
		X2     = X1;
		Y2     = Y1;
		X1     = temp_x;
		Y1     = temp_y;
	} // end if

	// now we know that p1 and p2 are in order
	if (Y3<Y1)
	{
		temp_x = X3;
		temp_y = Y3;
		X3     = X1;
		Y3     = Y1;
		X1     = temp_x;
		Y1     = temp_y;
	} // end if

	// finally test y3 against y2
	if (Y3<Y2)
	{
		temp_x = X3;
		temp_y = Y3;
		X3     = X2;
		Y3     = Y2;
		X2     = temp_x;
		Y2     = temp_y;
	} // end if

	// do trivial rejection tests

	//if ( Y3<scrInfo.top || Y1>scrInfo.frame.bottom ||
    //	(X1<scrInfo.frame.left && X2<scrInfo.frame.left && X3<scrInfo.frame.left) ||
    //	(X1>scrInfo.frame.right && X2>scrInfo.frame.right && X3>scrInfo.frame.right) )
	//	return;
	// test if top of triangle is flat
	if (Y1==Y2)
	{
		TopTriangleFill(X1,Y1, X2,Y2, X3,Y3, aColor);
	} else
	if (Y2==Y3)
	{
		BottomTriangleFill(X1,Y1, X2,Y2, X3,Y3, aColor);
	} else
	{
		// general triangle that needs to be broken up along long edge
		new_x = X1 + (int)((float)(Y2-Y1)*(float)(X3-X1)/(float)(Y3-Y1));

		// draw each sub-triangle
		BottomTriangleFill(X1,Y1,new_x,Y2,X2,Y2,aColor);
		TopTriangleFill(X2,Y2,new_x,Y2,X3,Y3,aColor);
   } // end else
}


void	
AYAGSScreen::Rect(const U16 left, const U16 top,
			const U16 right, const U16 bottom,
			const U08 aColor)
{
 	// Use the optimized line drawing routines to draw the sides.	
	// Do the top and bottom lines
	HLine(left,top,right,top, aColor); 
	HLine(left,bottom,right,bottom, aColor); 
		
	// Do the left and right lines
	VLine(left,top,left,bottom, aColor); 
	VLine(right,top,right,bottom, aColor); 
}

void	
AYAGSScreen::RectFill(const U16 left, const U16 top,
			const U16 right, const U16 bottom,
			const U08 aColor)
{
	RECT      fill_rect;
	long width = right - left + 1;
	long height = bottom - top + 1;
	
	//printf("left: %d top: %d right: %d bottom: %d\n",left,top,right,bottom);
	//printf(" width: %d height: %d\n",width, height);
	
	fill_rect = (RECT)CardHookAt(INDEX_RECT);
	(fill_rect)((long)left, (long)top, (long)right, (long)bottom,aColor);
}

void	
AYAGSScreen::Clear(const U08 aColor)
{
	RectFill(0, 0,fScreen.width-1, fScreen.height-1, aColor);
}



//==============================================================
//	Method: Ellipse
//==============================================================

void	
AYAGSScreen::Ellipse(const U16 x, const U16 y,
			const U16 radiusX, const U16 radiusY, 
			const U08 aColor)
{
	Y2DEllipse(fScreen,x, y,radiusX, radiusY, aColor);
}

/* 
	Fills an ellipse of X axis radius A and Y axis radius B in
	color Color centered at screen coordinate (X,Y). Radii must
	both be non-zero 
*/
void	
AYAGSScreen::EllipseFill(const U16 X, const U16 Y,
			const U16 A, const U16 B, 
			const U08 aColor)
{
	short WorkingX, WorkingY;
	long Threshold;
	long ASquared = (long) A * A;
	long BSquared = (long) B * B;
	long XAdjust, YAdjust;


   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which X is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (0,B) */
   /* Calculate all points along an arc of 1/8th of the ellipse and
      draw the line segment */
	WorkingX = 0;
	XAdjust = 0;
	YAdjust = ASquared * 2 * B;
	Threshold = ASquared / 4 - ASquared * B;

	short YOffset = -B;
	
	for (;;) 
	{
      /* Advance the threshold to the value for the next X point
         to be drawn */
		Threshold += XAdjust + BSquared;

		/* 
			If the threshold has passed 0, then the Y coordinate has
			advanced more than halfway to the next pixel and it's time
			to advance the Y coordinate by 1 and set the next threshold
			accordingly and draw the current line.
		*/
		if ( Threshold >= 0 ) 
		{
			// Draw the current lines before moving on
			Y2DHLine(fScreen,X-WorkingX,Y+YOffset, X+WorkingX,Y+YOffset,aColor);
			Y2DHLine(fScreen,X-WorkingX,Y-YOffset, X+WorkingX,Y-YOffset,aColor);
			YOffset ++;
			YAdjust -= ASquared * 2;
			Threshold -= YAdjust;
		}
		// Advance the X coordinate by 1
		XAdjust += BSquared * 2;
		WorkingX++;

		/* Stop if X is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( XAdjust >= YAdjust )
			break;
	}

	
   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which Y is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (A,0) */
   /* 
		Calculate all points along an arc of 1/8th of the ellipse
		Draw as we calculate.
	*/
	XAdjust = BSquared * 2 * A;
	YAdjust = 0;
	Threshold = BSquared / 4 - BSquared * A;
	short XOffset = 0;
	YOffset = 0;

	for (;;) 
	{
      /* Advance the threshold to the value for the next Y point
         to be drawn */
		Threshold += YAdjust + ASquared;

      /* If the threshold has passed 0, then the X coordinate has
         advanced more than halfway to the next pixel and it's time
         to advance the X coordinate by 1 and set the next threhold
         accordingly */
		if ( Threshold >= 0 ) 
		{
			XAdjust -= BSquared * 2;
			Threshold = Threshold - XAdjust;
			// advance along both axes
			XOffset++;			
		}

		// Draw the current lines
		// upper half
		Y2DHLine(fScreen,X-A+XOffset, Y-YOffset, X+A-XOffset, Y-YOffset,aColor);
		// Lower half
		Y2DHLine(fScreen,X-A+XOffset, Y+YOffset, X+A-XOffset, Y+YOffset,aColor);
		
		/* Advance the Y coordinate by 1 */
		YAdjust += ASquared * 2;
		YOffset++;
		
		/* Stop if Y is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( YAdjust > XAdjust )
			break;
	}
}


#define ASPECT_RATIO 1

void	
AYAGSScreen::Line( const Point3Dhf pt1, const Point3Dhf pt2, 
			const U08 aColor, const Camera aCamera)
{
	float width = fScreen.width;
	float height = fScreen.height;
	
	float x1,y1,z1,     // working variables
		x2,y2,z2;
	int ix1,iy1,        // integers used to hold screen coordinates
		ix2,iy2;

	// convert to screen ccordinates
	x1=(width/2  + pt1.x*aCamera.viewing_distance/pt1.z);
	y1=(height/2 - ASPECT_RATIO*pt1.y*aCamera.viewing_distance/pt1.z);

	x2=(width/2  + pt2.x*aCamera.viewing_distance/pt2.z);
	y2=(height/2 - ASPECT_RATIO*pt2.y*aCamera.viewing_distance/pt2.z);

	// convert floats to integers for line clipper
	ix1=(long)x1;
	iy1=(long)y1;
	ix2=(long)x2;
	iy2=(long)y2;

	// draw clipped lines
	if (Y2DClipLine(0,0,width,height, &ix1,&iy1,&ix2,&iy2))
	{
		Line((U16)ix1,(U16)iy1,(U16)ix2,(U16)iy2,aColor);
	} // end if clip
}

void	
AYAGSScreen::RenderWireObject(const FacetedObject3D *aObject,
			const U08 aColor, const Camera aCamera)
{
	// For each facet in the object
	for (int faceCtr=0; faceCtr < aObject->fNumFaces; faceCtr++)
	{
		// for each line in the facet
		for (int lineCtr=0; lineCtr < aObject->fFaces[faceCtr].fNumVertices-1; lineCtr++)
		{
			// draw the line
			Line(
				aObject->fFaces[faceCtr].fVertexList[lineCtr],
				aObject->fFaces[faceCtr].fVertexList[lineCtr+1],
				aObject->fFaces[faceCtr].fColor,
				aCamera);
		}
		
		// draw the closing line
		Line(
			aObject->fFaces[faceCtr].fVertexList[aObject->fFaces[faceCtr].fNumVertices-1],
			aObject->fFaces[faceCtr].fVertexList[0],
			aObject->fFaces[faceCtr].fColor,
			aCamera);
	}
}
