#include "yags2d.h"
#include "yags3d.h"
#include <string.h>


// Clip the line to the bounds rectangle
int Y2DClipLine(int left, int top, int right, int bottom, 
	int *x1,int *y1,int *x2, int *y2)
{
int point_1 = 0, point_2 = 0;  // tracks if each end point is visible or invisible
int clip_always = 0;           // used for clipping override
int xi,yi;                     // point of intersection

int right_edge=0,              // which edges are the endpoints beyond
    left_edge=0,
    top_edge=0,
    bottom_edge=0;


int success = 0;               // was there a successfull clipping

float dx,dy;                   // used to holds slope deltas

// test if line is completely visible

if ( (*x1>=left) && (*x1<=right) &&
     (*y1>=top) && (*y1<=bottom) )
     point_1 = 1;


if ( (*x2>=left) && (*x2<=right) &&
     (*y2>=top) && (*y2<=bottom) )
     point_2 = 1;

	// test endpoints
	if (point_1==1 && point_2==1)
		return(1);

	// test if line is completely invisible
	if (point_1==0 && point_2==0)
	{
		// must test to see if each endpoint is on the same side of one of
		// the bounding planes created by each clipping region boundary
		if ( ((*x1<left) && (*x2<left)) || // to the left
        ((*x1>right) && (*x2>right)) || // to the right
        ((*y1<top) && (*y2<top)) || // above
        ((*y1>bottom) && (*y2>bottom)) )  // below
        {
        	// No need to draw line
			return(0);
		} // end if invisible

		// if we got here we have the special case where the line cuts into and
		// out of the clipping region
		clip_always = 1;
	} // end if test for invisibly

	// take care of case where either endpoint is in clipping region
	if (( point_1==1) || (point_1==0 && point_2==0) )
	{
		// compute deltas
		dx = *x2 - *x1;
		dy = *y2 - *y1;
		// compute what boundary line need to be clipped against
		if (*x2 > right)
		{
			// flag right edge
			right_edge = 1;
			// compute intersection with right edge
			if (dx!=0)
				yi = (int)(.5 + (dy/dx) * (right - *x1) + *y1);
			else
				yi = -1;  // invalidate intersection
		} // end if to right
		else  if (*x2 < left)
		{
			// flag left edge
			left_edge = 1;
			// compute intersection with left edge
			if (dx!=0)
				yi = (int)(.5 + (dy/dx) * (left - *x1) + *y1);
			else
				yi = -1;  // invalidate intersection
		} // end if to left
		// horizontal intersections
		if (*y2 > bottom)
		{
			// flag bottom edge
			bottom_edge = 1;
			// compute intersection with right edge
			if (dy!=0)
				xi = (int)(.5 + (dx/dy) * (bottom - *y1) + *x1);
			else
				xi = -1;  // invalidate inntersection
		} // end if bottom
		else
		if (*y2 < top)
		{
			// flag top edge
			top_edge = 1;
			// compute intersection with top edge
			if (dy!=0)
				xi = (int)(.5 + (dx/dy) * (top - *y1) + *x1);
			else
				xi = -1;  // invalidate intersection
		} // end if top

		// now we know where the line passed thru
		// compute which edge is the proper intersection
		if (right_edge==1 && (yi>=top && yi<=bottom) )
		{
			*x2 = right;
			*y2 = yi;
			success = 1;
		} // end if intersected right edge
		else
		if (left_edge==1 && (yi>=top && yi<=bottom) )
		{
			*x2 = left;
			*y2 = yi;
			success = 1;
		} // end if intersected left edge

		if (bottom_edge==1 && (xi>=left && xi<=right) )
		{
			*x2 = xi;
			*y2 = bottom;
			success = 1;
		} // end if intersected bottom edge
		else
		if (top_edge==1 && (xi>=left && xi<=right) )
		{
			*x2 = xi;
			*y2 = top;
			success = 1;
		} // end if intersected top edge
	} // end if point_1 is visible

	// reset edge flags
	right_edge = left_edge= top_edge = bottom_edge = 0;

	// test second endpoint
	if ( (point_2==1) || (point_1==0 && point_2==0))
	{
		// compute deltas
		dx = *x1 - *x2;
		dy = *y1 - *y2;
		// compute what boundary line need to be clipped against
		if (*x1 > right)
      {
      // flag right edge
      right_edge = 1;
      // compute intersection with right edge
      if (dx!=0)
         yi = (int)(.5 + (dy/dx) * (right - *x2) + *y2);
      else
         yi = -1;  // invalidate inntersection
      } // end if to right
   else
   if (*x1 < left)
      {
      // flag left edge
      left_edge = 1;
      // compute intersection with left edge
      if (dx!=0)
         yi = (int)(.5 + (dy/dx) * (left - *x2) + *y2);
      else
         yi = -1;  // invalidate intersection
      } // end if to left

   // horizontal intersections
   if (*y1 > bottom)
      {
      // flag bottom edge
      bottom_edge = 1;
      // compute intersection with right edge
      if (dy!=0)
         xi = (int)(.5 + (dx/dy) * (bottom - *y2) + *x2);
      else
         xi = -1;  // invalidate inntersection
		} // end if bottom
		else
		if (*y1 < top)
		{
			// flag top edge
			top_edge = 1;
			// compute intersection with top edge
			if (dy!=0)
				xi = (int)(.5 + (dx/dy) * (top - *y2) + *x2);
			else
				xi = -1;  // invalidate inntersection
		} // end if top

		// now we know where the line passed thru
		// compute which edge is the proper intersection
		if (right_edge==1 && (yi>=top && yi<=bottom) )
		{
			*x1 = right;
			*y1 = yi;
			success = 1;
		} // end if intersected right edge
		else
		if (left_edge==1 && (yi>=top && yi<=bottom) )
		{
			*x1 = left;
			*y1 = yi;
			success = 1;
		} // end if intersected left edge

		if (bottom_edge==1 && (xi>=left && xi<=right) )
		{
			*x1 = xi;
			*y1 = bottom;
			success = 1;
		} // end if intersected bottom edge
		else
		if (top_edge==1 && (xi>=left && xi<=right) )
		{
			*x1 = xi;
			*y1 = top;
			success = 1;
		} // end if intersected top edge
	} // end if point_2 is visible
	return(success);
}

void 
Y2DLineBres(const U16 x1, const U16 y1,const U16 x2, const U16 y2, 
	const U08 aColor,pixel_buffer screenInfo)
{
	U08 *base = &((U08 *)screenInfo.pixels)[y1 * screenInfo.bytes_per_row + x1];
	register int dx, dy, xf, yf, a, b, c, i;
	U16 X1 = x1;
	U16 Y1 = y1;
	U16 X2 = x2;
	U16 Y2 = y2;
	
	if (x2>x1) 
	{
		dx = x2-x1;
		xf = 1;
	} else 
	{
		dx = x1-x2;
		xf = -1;
	}

	if (y2>y1) 
	{
		dy = y2-y1;
		yf = 1;
	} else 
	{
		dy = y1-y2;
		yf = -1;
	}


	if (dx>dy) 
	{
		a = dy+dy;
		c = a-dx;
		b = c-dx;
		for (i=0;i<=dx;i++) 
		{
			*base = aColor;
			base += xf;
			if (c<0) 
			{
				c += a;
			} else 
			{
				c += b;
				base += screenInfo.bytes_per_row * yf;
			}
		}
	} else 
	{
		a = dx+dx;
		c = a-dy;
		b = c-dy;
		for (i=0;i<=dy;i++) 
		{
			*base = aColor;
			base += screenInfo.bytes_per_row * yf;
			if (c<0) 
			{
				c += a;
			} else 
			{
				c += b;
				base += xf;
			}
		}
	}

}


void
Y2DPixelSpan(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U08 *pixels, const U16 spanLength)
{
	
	U08 *base = &((U08 *)screenInfo.pixels)[y1 * screenInfo.bytes_per_row + x1];
	
	
	for (U32 counter = 0; counter < spanLength; counter++)
	{
		base[counter] = pixels[counter];
	}
}

void
Y2DHLine(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U16 x2, const U16 y2, const U08 aColor)
{
	
	U08 *base = &((U08 *)screenInfo.pixels)[y1 * screenInfo.bytes_per_row + x1];
	size_t count = x2-x1+1;
	
	//printf("Y2DHLine - base == 0x%x\n", base);
	//printf("Y2DHLine - rowbytes == %d\n", screenInfo.bytes_per_row);
	//printf("Y2DHLine - count == 0x%x\n", count);
	
	// For really short lines just do straight byte
	// assignment.
	if (count < 8) 
	{
		while(count-- > 0)
			*base++ = aColor;
		return;
	}
	

	// For longer lines, try to move 4 bytes at a time
	// first line up with a 4 byte boundary.
	//printf("Base: %d\n",(long)base);
	while (((long)base & 0x03) && (count > 0)) 
	{
		*base = aColor;
		//printf("Inc Base: 0x%x\n",(long)base);
		base++;
		count--;
	}
	
	
	
	U32	c;
	U08	*cPtr = (U08*)&c;
	cPtr[0]=aColor;cPtr[1]=aColor;cPtr[2]=aColor;cPtr[3]=aColor;
	
	// Do the bulk of the copying 4 bytes at a time
	while(count > 3) 
	{
		*(U32 *)base = c;
		base += 4;
		count -= 4;
	}

	//printf("Y2DHLine - count remaining: %d\n", count);
	//return;

	// Clean up any remaining bytes
	while(count-- > 0) 
		*base++ = aColor;
}


//=================================================================
//
// Draws a horizontal run of pixels, then advances the bitmap pointer to
//   the first pixel of the next run. 
//=================================================================
inline void DrawHorizontalRun(const U16 bytes_per_row, U08 **ScreenPtr, 
	int XAdvance,
U16 RunLength, U08 Color)
{
   int i;
   U08 *WorkingScreenPtr = *ScreenPtr;

   for (i=0; i<RunLength; i++)
   {
      *WorkingScreenPtr = Color;
      WorkingScreenPtr += XAdvance;
   }
	// Advance to the next scan line
   	WorkingScreenPtr += bytes_per_row;
	*ScreenPtr = WorkingScreenPtr;
}

//=================================================================
//
// Draws a vertical run of pixels, then advances the bitmap pointer to
//   the first pixel of the next run.
//=================================================================
inline void DrawVerticalRun(const U16 bytes_per_row, U08 **ScreenPtr, 
	int XAdvance,
U16 RunLength, U08 Color)
{
   int i;
   U08 *WorkingScreenPtr = *ScreenPtr;

   for (i=0; i<RunLength; i++)
   {
      *WorkingScreenPtr = Color;
      WorkingScreenPtr += bytes_per_row;
   }

   // Advance to the next column
   WorkingScreenPtr += XAdvance;
   *ScreenPtr = WorkingScreenPtr;
}



//=================================================================
// Function: Y2DLine
//
// Uses a modified Bresenham integer line drawing algorithm.
//
// Input:  x1,y1 - coordinates of first endpoint
//         x2,y2 - coordinates of second endpoint
// Output:  The line is drawn
// Return:  none
//=================================================================
//=================================================================
//
// Draws a line between the specified endpoints in color Color.
//=================================================================
void 
Y2DLine(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U16 x2, const U16 y2, const unsigned char Color)
{
	int Temp, AdjUp, AdjDown, ErrorTerm, XAdvance, XDelta, YDelta;
	int WholeStep, InitialPixelCount, FinalPixelCount, i, RunLength;
	U16 XStart= x1, YStart=y1, XEnd=x2, YEnd=y2;   
	U08 *ScreenPtr;

	// We'll always draw top to bottom, to reduce the number of cases we have to
	// handle, and to make lines between the same endpoints draw the same pixels 
	if (y1 > y2) 
	{
		YStart = y2;
		YEnd = y1;
		XStart = x2;
		XEnd = x1;
	}

	// Point to the bitmap address first pixel to draw */
	ScreenPtr = &((U08*)screenInfo.pixels)[YStart * screenInfo.bytes_per_row + XStart];

	// Figure out whether we're going left or right, and how far we're
	// going horizontally
	XDelta = XEnd - XStart;
	if (XDelta < 0)
	{
		XAdvance = -1;
		XDelta = -XDelta;
	} else
	{
		XAdvance = 1;
	}

	// Figure out how far we're going vertically
	YDelta = YEnd - YStart;

	// Special case for vertical line
	if (XDelta == 0)
	{
      /* Vertical line */
      for (i=0; i<=YDelta; i++)
      {
         *ScreenPtr = Color;
         ScreenPtr += screenInfo.bytes_per_row;
      }
      return;
	}

	// Special case for horizontal line
	if (YDelta == 0)
	{
		Y2DHLine(screenInfo,Y2DMIN(x1,x2),y1,Y2DMAX(x2,x1),y2, Color);
		//memset(ScreenPtr,Color, XDelta);	
		return;
	}
	
	// Special case for diagonal lines
	if (XDelta == YDelta)
	{
		for (i=0; i<=XDelta; i++)
		{
			*ScreenPtr = Color;
			ScreenPtr += XAdvance + screenInfo.bytes_per_row;
		}
		return;
	}

	// Now for all the non-special lines   
	/* Determine whether the line is X or Y major, and handle accordingly */
	if (XDelta >= YDelta)
	{
		/* X major line */
		/* Minimum # of pixels in a run in this line */
		WholeStep = XDelta / YDelta;

		/* Error term adjust each time Y steps by 1; used to tell when one
         extra pixel should be drawn as part of a run, to account for
         fractional steps along the X axis per 1-pixel steps along Y */
		AdjUp = (XDelta % YDelta) * 2;

		/* Error term adjust when the error term turns over, used to factor
         out the X step made at that time */
		AdjDown = YDelta * 2;

		/* Initial error term; reflects an initial step of 0.5 along the Y
         axis */
		ErrorTerm = (XDelta % YDelta) - (YDelta * 2);

		/* The initial and last runs are partial, because Y advances only 0.5
         for these runs, rather than 1. Divide one full run, plus the
         initial pixel, between the initial and last runs */
		InitialPixelCount = (WholeStep / 2) + 1;
		FinalPixelCount = InitialPixelCount;

      /* If the basic run length is even and there's no fractional
         advance, we have one pixel that could go to either the initial
         or last partial run, which we'll arbitrarily allocate to the
         last run */
		if ((AdjUp == 0) && ((WholeStep & 0x01) == 0))
		{
			InitialPixelCount--;
		}
      /* If there're an odd number of pixels per run, we have 1 pixel that can't
         be allocated to either the initial or last partial run, so we'll add 0.5
         to error term so this pixel will be handled by the normal full-run loop */
         if ((WholeStep & 0x01) != 0)
      {
         ErrorTerm += YDelta;
      }
		/* Draw the first, partial run of pixels */
		DrawHorizontalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, InitialPixelCount, Color);
		
		/* Draw all full runs */
      for (i=0; i<(YDelta-1); i++)
      {
         RunLength = WholeStep;  /* run is at least this long */
         /* Advance the error term and add an extra pixel if the error
            term so indicates */
         if ((ErrorTerm += AdjUp) > 0)
         {
            RunLength++;
            ErrorTerm -= AdjDown;   /* reset the error term */
         }
         /* Draw this scan line's run */
         DrawHorizontalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, RunLength, Color);
      }
      /* Draw the final run of pixels */
      DrawHorizontalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, FinalPixelCount, Color);
      return;
   }
   else
   {
      /* Y major line */

      /* Minimum # of pixels in a run in this line */
      WholeStep = YDelta / XDelta;

      /* Error term adjust each time X steps by 1; used to tell when 1 extra
         pixel should be drawn as part of a run, to account for
         fractional steps along the Y axis per 1-pixel steps along X */
      AdjUp = (YDelta % XDelta) * 2;

      /* Error term adjust when the error term turns over, used to factor
         out the Y step made at that time */
      AdjDown = XDelta * 2;

      /* Initial error term; reflects initial step of 0.5 along the X axis */
      ErrorTerm = (YDelta % XDelta) - (XDelta * 2);

      /* The initial and last runs are partial, because X advances only 0.5
         for these runs, rather than 1. Divide one full run, plus the
         initial pixel, between the initial and last runs */
      InitialPixelCount = (WholeStep / 2) + 1;
      FinalPixelCount = InitialPixelCount;

      /* If the basic run length is even and there's no fractional advance, we
         have 1 pixel that could go to either the initial or last partial run,
         which we'll arbitrarily allocate to the last run */
      if ((AdjUp == 0) && ((WholeStep & 0x01) == 0))
      {
         InitialPixelCount--;
      }
      /* If there are an odd number of pixels per run, we have one pixel
         that can't be allocated to either the initial or last partial
         run, so we'll add 0.5 to the error term so this pixel will be
         handled by the normal full-run loop */
      if ((WholeStep & 0x01) != 0)
      {
         ErrorTerm += XDelta;
      }
      /* Draw the first, partial run of pixels */
      DrawVerticalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, InitialPixelCount, Color);

      /* Draw all full runs */
      for (i=0; i<(XDelta-1); i++)
      {
         RunLength = WholeStep;  /* run is at least this long */
         /* Advance the error term and add an extra pixel if the error
            term so indicates */
         if ((ErrorTerm += AdjUp) > 0)
         {
            RunLength++;
            ErrorTerm -= AdjDown;   /* reset the error term */
         }
         /* Draw this scan line's run */
         DrawVerticalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, RunLength, Color);
      }
      /* Draw the final run of pixels */
      DrawVerticalRun(screenInfo.bytes_per_row, &ScreenPtr, XAdvance, FinalPixelCount, Color);
		return;
	}
}

