/* ++++++++++
	zero.c
	A driver that zeroes things.
++++++++++ */

#include <SupportDefs.h>
#include <KernelExport.h>
#include <Errors.h>
#include <Drivers.h>
#include <string.h>
#include <fcntl.h>

/* -----
	foward declarations for hook functions
----- */

static status_t		zero_open(const char *name, uint32 flags, void **cookie);
static status_t		zero_close(void *cookie);
static status_t		zero_free(void *cookie);
static status_t		zero_control(void *cookie, uint32 op, void *data, size_t len);
static status_t		zero_read(void *cookie, off_t pos, void *data, size_t *len);
static status_t		zero_write(void *cookie, off_t pos, const void *data, size_t *len);

/* -----
	device_hooks structure - has function pointers to the
	various entry points for device operations
----- */

static device_hooks my_device_hooks = {
	&zero_open,
	&zero_close,
	&zero_free,
	&zero_control,
	&zero_read,
	&zero_write
};

/* -----
	list of device names to be returned by publish_devices()
----- */

static char		*device_name_list[] = {
	"my_zero",
	0
};


/* ----------
	publish_devices - return list of device names implemented by
	this driver.
----- */

const char	**
publish_devices(void)
{
	return device_name_list;
}

/* ----------
	find_device - return device hooks for a specific device name
----- */

device_hooks *
find_device (const char *name)
{
	if (!strcmp (name, device_name_list[0]))
		return &my_device_hooks;
	
	return NULL;
}


/* ----------
	zero_open - hook function for the open call.
----- */

static status_t	
zero_open(const char *name, uint32 flags, void **cookie)
{
	if ((flags & O_RWMASK) != O_RDONLY)
		return B_NOT_ALLOWED;
	return B_OK;
}

/* ----------
	zero_close - hook function for the close call.
----- */

static status_t
zero_close(void *cookie)
{
	return B_OK;
}


/* ----------
	zero_free - hook function to free the cookie returned
	by the open hook.  SInce the open hook did not return
	a cookie, this is a no-op.
----- */

static status_t	
zero_free(void *cookie)
{
	return B_OK;
}


/* ----------
	zero-control - hook function for the ioctl call.  No
	control calls are implemented for this device.
----- */

static status_t	
zero_control(void *cookie, uint32 op, void *data, size_t len)
{
	return EINVAL;
}


/* ----------
	zero_read - hook function for the read call.  Zeros the
	passed buffer.  Note that the position parameter, pos, is
	ignored - there is no 'position' for this device.
	
	We use the memset() function exported from the kernel
	to zero the buffer.
----- */

static status_t
zero_read(void *cookie, off_t pos, void *data, size_t *len)
{
	memset (data, 0, *len);
	return B_OK;
}


/* ----------
	zero_write - hook function for the write call.  This will
	never be called, because we catch it in the open hook.
----- */
 
static status_t	
zero_write(void *cookie, off_t pos, const void *data, size_t *len)
{
	return B_READ_ONLY_DEVICE;
}
