/* ++++++++++
	pci_dev.c
	A device driver for an imaginary PCI device.
	
	This device is a single PCI device, but it actually
	supports four separate logical devices on the same
	controller.  This is similar to an IDE controller
	being able to support both the master and slave IDE drive.
	
	Each logical device is able to initiate PCI bus master
	transactions to both read and write main memory.  When
	a client calls read, the driver programs the logical
	device to write directly into the client's buffer.
	When the client calls write(), the driver sets up to
	read directly from the client buffer.  What the device
	does with the data is left to your imagination.
	
	The PCI device has one base register, which points to
	the physical location of the memory-mapped control
	registers for running the four logical devices.
	
	The control registers are stupidly designed, however.
	I/O is accomplished by writing three values in sequence
	to a single register for each logical device:
	
		1st write:	0 to read from memory, non-zero to write
		2nd write:	memory address
		3rd write:	# bytes

	There is another control register for each logical device
	that sets the speed of operation.
	
	The PCI device also uses a couple of registers in its
	configuration space for self-test status and device
	initialization.
	
+++++ */

#include <SupportDefs.h>
#include <PCI.h>
#include <Drivers.h>
#include <Errors.h>
#include <KernelExport.h>
#include <OS.h>
#include "pci_dev.h"

#define DEBUG_NO_HW 1	/* 1 to debug without real hardware! */

#if 1					/* 1 for serial debug output */
#define ddprintf dprintf
#else
#define dprintf
#endif

/* -----
	private types
----- */

/* offsets in PCI configuration space to various registers */
enum {
	FAKE_CONFIG_STATUS = 0x80,
	FAKE_CONFIG_RESET = 0x84
};
	
typedef enum {						/* device opcodes */
	FAKE_DEV_READ = 0,
	FAKE_DEV_WRITE = 1
} dev_op;

/* layout of each logical device's control registers */

typedef struct {
	volatile int32		cmd;		/* write commands here */
	volatile int32		speed;		/* current speed */
} dev_regs;

/*	to keep scatter/gather table small, we limit ourselves
	to a fixed maximum size per i/o.  If a bigger one is
	requested, we just do more than transaction. */
	
#define MAX_IO			0x8000		/* biggest single i/o we do */
#define MAX_SCATTER		((MAX_IO/B_PAGE_SIZE) + 1)

/* the 'cookie' used to keep track of each logical device */

typedef struct {
	pci_info		pci;		/* pci info for this device */
	dev_regs		*regs;		/* -> control regs */
	int				id;			/* logical device id */
	sem_id			hw_lock;	/* lock for atomic h/w access */
	sem_id			io_done;	/* i/o complete semaphore */
	size_t			xfer_count;	/* # bytes done so far */
	dev_op			op;			/* read or write opcode */
	int				scat_index;	/* index to scatter entry */
	physical_entry	scatter [MAX_SCATTER];
								/* scatter/gather table */
#if DEBUG_NO_HW
	bool			DEBUG_INT_SEM;	/* for fake interrupt handler */
#endif
} dev_info;


/* -----
	foward declarations for hook functions, interrupt handler
----- */

static status_t		pci_open(const char *name, uint32 flags, void **cookie);
static status_t		pci_close(void *cookie);
static status_t		pci_free(void *cookie);
static status_t		pci_control(void *cookie, uint32 op, void *data, size_t len);
static status_t		pci_read(void *cookie, off_t pos, void *data, size_t *len);
static status_t		pci_write(void *cookie, off_t pos, const void *data, size_t *len);

static bool			fake_dev_inth(void *data);

/* -----
	device_hooks structure - has function pointers to the
	various entry points for device operations
----- */

static device_hooks my_device_hooks = {
	&pci_open,
	&pci_close,
	&pci_free,
	&pci_control,
	&pci_read,
	&pci_write
};

/* -----
	list of device names to be returned by publish_devices()
----- */

static char		*name_list[] = {
	"fake/device1",
	"fake/device2",
	"fake/device3",
	"fake/device4",
	0
};

/* -----
	globals for this driver.  The values are NOT preserved
	when the driver is unloaded and later reloaded.  The
	init_driver() call should set these up every time...
---- */

static area_id	reg_area;	/* area to map device registers */
static char		*reg_base;	/* base address of reg access area */
static int32	is_open;	/* bitfield of open logical devices */


/* ----------
	lookup_pci_device - locates the pci_info record for a particular
	pci device
----- */

static bool
lookup_pci_device (short vendor, short device, pci_info *returned_info)
{
	int	i;
	for (i = 0; ; i++) {
		if (get_nth_pci_info (i, returned_info) != B_OK)
			return false;
		if (returned_info->vendor_id == vendor &&
		    returned_info->device_id == device)
		    	break;
	}
	return true;
}

/* ----------
	init_hardware - check if the hardware is there, and if so
	put the hardware into a stable state.
----- */

status_t
init_hardware (void)
{
	pci_info	info;
	int32		val;
	
	if (!lookup_pci_device (FAKE_DEV_VENDOR_ID, 
	                        FAKE_DEV_DEVICE_ID, &info))
		return ENODEV;
	
	/* ---
		this imaginary device has a self-test status register
		which will return a non-zero value if the self-test
		fails.
		
		It also requires writing another register to reset
		the device to a known state.
	----- */
#if DEBUG_NO_HW
	return TRUE;
#endif
	val = read_pci_config (info.bus, info.device, info.function, 
		FAKE_CONFIG_STATUS,		/* offset in config space */
		4						/* size of register */
	);
	
	if (val) 		/* self-test failure? */
		return ENODEV;
	
	write_pci_config (info.bus, info.device, info.function, 
		FAKE_CONFIG_RESET,	/* offset in config space */
		4,					/* size of register */
		0					/* initialization value */
	);
	
	return B_OK;
}


/* ----------
	init_driver - allocate the various kernel resources we
	will use in the driver implementation.
----- */
status_t
init_driver (void)
{
	pci_info	info;
	int32		base;
	int32		size;
	
	if (!lookup_pci_device (FAKE_DEV_VENDOR_ID, 
	                        FAKE_DEV_DEVICE_ID, &info))
		return ENODEV;
	
	/* ---
		Our imaginary device has a single base register, in
		PCI memory space, providing memory-mapped access to
		a bunch of device control registers.  We need to map it.

		We play it safe, and assume that the BIOS did not
		neccessarily relocate the base register to a page 
		boundary.  We map enough physical address space to
		span the entire range spcified by the base register.
	--- */	
		
	base = info.u.h0.base_registers[0];
	size = info.u.h0.base_register_sizes[0];
	
	/* round base addr down to page boundary */
	base = base & ~(B_PAGE_SIZE - 1);
	/* fix size, given base addr rounding */
	size += info.u.h0.base_registers[0] - base;
	/* round size up to next multiple of page size */
	size = (size + (B_PAGE_SIZE - 1)) & ~(B_PAGE_SIZE - 1);
	
    reg_area = map_physical_memory (
       "fake_pci_device_regs",
       (void *) base,
       size,
       B_ANY_KERNEL_ADDRESS,
       B_READ_AREA + B_WRITE_AREA,
       &reg_base
    );
    if (reg_area < 0)
        return reg_area;

	is_open = 0;		/* no devices open yet */
	return B_OK;
}


/* ----------
	uninit_driver - free up the various kernel resources we
	used in the driver implementation.
----- */
void
uninit_driver (void)
{
	delete_area (reg_area);
}



/* ----------
	publish_devices - return list of device names implemented by
	this driver.
----- */

const char	**
publish_devices(void)
{
	return name_list;
}

/* ----------
	lookup_device_name - looks up device name, returns id
----- */
static int
lookup_device_name (const char *name)
{
	int i;
	
	for (i = 0; name_list[i]; i++)
		if (!strcmp (name_list[i], name))
			return i;
	return -1;
}

/* ----------
	find_device - return device hooks for a specific device name
----- */

device_hooks *
find_device (const char *name)
{
	if (lookup_device_name (name) >= 0)
		return &my_device_hooks;
				
	return NULL;
}


/* ----------
	pci_open - hook function for the open call.  We pass
	a pointer to the pci_info record for our device back to
	the kernel.  This will be the 'cookie' which we will
	use to get at the device in all the other device operations.
----- */

static status_t	
pci_open(const char *name, uint32 flags, void **cookie)
{
	dev_info	*d;
	int			id;
	int32		mask;
	status_t	err;
	char		sem_name [B_OS_NAME_LENGTH];
	
	/* figure out which sub-device, by looking at the name */
	id = lookup_device_name (name);
	if (id < 0)
		return EINVAL;
		
	/* only one client at a time */
	mask = 1 << id;
	if (atomic_or (&is_open, mask) & mask)
		return B_BUSY;
	
	/* allocate a per-logical-device record in the kernel
	   heap.  This will be our 'cookie'. Note that the kernel 
	   heap is always locked, so this will always be resident */
	  
	d = (dev_info *) malloc (sizeof (dev_info));
	if (!d)
		return B_NO_MEMORY;
	
	/* find the device */
	err = ENODEV;
	if (!lookup_pci_device (FAKE_DEV_VENDOR_ID,
	                        FAKE_DEV_DEVICE_ID, &d->pci))
		goto err1;
	
	/* setup device record */
	d->id = id;
	d->regs = (dev_regs *) reg_base + id;
	
	/* allocate a semaphore to enforce exclusive access to
	   the logical device's hardware.  Note the initial count
	   of one, to allow the first client to get it. */
	   
	sprintf (sem_name, "fake dev %d hw_lock", id);
	d->hw_lock = err = create_sem (1, sem_name);
	if (err < 0)
		goto err1;
	set_sem_owner(err, B_SYSTEM_TEAM);
	
	/*	allocate a semaphore to wait for the i/o to complete.
		The initial value of zero mean the client will
		block.  The interrupt handler will release the
		sempahore when the i/o is done. */
		
	sprintf (sem_name, "fake dev %d io_done", id);
	d->io_done = err = create_sem (0, sem_name);
	if (err < 0)
		goto err2;
	set_sem_owner(err, B_SYSTEM_TEAM);
	
	*cookie = d;
	
	/* install interrupt handler */
	set_io_interrupt_handler (
		d->pci.u.h0.interrupt_line,		/* interrupt number */
		fake_dev_inth,					/* -> interrupt handler */
		d								/* pass this to handler */
	);
	enable_io_interrupt (d->pci.u.h0.interrupt_line);
	
	return B_OK;

err2:
	delete_sem (d->hw_lock);
err1:
	free (d);
	return err;
}


/* ----------
	pci_close - hook function for the close call.
----- */

static status_t
pci_close(void *cookie)
{
	return B_OK;
}


/* ----------
	pci_free - hook function to free the cookie returned
	by the open hook.  Called after close, and after all
	read/write calls have returned.
	
	It sounds strange that a read/write call could still be
	pending after the device is closed, but it is certainly
	possible in a multi-threaded world.
	
	We could make the close call un-block any pending
	read/writes, but not today.  This example is complicated
	enough as is!
----- */

static status_t	
pci_free(void *cookie)
{
	dev_info *d = (dev_info *) cookie;
	
	disable_io_interrupt (d->pci.u.h0.interrupt_line);
	set_io_interrupt_handler (
		d->pci.u.h0.interrupt_line,	
		NULL,
		NULL
	);
	
	atomic_and (&is_open, ~(1 << d->id));
	delete_sem (d->hw_lock);
	delete_sem (d->io_done);
	free (d);
	
	return B_OK;
}

/* ----------
	pci-control - hook function for the ioctl call.  No
	control calls are implemented for this device.
----- */

static status_t	
pci_control(void *cookie, uint32 op, void *data, size_t len)
{
	dev_info *d = (dev_info *) cookie;
	
	switch (op) {
	
	case FAKE_DEV_RESET:			/* reset the device */
		write_pci_config (d->pci.bus, d->pci.device, 
			d->pci.function, 
			FAKE_CONFIG_RESET,	/* offset in config space */
			4,					/* size of register */
			0					/* initialization value */
		);
		break;
		
	case FAKE_DEV_SET_SPEED:		/* set device speed */
		d->regs->speed = *(int32 *)data;
		break;
		
	case FAKE_DEV_GET_SPEED:		/* return current speed */
		*(int32 *)data = d->regs->speed;
		break;
	
	default:
		return EINVAL;
	}
	
	return B_OK;
}

/* ----------
	start_io - start up a logical device processing a
	single contiguous chunk of the client buffer.
	
	Called by the read/write routines AND by the interrupt
	handler.
	
	Exclusive access to the hardware is guaranteed.
----- */
static bool
start_io (dev_info *d)
{
	int i;
	
	/* first see if we are already done w/client buffer */
	
	ddprintf ("start_io: index = %.8x\n", d->scat_index);
	i = d->scat_index;
	if (i == MAX_SCATTER || d->scatter[i].size == 0)
		return FALSE;
	
	d->scat_index++;
	
	/* write the opcode, the address, and the size, in order.
	   The ram_address() call gets the address of the
	   main system memory location as viewed from the PCI bus. */
	   
#if DEBUG_NO_HW
	return TRUE;
#endif
	d->regs->cmd = d->op;
	__eieio();
	d->regs->cmd = (int32) ram_address (d->scatter[i].address);
	__eieio();
	d->regs->cmd = d->scatter[i].size;
	__eieio();
}

#if DEBUG_NO_HW
/* ----------
	fake_interrupter - since we have no real hardware, we
	must fake calling the interrupt handler.  This routine
	does so every couple of seconds.
----- */

static int32
fake_interrupter (void *data)
{
	dev_info *d = (dev_info *)data;
	for (;;) {
		if (acquire_sem_etc (d->DEBUG_INT_SEM, 1, 
			B_CAN_INTERRUPT | B_TIMEOUT, 2000000L) == B_OK)
				break;
		fake_dev_inth (d);
	}
	delete_sem (d->DEBUG_INT_SEM);
}
#endif

/* ----------
	do_device_io - common code to do the trick work of
	starting up some i/o on a logical device.
----- */

static status_t
do_device_io (dev_info *d, dev_op op, void *data, size_t *len)
{
	status_t	err;
	size_t		xfer_count = 0;
	int32		flags;
	
	ddprintf ("do_device_io: data=%.8x *len = %.8x\n", data, *len);

	if (!*len)			/* nothing to do? */
		return B_OK;
	
	/*	we must tell the kernel that we are going to DMA
		to the client buffer.  There are computers out there
		that do not maintain cache coherency for dma operations
		(shameful!), so the kernel must mark the affected
		memory as non-cacheable for the duration of the lock.
		
		We must also tell the kernel if we intend to dma into
		memory (B_READ_DEVICE), so it can mark the affected
		memory as 'dirty' */
		
	flags = B_DMA_IO;
	if (op == FAKE_DEV_READ)
		flags |= B_READ_DEVICE;
	
	/* ensure client buffer is not paged out */
	err = lock_memory (data, *len, flags);
	if (err != B_OK)
		goto err0;
	
	/* get exclusive access to h/w.  If there are multiple
	   requests, all but the first will block here. */
	   
	err = acquire_sem_etc (d->hw_lock, 1, B_CAN_INTERRUPT, 0);
	if (err != B_OK)
		goto err1;
	
	d->xfer_count = 0;		/* no i/o done yet */

	/* get physical pages comprising client buffer */
	err = get_memory_map (data, *len, d->scatter, MAX_SCATTER);
	if (err != B_OK)
		goto err2;

#if DEBUG_NO_HW
	{
		int j;
		for (j = 0; j < MAX_SCATTER; j++)
			ddprintf ("do_device_io: scatter[j] = %.8x %.8x\n", d->scatter[j].address, d->scatter[j].size);
	}
#endif

	/* start up the i/o on the first scatter/gather entry. */
	
	d->scat_index = 0;
	d->op = op;
	start_io (d);
	
#if DEBUG_NO_HW
	/* start up the fake interrupt provoker thread */
	d->DEBUG_INT_SEM = create_sem (0, "fake int sem");
	resume_thread (spawn_kernel_thread (
		fake_interrupter, 
		"fake_interrupter",
		B_NORMAL_PRIORITY,
		d
	));
#endif

	/* wait for the i/o to complete.  The interrupt handler 
	   will start the i/o on the rest of the scatter entries. */
	   
	err = acquire_sem_etc (d->io_done, 1, B_CAN_INTERRUPT, 0);
	if (err != B_OK)
		goto err2;
	
#if DEBUG_NO_HW
	/* stop the fake interrupter thread */
	release_sem (d->DEBUG_INT_SEM);
#endif
	xfer_count = d->xfer_count;
	err = B_OK;
	
err2:
	release_sem (d->hw_lock);
err1:
	unlock_memory (data, *len, flags);
err0:
	*len = xfer_count;
	
	return err;
}

/* ----------
	pci_read - hook function for the read call.
----- */

static status_t
pci_read(void *cookie, off_t pos, void *data, size_t *len)
{
	return do_device_io ((dev_info *)cookie, FAKE_DEV_READ, 
		data, len);
}


/* ----------
	pci_write - hook function for the write call.  Writing
	to this device does not make sense.
----- */
 
static status_t	
pci_write(void *cookie, off_t pos, const void *data, size_t *len)
{
	return do_device_io ((dev_info *)cookie, FAKE_DEV_WRITE,
		(void *)data, len);
}

/* ----------
	fake_dev_inth - the interrupt handler.
----- */
static bool
fake_dev_inth(void *data)
{
	dev_info	*d = (dev_info *)data;
	
	/* start up i/o on the next chunk of scattered physical
	   memory.  If we are all done, unblock the thread that
	   initialed the i/o.
	   
	   In general, interrupt handlers should always release
	   semaphores with the do-not-reschedule flag.  This is
	   a must if anything significant is done after the
	   release call.  If the flag is not set, the release
	   call could reschedule the current thread that is
	   processing the interrupt.  When it wakes up, potentially
	   much later, other driver operations may have
	   been done by other threads.  This could lead to
	   unexpected behavior if not carefully thought out!
	   
	   The easiest thing is to use the do-not-reschedule
	   flag.  The release call will always return right
	   away. */
	   
	if (!start_io (d))
		release_sem_etc(d->io_done, 1, B_DO_NOT_RESCHEDULE);
	
	return TRUE;
}

	
