// =============================================================================
//     BTSNetMsgServer.cpp
// =============================================================================
/*	Defines a "message server" that communicates with clients by converting
	BMessages to and from network data. The server acts by taking messages that
	are posted to it and either repeating them to all connected clients
	or to a specified target client. The server also has an associated
	BLooper that is the message receiver. Any messages received from clients
	are posted to this message receiver.
*/
#include <stdio.h>
#include <string.h>
#include <net/netdb.h>
#include <app/Application.h>
#include <malloc.h>
#include <support/Debug.h>

#include "BTSNetMsgServer.h"
#include "BTSNetMsgUtils.h"

// Server- related thread names
const char* kConnectionRequestHandlerName = "Server Request Handler";
const char* kClientListenerName = "Client Listener";

// Put this in a message to target the message at a specific socket.
// Otherwise messages go to all clients.
const char* TARGET_SOCKET = "Target Socket";

// =============================================================================
//     BTSNetMsgServer
// =============================================================================
BTSNetMsgServer::BTSNetMsgServer(const unsigned short port, 
						BLooper* messageReceiver,
						const long priority, 
						const int maxConnections, 
						const unsigned long address, 
						const int family, 
						const int type, 
						const int protocol  ) :
						fSocket(type,protocol, family),
						fAddress(family,port, address),
						fPriority(priority),
						fMaxConnections(maxConnections)
{
	
	if (messageReceiver == NULL)
	{
		messageReceiver = be_app;
	}
	fIsExiting = FALSE;
	fSocketListSem = ::create_sem(1, "Client Socket List Sem");
	
	::acquire_sem(fSocketListSem);
	fMessageReceiver = messageReceiver;
	return;
}

// =============================================================================
//     ~BTSNetMsgServer
// =============================================================================
BTSNetMsgServer::~BTSNetMsgServer()
{
	// Close all client connections.
	if (!(fClientSocketList.IsEmpty()))
	{
		::acquire_sem(fSocketListSem);
	}
	while(!(fClientSocketList.IsEmpty()))
	{
		fClientSocketList.RemoveItem((long)0);
	}
	::release_sem(fSocketListSem);
	return;
}

// =============================================================================
//     QuitRequested
// =============================================================================
bool
BTSNetMsgServer::QuitRequested()
{
	fSocket.Close();
}


// =============================================================================
//     Run
// =============================================================================
thread_id
BTSNetMsgServer::Run()
{
	thread_id 	theID = -1;		// To return thread id of this thread
	int 		result = 0;		// Results of socket function calls
	
	
	// Bind the socket to the port/address specified in sockAddr
	result = fSocket.BindTo(fAddress);
	
	if (result >= 0)							// Was bind successful?
	{
		// Start listening for connections.
		result = fSocket.Listen(fMaxConnections);
		if (result >= 0)
		{
			//Start the main server thread.
			theID = BLooper::Run();
			if (theID > B_NO_ERROR)				// Did server thread start?
			{
				// Start separate thread to handle conn. requests.
				fConnectionRequestHandlerID = 
				::spawn_thread(HandleConnectionRequests, 
							kConnectionRequestHandlerName, 
							fPriority, 
							(void*)this);
				if (fConnectionRequestHandlerID > B_NO_ERROR)
				{
					::resume_thread(fConnectionRequestHandlerID);
				}
				
				// Start separate thread to listen for incoming client data.
				fClientListenerID = ::spawn_thread(ListenToClients, 
												kClientListenerName,
												fPriority,
												(void*)this);
				if (fClientListenerID > B_NO_ERROR)
				{
					::resume_thread(fClientListenerID);
				}
			}
		}
		else
		{
			// Error
		}
	}
	else
	{
		// Error
	}

	PRINT(( "Message Server now running...\n"));
	return theID;
}


// =============================================================================
//     SendToClients
// =============================================================================
long 
BTSNetMsgServer::SendToClients(BMessage* message)
{
	BTSSocket* 	sourceSocket;
	int			sourceID = -1;
	char* 		buf = NULL; 
	long 		numBytes = -1;
	int			i;
	BTSSocket* 	socket;
	long 		result = B_NO_ERROR;
	
	message->Flatten(&buf, &numBytes);
	
	// Check to see if it came from a client, so we
	// don't send it back to the same client.
	if (message->HasObject(SOURCE_SOCKET))
	{
		sourceSocket = (BTSSocket*)message->FindObject(SOURCE_SOCKET);
		sourceID = sourceSocket->ID();
	}
	if (buf != NULL && numBytes > 0)
	{
		// Make a copy of the client list in case it changes.
		::acquire_sem(fSocketListSem);
		BList socketList(fClientSocketList);
		::release_sem(fSocketListSem);
		
		// Loop and send data to all clients.
		for (i = 0; i < socketList.CountItems(); i++)
		{
			socket = (BTSSocket*)socketList.ItemAt(i);
			if (socket->ID() != sourceID)
			{
				PRINT(("BTSNetMsgServer - Send to client %ld\n", socket->GetID()));
				result = SendNetMessageData(*socket, numBytes, buf);
			}
			// Remove client if send fails.
			if (result != B_NO_ERROR) RemoveClient(socket);
		}
		free(buf);
	}
	return result;
}


// =============================================================================
//     MessageReceived
// =============================================================================
void 
BTSNetMsgServer::MessageReceived(BMessage* inMessage)
{
	// Messages that are not control messages are repeated either to the 
	// specified target client, or to all clients if no target is specified.
	
	PRINT(("BTSNetMsgServer::MessageReceived: ENTER\n"));
	switch (inMessage->what)
	{
		case DEAD_CONNECTION_MSG:
			// A connection has died or been closed. Delete it. This is 
			// a synchronous message, we don't want anyone to do anything
			// until we get rid of the dead socket. Ick.
			if (inMessage->HasObject(SOURCE_SOCKET))
			{	
				BTSSocket* socket = (BTSSocket*)inMessage->FindObject(SOURCE_SOCKET);
				if (RemoveClient(socket))
				{
					BMessenger* 	messenger = new BMessenger(fMessageReceiver);
					BMessage*		reply;
					BMessage* deadMessage = new BMessage(inMessage);

					PRINT(("Client removed.\n"));
					messenger->SendMessage(deadMessage, &reply);
					delete messenger;
					delete reply;
				}
			}
		break;
		
		case NEW_CLIENT_MSG:
			// Register the client's socket in our list of sockets.
			if (inMessage->HasObject(SOURCE_SOCKET))
			{
				BTSSocket* socket = (BTSSocket*)inMessage->FindObject(SOURCE_SOCKET);
				AddClient(socket);
				
				// Be sure to detach before reposting!
				DetachCurrentMessage();
				fMessageReceiver->PostMessage(inMessage);
			}
		break;
		
		default:
			BTSSocket* socket; 
			if (inMessage->HasObject(TARGET_SOCKET))
			{
				socket = (BTSSocket*)inMessage->FindObject(TARGET_SOCKET);
				SendNetMessage(*socket, inMessage);
			}	
			else if (!fClientSocketList.IsEmpty())
			{
				long result = SendToClients(inMessage);
			}
		break;
	}
	
	PRINT(("BTSNetMsgServer::MessageReceived: EXIT\n"));
	return;
}

// =============================================================================
//     AddClient
// =============================================================================
void
BTSNetMsgServer::AddClient(BTSSocket* socket)
{
	// Adds a client's socket id to the list. When the socket list is empty, 
	// the semaphore is alread aquired (see removeclient, constructor)
	
	if (!(fClientSocketList.IsEmpty()))
	{
		::acquire_sem(fSocketListSem);
	}
	fClientSocketList.AddItem((void*)socket);
	::release_sem(fSocketListSem);
	return;
}


// =============================================================================
//     RemoveClient
// =============================================================================
bool
BTSNetMsgServer::RemoveClient(BTSSocket* socket)
{
	// Removes 
	int 	result;				// Result of socket close.
	bool	removed = FALSE;	// Reports whether remove is successful.
	
	PRINT(( "Removing client listening at socket %d\n", socket));
	if (!(fClientSocketList.IsEmpty()))
	{
		::acquire_sem(fSocketListSem);
		if ((removed = fClientSocketList.RemoveItem((void*)socket)) == TRUE)
		{
			PRINT( ("Closing client socket %d\n", socket));
			
			// Close the socket.
			result = socket->Close();
			PRINT(( "Client socket %d closed, result is %d\n", 
					socket, result));
		}
		if (!(fClientSocketList.IsEmpty()))
		{
			::release_sem(fSocketListSem);
		}
	}
	return removed;
}

// =============================================================================
//     ListenToClients
// =============================================================================
long
BTSNetMsgServer::ListenToClients(void* arg)
{
	BTSNetMsgServer*	thisServer = (BTSNetMsgServer*)arg;
	BList*				serverSocketList = thisServer->SocketList();
	BLooper*			messageReceiver = thisServer->MessageReceiver();
	BMessage*			newMessage = NULL;
	sem_id				socketListSem = thisServer->SocketListSem();
	long				result = B_NO_ERROR;
	struct timeval 		tv;
	struct fd_set		readBits;
	BTSSocket* 			socket;
	int					i;
	
	// Set delay to wait for client data before returning.
	tv.tv_sec = 1;
	tv.tv_usec = 0;

	PRINT(("BTSNetMsgServer::ListenToClients - THREAD ENTER\n"));
	
	for (;;)
	{
		// Set the select bits for the known sockets
		FD_ZERO(&readBits);
		PRINT(("Acquiring socket list semaphore\n"));
		::acquire_sem(socketListSem);
		BList socketList(*serverSocketList);
		::release_sem(socketListSem);
		
		if (socketList.IsEmpty()) goto LOOPEND;
		for (i = 0; i < socketList.CountItems(); i++)
		{
			socket = (BTSSocket*)socketList.ItemAt(i);
			FD_SET(socket->ID(), &readBits);
		}
		
		// Blocks here until data arrives on any socket.
		if (::select(32, &readBits, NULL, NULL, &tv) <= 0) goto LOOPEND;
			
		PRINT(("Server received data\n"));
		for (i = 0; i < socketList.CountItems(); i++)
		{
			socket = (BTSSocket*)socketList.ItemAt(i);
			
			// Did data arrive on this socket?
			if (!(FD_ISSET(socket->ID(), &readBits))) goto SOCKETEND;
		
			// Yes..assume flattened BMessage format and go get it
			result = ReceiveNetMessage(*socket, &newMessage);
			
			if ( result != B_NO_ERROR ) goto SOCKETEND;
			
			// Post it to the message receiver.
			messageReceiver->PostMessage(newMessage);
							
			// Clean up before checking next socket.
			SOCKETEND:
							
			if (result == ECONNABORTED)
			{
				// Inform the server that a client went away.
				BMessage*	deadMessage = new BMessage(DEAD_CONNECTION_MSG);
				BMessenger* messenger = new BMessenger(thisServer);
				BMessage* 	reply = NULL;
				
				PRINT(("Connection aborted\n"));
				if (deadMessage != NULL && messenger != NULL)
				{
					if (deadMessage->Error() == B_NO_ERROR)
					{
						deadMessage->AddObject(SOURCE_SOCKET, (BObject*)socket);
						if (deadMessage->Error() == B_NO_ERROR)
						{
							messenger->SendMessage(deadMessage, &reply);
						}
						else delete deadMessage;
					}
					else delete deadMessage;
				}
				
				if (messenger != NULL) delete messenger;
				if (reply != NULL) delete reply;
			}
			result = B_NO_ERROR;
		}
		
		LOOPEND:
		// See if exit flag has been set.
		if (thisServer->IsExiting()) break;
	}
	
	PRINT(("BTSNetMsgServer::ListenToClients - THREAD EXIT\n"));
	exit_thread(result);
}

// =============================================================================
//     HandleConnectionRequests
// =============================================================================
long
BTSNetMsgServer::HandleConnectionRequests(void* arg)
{
	BTSNetMsgServer* 	thisServer = (BTSNetMsgServer*)arg;
	BTSSocket 			acceptSocket = thisServer->Socket();
	int 				clientSocket;
	sockaddr_in 		clientInterface;
	int					clientIntfSize = sizeof(sockaddr_in);
	long				result = 0;
	PRINT( ("HandleConnectionRequests - THREAD ENTER\n"));
	// Thread blocks here, on accept().
	while ((clientSocket = ::accept(acceptSocket.ID(), 
									(struct sockaddr*)&clientInterface,
									&clientIntfSize)) >= 0)
	{		
		PRINT(( "Connection request, new socket is %d\n", clientSocket));
		// A client has requested a connection. Make a handler for the
		// client socket.

		if (clientSocket >= 0)
		{
			BTSSocket* newClient = new BTSSocket(clientSocket);
			// Tell the server about the new client.
			BMessage* aMessage = new BMessage(NEW_CLIENT_MSG);
			if (aMessage != NULL)
			{
				if (aMessage->Error() == B_NO_ERROR)
				{
					aMessage->AddObject(SOURCE_SOCKET, (BObject*)newClient);
					if (aMessage->Error() == B_NO_ERROR)
					{
						thisServer->PostMessage(aMessage);
					}
					else delete aMessage;
				}
				else delete aMessage;
			}
		}
		else break;
		clientIntfSize = sizeof(sockaddr_in);
	}
	PRINT( ("HandleConnectionRequests - THREAD EXIT\n"));
	exit_thread(result);
}

