#include <net/socket.h>
#include <net/netdb.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <support/Debug.h>
#include <app/Application.h>

#include "BTSNetMsgClient.h"
#include "BTSNetMsgUtils.h"

// =============================================================================
//     BTSNetMsgClient
// =============================================================================
BTSNetMsgClient::BTSNetMsgClient(const unsigned short port, 
								const char* serverHostName,
								BLooper* messageReceiver, 
								const int family, 
								const int type, 
								const int protocol,  
								const long priority): 
								BLooper(), 
								fSocket(type, protocol, family),
								fAddress(family, port, serverHostName),
								fPriority(priority)
{
	// Make a socket connection and start listening for clients.
	fServerHostName	= ::strdup(serverHostName);
	fConnectorID = -1;
	fConnected = FALSE;
	fExiting = FALSE;
	
	if (messageReceiver == NULL);
	{
		messageReceiver = be_app;
	}
	fMessageReceiver = messageReceiver;
		return;
}


// =============================================================================
//     Run
// =============================================================================
thread_id
BTSNetMsgClient::Run()
{
	thread_id	id = -1;

	if (StartConnecting())
	{
			id = BLooper::Run();
	}
	return id;
}


// =============================================================================
//     Quit
// =============================================================================
void
BTSNetMsgClient::Quit()
{
	fExiting = TRUE;
	if (fConnectorID > -1 )
	{
		// Just disconnect.
		fSocket.Close();
	}
	
	BLooper::Quit();
	return;
}

// =============================================================================
//     StartConnecting
// =============================================================================
bool
BTSNetMsgClient::StartConnecting()
{
	bool	setupOK = TRUE;
	
	fConnected = FALSE;
	
	// Get a fresh socket connection each time.
	fSocket.Close();
	fSocket.Open();
			

	PRINT(("About to connect to server...\n"));
	// Spawn a thread that will connect to the server
	fConnectorID = spawn_thread(BTSNetMsgClient::ConnectToServer, 
					kConnectThreadName, fPriority, (char*)this);
	if (fConnectorID > 0)
	{
		resume_thread(fConnectorID);
	}
	else setupOK = FALSE;
	
	return setupOK;
}

// =============================================================================
//     MessageReceived
// =============================================================================
void 
BTSNetMsgClient::MessageReceived(BMessage* inMessage)
{
	PRINT(("BTSNetMsgClient::MessageReceived - ENTER\n"));
	if (!fExiting)
	{
		switch (inMessage->what)
		{
			case DEAD_CONNECTION_MSG:
				fConnected = FALSE;
			break;
			
			case CONNECT_MSG:
				DetachCurrentMessage();
				fConnectorID = -1;
				fConnected = TRUE;
				fMessageReceiver->PostMessage(inMessage);
				fSocketListenerID = spawn_thread(ListenToServer, "Client socket listener",
						fPriority, this);
				resume_thread(fSocketListenerID);
			break;
			
			default:
				if (fConnected)	SendNetMessage(fSocket, inMessage);
			break;
		}
	}	
	PRINT(("BTSNetMsgClient::MessageReceived - EXIT\n"));
	return;
}

// =============================================================================
//     ConnectToServer
// =============================================================================
long 
BTSNetMsgClient::ConnectToServer(void* arg)
{
	// Static function that runs in a separate thread. His whole purpose
	// is to connect to a server, then he goes away. This prevents the
	// main client thread from blocking if a server isn't immediately
	// available.
	BTSNetMsgClient* 	thisClient = (BTSNetMsgClient*)arg;
	BTSAddress			address = thisClient->Address();
	BTSSocket			socket = thisClient->Socket();
	int					result;
		
	// Specify server connection info.
	result = socket.ConnectToAddress(address);	
	if (result >= 0 && !(thisClient->IsExiting()))
	{
		// Since we connected ok, create a socket handler for the 
		// client socket. Also, notify client that we are connected.
		thisClient->PostMessage(CONNECT_MSG);
		PRINT(("connected to server!\n"));		
	}
	exit_thread(result);
}

// =============================================================================
//     ListenToServer
// =============================================================================
long 
BTSNetMsgClient::ListenToServer(void* arg)
{
	BTSNetMsgClient* 	thisClient = (BTSNetMsgClient*)arg;
	BLooper*			messageReceiver = thisClient->MessageReceiver();
	BMessage*			newMessage = NULL;
	struct fd_set		readBits;
	struct timeval 		tv;
	long 				result;
	BTSSocket 			socket = thisClient->Socket();
	int					socketID = socket.ID();
	bool				exit = FALSE;
	
	tv.tv_sec = 1;
	tv.tv_usec = 0;
	while (exit == FALSE)
	{
		FD_ZERO(&readBits);
		FD_SET(socketID, &readBits);
		if (::select(socketID + 1, &readBits, NULL, NULL, &tv) > 0)
		{
			if (FD_ISSET(socketID, &readBits))
			{
				PRINT(("BTSNetMsgClient::ListenToServer - SERVER MSG on %d\n", 
							socketID));
				result = ReceiveNetMessage(socket, &newMessage);
				if (result == B_NO_ERROR && newMessage != NULL)
				{
					// Post it to the message receiver.
					messageReceiver->PostMessage(newMessage);
				}
				
				else if (result == ECONNABORTED)
				{
					// Connection has died
					if (!thisClient->IsExiting())
					{
						thisClient->PostMessage(DEAD_CONNECTION_MSG);
					}
					exit = TRUE;
				}
			}

		}
	 	if (thisClient->IsExiting())
		{
			exit = TRUE;
		}
	}
	exit_thread(result);
}
