#include <MidiStore.h>
#include <File.h>
#include <Entry.h>
#include <malloc.h>
#include <stdio.h>
#include <Path.h>
#include <Directory.h>
#include <Midi.h>
#include <MidiDefs.h>
#include <MidiSynthFile.h>
#include <ctype.h>
#include "axe_names.h"
#include "p2kn.h"
#include <MediaKit.h>

int32 prev_time;
int32 max_notes;
int32 note_ktr;
int32 last_timestamp;
char record_buf[8192];
BMidiStore *store;
char output_filename[B_PATH_NAME_LENGTH];
BFile sound_file;
bool play_it, debug_it;
float off_map[17][128];
int32 off_vmap[17][128];
reverb_mode reverb;
bool do_record;
size_t file_byte_count;
int32 mute_map[17];
int32 sampling_rate;
bool jarek_mode, benoit_mode;
bool finish_override;

#define ENV_COUNT 1024

typedef struct _envelope {
	int32 start_time;
	int32 stop_time;
	float start_val;
	float stop_val;
	int32 channel;
	int32 keynum;
	int32 control;
} envelope;

envelope envs[ENV_COUNT];

status_t get_line(char **buf, char *out);
void set_axe(char *buf, uchar channel, float time);
void envelope_start(float val, float time,
	int32 channel, int32 keynum, int32 control);
void envelope_stop(float val, float time,
	int32 channel, int32 keynum, int32 control);
void apply_envelope(envelope env);
void set_pan(char *buf, int32 channel, float time);
int32 get_keynum(char *note, int32 transposition);
int32 do_note(char *ptr, int32 channel, float time, float pulse,
	int32 transposition);
void proc_buf(char *buf);
status_t get_output_ref(entry_ref *ref);
void set_reverb(char *buf);
void set_axe(char *buf, int32 channel, float time);
void init_sound_file();
void finish_writing();
bool record_proc(void *, char *buffer, size_t count,void *);
status_t stop_recording(void *buf, status_t error);

#define skew2num(x) (((x+1.0)/2.0)*127)
#define loc2num(x) (((x+1.0)/2.0)*127)

void envelope_start(float val, float time,
	int32 channel, int32 keynum, int32 control)
{
	int32 ktr;
	for (ktr = 0; ktr < ENV_COUNT; ktr++) 
		if (envs[ktr].channel == 0) 
			break;
	if (ktr == ENV_COUNT)
		return;

	envs[ktr].start_val = val;
	envs[ktr].start_time = time;
	envs[ktr].channel = channel;
	envs[ktr].keynum = keynum;
	envs[ktr].control = control;
}

void envelope_stop(float val, float time,
	int32 channel, int32 keynum, int32 control)
{
	int32 ktr;
	
	for (ktr = 0; ktr < ENV_COUNT; ktr++) 
		if (envs[ktr].channel == channel &&
			envs[ktr].keynum == keynum &&
			envs[ktr].control == control) 
				break;
	
	if (ktr == ENV_COUNT)
		return;
	else {
		envs[ktr].stop_val = val;
		envs[ktr].stop_time = time;
	}
	apply_envelope(envs[ktr]);
	envs[ktr].channel = 0;
}

/* Update latency in millilseconds */
#define CLICK_RATE 10

void apply_envelope(envelope env)
{
	int32 ktri;
	int32 dur;
	int32 clicks;
	float incr;
	int32 this_val, prev_val, this_time;

	if (env.stop_time <= env.start_time)
		return;
		
	dur = env.stop_time - env.start_time;
	clicks = dur/CLICK_RATE;
	incr = env.stop_val - env.start_val;
	incr /= clicks;

	for (ktri = 0; ktri < clicks; ktri++) {
		this_val = env.start_val + (ktri*incr);
		if (this_val == prev_val) continue;
		prev_val = this_val;
		this_time = env.start_time + (ktri*CLICK_RATE);
	
		// Teeny offset to avoid hitting all channels at once.
		this_time+=env.channel;

		switch (env.control) {
			case B_KEY_PRESSURE	:
				store->KeyPressure(env.channel, env.keynum, 	
					this_val, this_time);
				break;
			case B_CHANNEL_PRESSURE	:
				store->ChannelPressure(env.channel, this_val, this_time);	
				break;
			case B_PITCH_BEND:
				store->PitchBend(env.channel, 0, this_val, this_time);	
				break;
			default:
				store->ControlChange(env.channel, env.control,
					this_val, this_time);
				break;
		}
	}
}

status_t get_line(char **buf, char *out)
{
	char *stake;
	if ((stake=strchr(*buf, '\n')) == 0) 
		return -1;
	
	*stake = 0;
	strcpy(out, *buf);
	*buf = stake+1;
	return 0;
}

void set_reverb(char *buf)
{
	while (isspace(*buf)) buf++;

	if (isdigit(*buf))
		sscanf(buf, "%d", &reverb);
	else {
		if (strcmp(buf, "B_REVERB_NONE") == 0)
			reverb = (reverb_mode)1;
		else if (strcmp(buf, "B_REVERB_CLOSET") == 0)
			reverb = (reverb_mode)2;
		else if (strcmp(buf, "B_REVERB_GARAGE") == 0)
			reverb = (reverb_mode)3;
		else if (strcmp(buf, "B_REVERB_BALLROOM") == 0)
			reverb = (reverb_mode)4;
		else if (strcmp(buf, "B_REVERB_CAVERN") == 0)
			reverb = (reverb_mode)5;
		else if (strcmp(buf, "B_REVERB_DUNGEON") == 0)
			reverb = (reverb_mode)6;
	}
}	

void set_axe(char *buf, int32 channel, float time)
{
	char *axe_name_ptr;
	int32 ktr;
	
	while (isspace(*buf)) buf++;
	if (isdigit(*buf))
		sscanf(buf, "%d", &ktr);
	else {
		for (ktr = 0; ktr < 128; ktr++) {
			axe_name_ptr = &axe_names[ktr][0];
			if (strncmp(buf, axe_name_ptr, strlen(buf)) == 0)
				break;
		}
	}

	store->ProgramChange(channel, ktr, time);
}

void set_pan(char *buf, int32 channel, float time)
{
	float pan;
	sscanf(buf, "%f", &pan);
	pan = max_c(min_c(pan, 1.0), -.9999);
	store->ControlChange(channel, B_PAN, loc2num(pan), time);
}

int32 get_keynum(char *note, int32 transposition)
{
	int32 kn, ktr;
	char *pptr;

	while (isspace(*note)) note++;

	if (isdigit(*note))
		sscanf(note, "%d", &kn);
	else {
		for (ktr = 0; ktr < PITCH_COUNT; ktr++) {
			pptr = p2kns[ktr].p;
			if (strcmp(note, p2kns[ktr].p) == 0) {
				kn = p2kns[ktr].kn;
				break;
			}
		if (ktr == PITCH_COUNT)
			return -1;
		}
	}	
	kn += transposition;
	if (kn < 0 || kn > 127)
		return -1;
	return kn;
}


int32 do_note(char *ptr, int32 channel, float time, float pulse,
	int32 transposition)
{
	char note[6];
	float dur = 1.0;
	float on_vel=.9, off_vel;
	int32 kn;
	int32 scan_count;

	scan_count = sscanf(ptr, "%s %f %f %f", note, &dur, &on_vel, &off_vel);

	if ((kn = get_keynum(note, transposition)) < 0) return -1;
	dur *= 1000;

	if (dur < 0) 
		dur *= -1;
	else
		dur *= pulse;
	
	if (scan_count < 4) off_vel = on_vel;
	on_vel *= 127;
	off_vel *= 127;
	if (time > off_map[channel][kn] && off_map[channel][kn] != -1) {
		store->NoteOff(channel, kn, off_vmap[channel][kn], 
			off_map[channel][kn]);
		off_map[channel][kn] = -1;		
		note_ktr--;
	}
	store->NoteOn(channel, kn, on_vel, time);
	note_ktr++;
	max_notes = max_c(max_notes, note_ktr);
	
/*	if (off_map[channel][kn] > time)
		store->NoteOff(channel, kn, off_vmap[channel][kn], 
			time-.052);
	else
		note_ktr++;*/
	off_map[channel][kn] = (time+dur);
	last_timestamp = max_c(last_timestamp, off_map[channel][kn]);
	off_vmap[channel][kn] = off_vel;
	return kn;
}

void proc_buf(char *buf)
{
	char line[1024];
	char *ptr;
	float pulse = 1.0;
	int32 transposition = 0;
	int32 channel = 1;
	float tmp_float =0;
	int32 tmp_int;
	float time = 0.0;
	float time2 = 0.0;
	float offset = 0.0;
	float tmp_time;
	uint32 ktr,ktr1;
	char env_char;
	float env_val;
	int32 env_num;
	char env_key[6];
	char tmp_string[32];
	int32 env_kn;
	int32 env_control;
	bool in_comment = false;
	prev_time = 0;
	reverb = (reverb_mode)1;
	memset(line, 0, 1024);
	max_notes = 0;
	last_timestamp = 0;
	
	for (ktr = 0; ktr < ENV_COUNT; envs[ktr++].channel = 0) {}

	for(ktr=1; ktr<17;ktr++)
		for(ktr1=0;ktr1<128;ktr1++)
			off_map[ktr][ktr1] = -1;
			
	while (get_line(&buf, line) == 0) {
		ptr = line;
		if (in_comment) {
			if (line[0] == '#') 
				in_comment = false;
			continue;
		}
		switch (line[0]) {
	 	case ';': 
	 		break;
	 	case '#':
	 		in_comment = true;
	 		break; 
		case 'X':
	 		ptr++;
	 		sscanf(ptr, "%d", &tmp_int);
			mute_map[tmp_int] = 1;
			printf("Setting mute %d\n", tmp_int);
			break;
	 	case 'O':
	 		ptr++;
	 		sscanf(ptr, "%s", output_filename);
	 		break;
	 	case 'P':
	 		play_it = true;
	 		break;
	 	case 'D':
	 		debug_it = true;
	 		break;
	 	case 'R':
	 		ptr++;
	 		set_reverb(ptr);
	 		printf("Reverb %d\n", reverb);
	 		break;
	 	case 'B':
	 		ptr++;
	 		sscanf(ptr, "%f", &pulse);
	 		break;
		case 'C':
			ptr++;
			sscanf(ptr, "%d", &channel);
			if (mute_map[channel] == 1) {
				break;
			}
			time = 0;
			time2 = 0;
			transposition = 0;
			if (jarek_mode) {
				if (channel == 16) 
					set_pan("1.0", channel, (offset+time+time2));
				else
					set_pan("-1.0", channel, (offset+time+time2));
			}		
			if (benoit_mode) {
				set_pan("0.0", channel, (offset+time+time2));
			}
			break;
		case 'Q':
			do_record = true;
			break;
			
		case 'G':
			ptr++;
			sscanf(ptr, "%d", &tmp_int);
			sampling_rate = tmp_int;
			break;
		default:	
			if (mute_map[channel] == 1)
				break;
			switch(line[0]) {
			case 'F':
				ptr++;
				sscanf(ptr, "%s", tmp_string);
				fprintf(stderr, "%s -> %f\n",tmp_string,
					(offset+time+time2)*.001);
			case 'T':	
				ptr++;
				sscanf(ptr, "%d", &transposition);
				break;
			case 'S':
				ptr++;
				sscanf(ptr, "%f", &tmp_float);
				store->PitchBend(channel, 1, skew2num(tmp_float), 
					(offset+time+time2));
				break;
		 	case 'A':
		 		ptr++;
		 		sscanf(ptr, "%f", &tmp_float);
		 		store->ControlChange(channel, B_MAIN_VOLUME, tmp_float*127,
					(offset+time+time2));
		 		break;
		 	case 'K':
		 		ptr++;
		 		sscanf(ptr, "%d %f", &tmp_int, &tmp_float);
		 		store->KeyPressure(channel, tmp_int, tmp_float*127,
					(offset+time+time2));
		 		break;
		 	case 'U':
		 		ptr++;
		 		sscanf(ptr, "%f", &tmp_float);
		 		store->ChannelPressure(channel, tmp_float*127,
					(offset+time+time2));
		 		break;
		 	case 'W':
		 		ptr++;
		 		sscanf(ptr, "%f", &tmp_float);
		 		store->ControlChange(channel, B_MODULATION, tmp_float*127,
					(offset+time+time2));
		 		break;
			case 'V':
				ptr++;
				set_axe(ptr, channel, (offset+time+time2));
				break;
			case 'L':
				if (jarek_mode||benoit_mode) break;
				ptr++;
				set_pan(ptr, channel, (offset+time+time2));
				break;
			case 'M':
				ptr++;
				store->ControlChange(channel, B_MONO_MODE_ON, 1, 
					(offset+time+time2));
				break;
			case 'N':
				ptr++;
				store->ControlChange(channel, B_POLY_MODE_ON, 1, 
					(offset+time+time2));
				break;
			case '-':
				ptr++;
				sscanf(ptr, "%f", &tmp_time);
				offset += (tmp_time*pulse*1000);
				time =0.0;
				time2 = 0.0;
				break;
			case '+':
				ptr++;
				sscanf(ptr, "%f", &tmp_time);
				time += (tmp_time*pulse*1000);

				time2 = 0.0;
				break;
			case '~':
				ptr++;
				sscanf(ptr, "%f", &tmp_time);
				time2 += (tmp_time*pulse*1000);
				break;
			case '!':
				ptr++;
				sscanf(ptr, "%f", &tmp_time);
				time = (tmp_time*pulse*1000);
				break;

			case 'E':
				env_char = *++ptr;
				if (sscanf(++ptr, "%f %s", &env_val, env_key) == 2) {
					if ((env_kn = get_keynum(env_key, transposition)) < 0)
						break;
				}
				else
					env_kn = 0;
				switch (env_char) {
					case 'a': /* main volume*/
					case 'A':
						env_control = B_MAIN_VOLUME;
						env_num = env_val*127;
						break;
					case 'k': /* key pressure */
					case 'K':
						env_control = B_KEY_PRESSURE;
						env_num = env_val*127;
						break;
					case 'c': /* key pressure */
					case 'C':
						env_control = B_CHANNEL_PRESSURE;
						env_num = env_val*127;
						break;
					case 'b': /* key pressure */
					case 'B':
						env_control = B_PITCH_BEND;
						env_num = skew2num(env_val);
						break;
					case 'm': /* modulation */
					case 'M':
						env_num = env_val*127;
						env_control = B_MODULATION;
						break;
				}

				if (isupper(env_char))
					envelope_start(env_num, (offset+time+time2),
						channel, env_kn, env_control);
				else
					envelope_stop(env_num, (offset+time+time2),
						channel, env_kn, env_control); 
				break;
				
			default:
				while (isspace(*ptr)) ptr++;
				
				if (*ptr != NULL)
					do_note(ptr, channel, (offset+time+time2), 
						pulse, transposition);
				break;
			}
		}
	}
	for (ktr=1; ktr < 17; ktr++)
		for (ktr1=0; ktr1<128; ktr1++) 
			if (off_map[ktr][ktr1] != -1) {
				store->NoteOff(ktr, ktr1, off_vmap[ktr][ktr1], 
					off_map[ktr][ktr1]);
 				note_ktr--;
			}
}		

status_t get_output_ref(entry_ref *ref)
{
	BPath out_path;			
	entry_ref out_ref;
	BEntry out_entry;
	BFile f;
	out_path.SetTo(output_filename);
	if (out_path.InitCheck() != B_NO_ERROR) {
		printf("10: %s\n", strerror(out_path.InitCheck()));
		return -13;
	}
	
	out_entry.SetTo(out_path.Path());
	if (out_entry.InitCheck() != B_NO_ERROR) {
		printf("11: %s\n", strerror(out_entry.InitCheck()));
		return -13;
	}

	if (!out_entry.Exists()) {
		out_path.GetParent(&out_path);
		create_directory(out_path.Path(), 0777);
	}

	f.SetTo(&out_entry, B_CREATE_FILE|B_ERASE_FILE|B_READ_WRITE);
	f.Unset();	
	out_entry.GetRef(&out_ref);
	*ref = out_ref;
	return B_OK;
}

#define HEADER_SIZE 128
void init_sound_file()
{
	if (benoit_mode) {
		sound_file.Seek(0,0);
		return;
	}

	char null_buf[HEADER_SIZE];
	memset(null_buf, 0, HEADER_SIZE);
	sound_file.Seek(0,0);
	sound_file.Write(null_buf, HEADER_SIZE);
	sound_file.Seek(HEADER_SIZE, 0);
}

void finish_writing()
{
	if (benoit_mode)
		return;
		
	char hdrbuf[4];
	sound_file.Seek(0,0);
	sound_file.Write(".snd", 4);
	(*((int *)(hdrbuf)))=HEADER_SIZE;
	sound_file.Write(hdrbuf, 4);
	(*((int *)(hdrbuf)))=file_byte_count;
	sound_file.Write(hdrbuf, 4);
	(*((int *)(hdrbuf)))=3;
	sound_file.Write(hdrbuf, 4);
	(*((int *)(hdrbuf)))=44100;
	sound_file.Write(hdrbuf, 4);
	(*((int *)(hdrbuf)))=2;
	sound_file.Write(hdrbuf, 4);
	(*((int *)(hdrbuf)))=0;
	sound_file.Write(hdrbuf, 4);
}

BMidiSynthFile *msf;
int32 safety = 36;

bool record_proc(void *, char *buffer, size_t count,void *)
{
	int ktr;
	int16 *bptr;
	int32 *sptr;
	int32 real_count;
	static bool doit = false;

	sptr = (int32 *)buffer;
	bptr = (int16 *)buffer;
	if (!finish_override && msf->IsFinished() && safety-- < 0) {

		for (ktr = 0; ktr < 400; ktr++) {
			if (buffer[ktr] != 0)
				break;
		}
		if (ktr == 400) {
			printf("DONE RECORDING %d\n", file_byte_count);
			return false;
		}
	}
	else {
		for (ktr = 0; ktr < count/4; ktr++) {
			if (*sptr != 0 || doit) {
				real_count = count-(ktr*4);
				memcpy(record_buf, (char *)sptr, real_count);
				sound_file.Write(record_buf, real_count);
				file_byte_count += real_count;
				doit = true;
				break;
			}
			sptr++;
		}
	}
	return true;
}

sem_id record_sem;

BSubscriber *recorder;

status_t stop_recording(void *, status_t error)
{
	acquire_sem(record_sem);	
	if (do_record) {
		do_record = true;
		finish_writing();
	}
	release_sem(record_sem);
	return error;
}	

int main(int32 argc, char *argv[])
{
	BFile in;
	BPath in_path;
	BFile f;
	char *buf;
	off_t file_size;
	status_t e;
	entry_ref out_ref;
	int32 ktr, ktr1;
	int32 tmp;
	BDACStream *dac;
	char sound_filename[B_FILE_NAME_LENGTH];

	store = new BMidiStore();
	play_it = false;
	debug_it = false;
	finish_override = false;
	output_filename[0] = NULL;
	sampling_rate = 44100;
	do_record = false;
	jarek_mode = benoit_mode = false;
	record_sem = create_sem(0, "record sem");
	bool first_solo = true;
	char rec_dir[B_FILE_NAME_LENGTH];
	char solo_name[B_FILE_NAME_LENGTH];
	char *vptr;

	for (ktr=1; ktr<17; ktr++) {
		mute_map[ktr] = 0;
	}
	if (argc < 2) {
		printf("Usage: edmidi InFile [ m o 2 r[Dir] R Xn Sn ] \n");
		printf(" m	create raw mono data\n");
		printf(" o	disable autofinish (see warning below)\n");
		printf(" 2	sampling rate = 22050 (default is 44100)\n");
		printf(" r[Dir]	record output as a sound file (named DirInFile.snd)\n");
		printf(" R	rehearsal mode; chan[1-15] = left, chan16 = right\n");
		printf(" Xn	mute channel n\n");
		printf(" Sn	solo channel n (you can solo more than one channel)\n\n");
		printf("* The InFile must come before the options.\n\n");
		printf("* No option dashes are needed.\n\n");
		printf("* All options must be separated.  For example:\n");
		printf("  $ edmidi lbjtest.lbj o r/boot/home/snds/ S1 S2 S3\n\n");
		printf("* Solo'd channels are named in the sound file name. For example:\n");
		printf(" 	/boot/home/snds/lbjtestS1S2S3.snd\n\n");
		printf("* If you're recording, only the first performance is recorded.\n");
		printf("  Replays (through the Replay? prompt) aren't recorded.\n\n");
		printf("* The sound file dir (r[Dir]) is prepended literally(!). \n");
		printf("  If you intend to send the file to a directory, you must\n");
		printf("  include the trailing slash (as shown above).\n\n");
		printf("* OPTION 'o' WARNING:  Normally, the performance ends when the\n");
		printf("  midi file is finished playing.  However, this may clip the\n");
		printf("  the end of a recording.  To work around this, use the 'o' option,\n");
		printf("  But be sure to actually stop the recording (by responding to the\n");
		printf("  Replay? prompt) or you'll end up with a huge sound file.\n\n");
		return -13;
	}
	memset(solo_name, 0, B_FILE_NAME_LENGTH);
	memset(rec_dir, 0, B_FILE_NAME_LENGTH);
	memset(sound_filename, 0, B_FILE_NAME_LENGTH);

	if (argc > 2) {
		for (ktr = 2; ktr < argc; ktr++) {

			if (strcmp(argv[ktr], "m") == 0) {
				benoit_mode = true;
				continue;
			}
			if (strcmp(argv[ktr], "o") == 0) {
				finish_override = true;
				continue;
			}
			if (strcmp(argv[ktr], "2") == 0) {
				sampling_rate = 22050;
				continue;
			}
			if (strncmp(argv[ktr], "r", 1) == 0) {
				play_it = true;
				do_record = true;
				if (strlen(argv[ktr]) > 0) {
					vptr = argv[ktr];
					vptr++;
					strcpy(rec_dir, vptr);
					printf("Recording to %s\n", rec_dir);
				}
				continue;
			}
			if (argv[ktr][0] == 'X') {
				sscanf(argv[ktr], "X%d", &tmp);
				if (tmp < 17) 
					mute_map[tmp] = 1;
			}
			if (argv[ktr][0] == 'R') {
				jarek_mode = true;
			}
			
			if (argv[ktr][0] == 'S') {
				sscanf(argv[ktr], "S%d", &tmp);
				if (tmp < 17) {
					if (first_solo) {
						first_solo = false;
						for (ktr1=1; ktr1<17; ktr1++) 
							mute_map[ktr1] = 1;
					}
					mute_map[tmp] = 0;
					sprintf(solo_name, "%sS%d", solo_name, tmp);
				}
			}

		}
	}
	in_path.SetTo(argv[1]);
	if (in_path.InitCheck() != B_NO_ERROR) {
		printf("1: %s\n", strerror(in_path.InitCheck()));
		return -13;
	}
	
	in.SetTo(in_path.Path(), B_READ_ONLY);
	if (in.InitCheck() != B_NO_ERROR) {
		printf("2: %s\n", strerror(in.InitCheck()));
		return -13;
	}
	
	in.GetSize(&file_size);
	buf = (char *)malloc(file_size+1);
	if (!buf) {
		printf("3:  Couldn't malloc\n");
		return -13;
	}
 
 	if ((e=in.Read(buf, file_size)) != file_size) {
 		printf("4: %s\n", strerror(e));
 		return -13;
 	}
 	proc_buf(buf);
	store->SortEvents();
	printf("Last timestamp: %f\n", last_timestamp*.001);
	printf("Max notes: %d\n", max_notes);
	
	if (*output_filename) {
		if (get_output_ref(&out_ref) != B_OK) {
			printf("5:  Couldn't getoutput ref\n");
			return -13;
		}	
		printf("Writing midi %s\n", output_filename);
		store->Export(&out_ref, 1);
	}
	
	if (debug_it) {
		BMidiText *mt = new BMidiText();
		store->Connect(mt);
	}
	
	if (do_record) {
		if (rec_dir) {
			strcpy(sound_filename, rec_dir);
			//strcat(sound_filename, "/");
		}
		strcat(sound_filename, argv[1]);
		char *ptr = strchr(sound_filename, '.');
		if (ptr)
			*ptr = '\0';
		strcat(sound_filename, solo_name);
		strcat(sound_filename, ".snd\0");
			
		sound_file.SetTo(sound_filename, B_CREATE_FILE|B_ERASE_FILE|
			B_WRITE_ONLY);

		if ((e=sound_file.InitCheck()) != B_OK) {
			printf("6: %s %s\n", strerror(e), sound_filename);
			return -13;
		}

		printf("Writing sound %s\n", sound_filename);
		init_sound_file();
	}
	if (play_it && *output_filename) {
		msf = new BMidiSynthFile();
		msf->LoadFile(&out_ref);
	}
	else
		play_it = false;

	if (debug_it) {
		store->Start();
	}

	file_byte_count = 0;
	if (play_it) {
		char c;
		be_synth->SetVoiceLimits(32,0,12);

		be_synth->SetSamplingRate(sampling_rate);	
		printf("Setting sampling rate %d\n", sampling_rate);
		if (reverb == 1)
			be_synth->EnableReverb(false);
		else {
			be_synth->EnableReverb(true);
			be_synth->SetReverb((reverb_mode)reverb);
		}

		if (do_record) {
			dac = new BDACStream();
			recorder = new BSubscriber();
			if ((e=recorder->Subscribe(dac)) != B_OK) {
				printf("Couldn't subscribe\n");
				return -13;
			}
		}

		while (1) {
			if (do_record) {
				if ((e= recorder->EnterStream(NULL, false, NULL,
					record_proc, stop_recording, true)) != B_OK) {
					printf("Couldn't enter %x\n", e);
					return -13;
				}
			}

			msf->Start();
			printf("Replay [y]?");
			fflush(stdout);
			c = getchar();
			if (do_record)
				recorder->ExitStream();
			getchar();
			
			if (c != 'y')
				break;
		}
	}
	while (store->IsRunning())
		snooze(2000000);

}

