#include "MotorMix.h"
#include "MMDevice.h"
#include "MMProducer.h"
#include "MMConsumer.h"

#include <Debug.h>
#include <Messenger.h>
#include <MidiRoster.h>

#define FUNCTION PRINT

/* MotorMix */
MotorMix::MotorMix(const char *name) :
	mName(NULL),
	mRoster(BMidiRoster::MidiRoster()),
    mDevices(NULL),
	mChannels(NULL),
	mDeviceIDs(0),
	mDeviceCount(0),
	mChannelCount(0),
	mCurrentChannel(0),
	mMessenger(NULL),
	mStoreState(true)
{
	FUNCTION(("MotorMix Constructor\n"));
	mName = strdup(name);
}

MotorMix::~MotorMix()
{
	FUNCTION(("MotorMix Destructor\n"));
	delete[] mChannels;
}

status_t 
MotorMix::Init()
{
	FUNCTION(("MotorMix Init\n"));
	return B_OK;
}

//#pragma mark --DeviceManagment--
int32 
MotorMix::AddDevice(const char *name, uchar midiChannel, BMidiProducer *input, BMidiConsumer *output, MotorMix *motormix)
{
	FUNCTION(("MotorMix AddDevice %s\n", name));
	if (!input || !output)
		return -1;
	MotorMixDevice **dev = &mDevices;
	while (*dev)
		dev = &(*dev)->mNext;
	*dev = new MotorMixDevice(name, mDeviceIDs++, midiChannel, input, output, motormix);
	PRINT(("MotorMixDevice new dev: 0x%x\n", dev));
	if ((*dev)->InitCheck() < B_OK)
		PRINT(("MotorMixDevice failed InitCheck\n"));
	mDeviceCount++;
	return (*dev)->ID();
}

void 
MotorMix::RemoveDevice(int32 device_id)
{
	FUNCTION(("MotorMix RemoveDevice %ld\n", id));
	if (device_id < 0)
		return;
	MotorMixDevice **dev = &mDevices;
	while (*dev) {
		if ((*dev)->ID() == device_id) {
			MotorMixDevice *del = *dev;
			*dev = del->mNext;
			delete del;
			mDeviceCount--;
		}
		dev = &(*dev)->mNext;
	}
}

int32 
MotorMix::CurrentChannelFor(int32 device_id)
{
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ID() == device_id)
			return dev->CurrentChannel();
		else
			dev = dev->NextDevice();
	}
	return 0;
}

void 
MotorMix::SetCurrentChannelFor(int32 device_id, int32 channel)
{
	/* dug - do the right thing */
	if (channel > mChannelCount - (mDeviceCount * MMCHANNEL_COUNT))
		channel = mChannelCount - (mDeviceCount * MMCHANNEL_COUNT);
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ID() == device_id) {
			dev->SetCurrentChannel(channel);
			return;
		}
		dev = dev->NextDevice();
	}
}

void 
MotorMix::SetAllFor(int32 device_id)
{
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ID() == device_id) {
			/* do all the stuff */
			return;
		}
		dev = dev->NextDevice();
	}	
}

void 
MotorMix::ResetAllFor(int32 device_id)
{
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ID() == device_id) {
			/* do all the stuff */
			return;
		}
		dev = dev->NextDevice();
	}	
}

//#pragma mark --StateManagment--
int32 
MotorMix::CountChannels()
{
	FUNCTION(("MotorMix CountChannels %ld\n", mChannelCount));
	return mChannelCount;
}

void 
MotorMix::SetChannelCount(int32 count)
{
	FUNCTION(("MotorMix SetChannelCount %ld\n", count));
	MotorMixChannel *channels = new MotorMixChannel[count];
	memset(channels, 0, sizeof(MotorMixChannel) * count);
	memcpy(channels, mChannels, sizeof(MotorMixChannel) * MIN(mChannelCount, count));
	delete[] mChannels;
	mChannels = channels;
	mChannelCount = count;
}

void 
MotorMix::SetTitle(int32 channel, const char *title)
{
	FUNCTION(("MotorMix SetTitle channel %ld: %s\n", channel, title));
	ASSERT(channel >= 0 && channel < mChannelCount);
	strncpy(mChannels[channel].mTitle, title, MOTOR_MIX_NAME_LENGTH);
	MotorMixDevice *dev = mDevices;
	size_t size = 0;
	uchar str[size];
	while (dev) {
		if (dev->ChannelIsVisible(channel)) {
			dev->Producer()->SprayData((void *)str, size, false);
		}
		dev = dev->NextDevice();
	}
}

void 
MotorMix::SetSelect(int32 channel, light_state state)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	mChannels[channel].mSelect = state;
	uchar lsb = 0x01 + state;
	SetChannelLED(channel, lsb);
}

void 
MotorMix::SetBypass(int32 channel, light_state state)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	mChannels[channel].mBypass = state;
	uchar lsb = 0x04 + state;
	SetChannelLED(channel, lsb);
}

void 
MotorMix::SetRecord(int32 channel, light_state state)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	mChannels[channel].mRecord = state;
	uchar lsb = 0x05 + state;
	SetChannelLED(channel, lsb);
}

void 
MotorMix::SetSolo(int32 channel, light_state state)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	mChannels[channel].mSolo = state;
	uchar lsb = 0x03 + state;
	SetChannelLED(channel, lsb);
}

void 
MotorMix::SetMute(int32 channel, light_state state)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	mChannels[channel].mMute = state;
	uchar lsb = 0x02 + state;
	SetChannelLED(channel, lsb);
}

void 
MotorMix::SetFader(int32 channel, int16 value)
{
	FUNCTION(("MotorMix SetFader channel %ld: %d\n", channel, value));
	ASSERT(channel >= 0 && channel < mChannelCount);
	if (value > MMFADER_MAX)
		value = MMFADER_MAX;
	else if (value < MMFADER_MIN)
		value = MMFADER_MIN;
	mChannels[channel].mFader = value;
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ChannelIsVisible(channel)) {
			uchar con = (uchar)(channel - dev->CurrentChannel());
			uchar msb = (uchar)(value >> 7);
			uchar lsb = (uchar)(value & 0x7f);
			dev->SetControl(con, msb, lsb);
		}
		dev = dev->NextDevice();
	}
}

const char *
MotorMix::GetTitle(int32 channel)
{
}

light_state 
MotorMix::GetSelect(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mSelect;
}

light_state 
MotorMix::GetBypass(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mBypass;
}

light_state 
MotorMix::GetRecord(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mRecord;
}

light_state 
MotorMix::GetSolo(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mSolo;
}

light_state 
MotorMix::GetMute(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mMute;
}

int16 
MotorMix::GetFader(int32 channel, bigtime_t *last)
{
	ASSERT(channel >= 0 && channel < mChannelCount);
	return mChannels[channel].mFader;
}

//#pragma mark --Notification--
void 
MotorMix::SetMessenger(BMessenger *messenger)
{
	FUNCTION(("MotorMix SetMessenger\n"));
	mMessenger = messenger;
}

void 
MotorMix::SetStateStorage(bool on)
{
	mStoreState = on;
}

void 
MotorMix::Selected(int32 device_id, int32 channel, switch_state state, bigtime_t time)
{
	if (mStoreState) {
		switch (state) {
			case MM_PRESS:
				break;
			case MM_RELEASE:	
				if (mChannels[channel].mSelect == MM_OFF)
					SetSelect(channel, MM_ON);
				else
					SetSelect(channel, MM_OFF);
				break;
		}
	}
	int16 value = state;
	BMessage msg(MM_SELECT_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

void 
MotorMix::Bypassed(int32 device_id, int32 channel, switch_state state, bigtime_t time)
{
	if (mStoreState) {
		switch (state) {
			case MM_PRESS:
				break;
			case MM_RELEASE:	
				if (mChannels[channel].mBypass == MM_OFF)
					SetBypass(channel, MM_ON);
				else
					SetBypass(channel, MM_OFF);
				break;
		}
	}
	int16 value = state;
	BMessage msg(MM_BYPASS_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

void
MotorMix::Recorded(int32 device_id, int32 channel, switch_state state, bigtime_t time)
{
	if (mStoreState) {
		switch (state) {
			case MM_PRESS:
				break;
			case MM_RELEASE:	
				if (mChannels[channel].mRecord == MM_OFF)
					SetRecord(channel, MM_ON);
				else
					SetRecord(channel, MM_OFF);
				break;
		}
	}
	int16 value = state;
	BMessage msg(MM_RECORD_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

void 
MotorMix::Soloed(int32 device_id, int32 channel, switch_state state, bigtime_t time)
{
	if (mStoreState) {
		switch (state) {
			case MM_PRESS:
				break;
			case MM_RELEASE:	
				if (mChannels[channel].mSolo == MM_OFF)
					SetSolo(channel, MM_ON);
				else
					SetSolo(channel, MM_OFF);
				break;
		}
	}
	int16 value = state;
	BMessage msg(MM_SOLO_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

void 
MotorMix::Muted(int32 device_id, int32 channel, switch_state state, bigtime_t time)
{
	if (mStoreState) {
		switch (state) {
			case MM_PRESS:
				break;
			case MM_RELEASE:	
				if (mChannels[channel].mMute == MM_OFF)
					SetMute(channel, MM_ON);
				else
					SetMute(channel, MM_OFF);
				break;
		}
	}
	int16 value = state;
	BMessage msg(MM_MUTE_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

void 
MotorMix::FaderMoved(int32 device_id, int32 channel, int16 value, bigtime_t time)
{
	//FUNCTION(("MotorMix FaderMoved channel %ld: %d %Ld\n", channel, value, time));
	if (mStoreState) {
		SetFader(channel, value);
	}
	BMessage msg(MM_FADER_MOVED_MSG);
	msg.AddInt32("channel", channel);
	msg.AddInt16("value", value);
	msg.AddInt64("time", time);
	mMessenger->SendMessage(&msg);
}

//#pragma mark --private--
void 
MotorMix::SetChannelLED(int32 channel, uchar lsb)
{
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ChannelIsVisible(channel)) {
			uchar con = 0x0c;
			uchar msb = (uchar)(channel - dev->CurrentChannel());
			dev->SetControl(con, msb, lsb);
		}
		dev = dev->NextDevice();
	}
}

void 
MotorMix::SetFunctionLED(uchar msb, uchar lsb)
{
}

void 
MotorMix::SetControlFor(int32 device_id, uchar msb, uchar lsb)
{
	MotorMixDevice *dev = mDevices;
	while (dev) {
		if (dev->ID() == device_id) {
			uchar con = 0x0c;
			dev->SetControl(con, msb, lsb);
			return;
		}
		dev = dev->NextDevice();
	}	
}

void 
MotorMix::RecallSettings()
{
}

void 
MotorMix::SaveSettings()
{
}

