#include "MMDevice.h"
#include "MMConsumer.h"
#include "MMProducer.h"
#include <Debug.h>
#include <MidiRoster.h>
#include <string.h>

const size_t CONTROL_SIZE = 5;

MotorMixDevice::MotorMixDevice(const char *name,
								int32 id,
								uchar channel,
								BMidiProducer *input,
								BMidiConsumer *output,
								MotorMix *motormix) :
	mID(id),
	mMidiChannel(channel),
	mCurrentChannel(0),
	mThread(-1),
	mSemaphore(create_sem(0, "MotorMixKeepAlive")),
	mKeepAlive(true),
	mDetected(false),
	mInput(input),
	mOutput(output),
	mProducer(new MotorMixProducer(name, this)),
	mConsumer(new MotorMixConsumer(name, channel, this)),
	mMotorMix(motormix),
	mNext(NULL)
{
	PRINT(("MotorMixDevice %s constructor\n", name));
	mName = strdup(name);
	/* register the producer and consumer */
	if (BMidiRoster::Register(mProducer) < B_OK)
		PRINT(("Failed to register producer\n"));
	if (BMidiRoster::Register(mConsumer) < B_OK)
		PRINT(("Failed to register consumer\n"));
	if (mInput->Connect(mConsumer) < B_OK)
		PRINT(("Failed to connect consumer to input\n"));
	if (mProducer->Connect(mOutput) < B_OK)
		PRINT(("Failed to connect producer to output\n"));
}


MotorMixDevice::~MotorMixDevice()
{
	PRINT(("MotorMixDevice destructor\n"));
	mInput->Disconnect(mConsumer);
	mProducer->Disconnect(mOutput);
	BMidiRoster::Unregister(mProducer);
	BMidiRoster::Unregister(mConsumer);
}

status_t 
MotorMixDevice::InitCheck()
{
	if (!mInput->IsConnected(mConsumer))
		return B_ERROR;
	if (!mProducer->IsConnected(mOutput))
		return B_ERROR;
	return B_OK;
}

bool 
MotorMixDevice::ChannelIsVisible(int32 channel)
{
	if (channel >= mCurrentChannel && channel < (mCurrentChannel + MMCHANNEL_COUNT))
		return true;
	else
		return false;
}

void 
MotorMixDevice::SetControl(uchar control, uchar msb, uchar lsb)
{
	size_t size = CONTROL_SIZE;
	uchar op = 0xB0 + mMidiChannel - 1;
	uchar c2 = control + 0x20;
	uchar output[] = {op, control, msb, c2, lsb};	/* watch CONTROL_SIZE */
	PRINT(("MotorMixDevice::SetControl %02x %02x %02x %02x %02x \n", op, control, msb, c2, lsb));			
	mProducer->SprayData((void *)output, size, false);
}

void 
MotorMixDevice::ControlChange(uchar control, uchar msb, uchar lsb, bigtime_t time)
{
	if (control < 0x8) {
		/* it's a fader moving */
		PRINT(("It's a fader\n"));
		int32 channel = (int32)control + mCurrentChannel;
		int16 value = ((int16)msb << 7) + lsb;
		mMotorMix->FaderMoved(mID, channel, value, time);
	} else if (control == 0xf) {
		/* it's a switch up or down */
		PRINT(("It's a switch\n"));
		switch_state state = ((lsb & MM_PRESS) ? MM_PRESS : MM_RELEASE);
		if (control < 0x8) {
			int32 channel = (int32)msb + CurrentChannel();
			switch (lsb & 0xf) {
				case FADER:
					break;
				case SELECT:
					mMotorMix->Selected(mID, channel, state, time);
					break;
				case MUTE:
					mMotorMix->Muted(mID, channel, state, time);
					break;
				case SOLO:
					mMotorMix->Soloed(mID, channel, state, time);
					break;
				case BYPASS:
					mMotorMix->Bypassed(mID, channel, state, time);
					break;
				case RECORD:
					mMotorMix->Recorded(mID, channel, state, time);
					break;
				default:
					break;
			}
		}	
	} else {
		PRINT(("It's a rotary knob\n"));
	}
}

void 
MotorMixDevice::Connected()
{
	PRINT(("MotorMixDevice::Connected\n"));
	/* spawn keep alive thread to ping motormix */
	mThread = spawn_thread(MotorMixDevice::KeepAliveEntry, "MotorMixKeepAlive", B_NORMAL_PRIORITY, this);
	resume_thread(mThread);
}

void 
MotorMixDevice::Disconnected()
{
	PRINT(("MotorMixDevice::Disconnected\n"));
	status_t err;
	mKeepAlive = false;
	wait_for_thread(mThread, &err);
}

void 
MotorMixDevice::PingResponse()
{
	if (!mDetected) {
		PRINT(("Connection to MotorMix : %s detected\n", mName));
		mDetected = true;
		/* notify MotorMix */
		release_sem(mSemaphore);
	}
}

void 
MotorMixDevice::NoResponse()
{
	PRINT(("Couldn't connect to MotorMix : %s\n", mName));
	/* notify MotorMix */
	mDetected = false;
}

void 
MotorMixDevice::KeepAlive()
{
	status_t err;
	uchar ping[] = { 0x90 + mMidiChannel-1, 0x00, 0x00 };
	PRINT(("ping data : %02x %02x %02x\n", ping[0], ping[1], ping[2]));
	while (mKeepAlive) {
		/* ping motormix */	
		mProducer->SprayData((void *)ping, 3, true);
		/* wait for reply */
		if (!mDetected) {
			err = acquire_sem_etc(mSemaphore, 1, B_TIMEOUT, 6*1000*1000);	/* six seconds */
			if (err == B_TIMED_OUT)
				NoResponse();
			else if (err < B_OK)
				return;
		}
		snooze(500*1000);	/* 500 msec */
	}
}

status_t 
MotorMixDevice::KeepAliveEntry(void *dev)
{
	((MotorMixDevice *)dev)->KeepAlive();
	return B_OK;
}

