////////////////////////////////////////////////////////////////////////////////
// Whistler.cpp
// ------------
// Implements the Whistler class.
//
// Copyright 1999, Be Incorporated. All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include "Whistler.h"
#include <stdio.h>
#include <InterfaceDefs.h>
#define PITCH_RANGE 48
#define PULSE 50000
#define TONIC 48

#define NOTE1 (TONIC+24)
#define NOTE2A (TONIC+39)
#define NOTE2B (TONIC+40)
#define NOTE3 (TONIC+30)
#define NOTE4A (TONIC+45)
#define NOTE4B (TONIC+46)

#define BNOTE1 TONIC
#define BNOTE2 (TONIC-5)

#define BASS2_OFFSET (-12)

#define MOUSE_DOWN_NOTE 38
#define MOUSE_UP_NOTE 45

int32 Whistler::_run(void* arg)
{
	((Whistler *)arg)->run();
	return B_OK;
}

Whistler::Whistler()
{
	whistle_thread = spawn_thread(_run, 
		"whistle thread",
		B_NORMAL_PRIORITY, this);
		
	whistle_port = create_port(10, "whistle port");

	ReckonScreen();

	transposition = 0;

	for (int32 ktr = 0; ktr < NOTE_COUNT; notes[ktr++] = -1);
	chans[KEYBOARD] = KEYBOARD_CHAN;
	chans[MOUSE_MOVE_BASS_1] = MOUSE_MOVE_BASS_1_CHAN;
	chans[MOUSE_MOVE_BASS_2] = MOUSE_MOVE_BASS_2_CHAN;
	chans[MOUSE_MOVE_COMP_1] = MOUSE_MOVE_COMP_CHAN;
	chans[MOUSE_MOVE_COMP_2] = MOUSE_MOVE_COMP_CHAN;
	chans[MOUSE_MOVE_COMP_3] = MOUSE_MOVE_COMP_CHAN;
	chans[MOUSE_MOVE_COMP_4] = MOUSE_MOVE_COMP_CHAN;
	chans[MOUSE_DOWN] = MOUSE_BUTTONS_CHAN;
	chans[MOUSE_UP] = MOUSE_BUTTONS_CHAN;
	enabled = true;
}

void Whistler::Go()
{
	prep_synth();
	resume_thread(whistle_thread);
}

Whistler::~Whistler()
{
	Enable(false);
	snooze(1000000);
	synth->AllNotesOff(false);
	delete_port(whistle_port);
	kill_thread(whistle_thread);
};

void Whistler::SetMouseVol(float value)
{
	set_vol(value, MOUSE_VOL);
}

void Whistler::SetKeyboardVol(float value)
{
	set_vol(value, KEYBOARD_VOL);
}

void Whistler::SetButtonsVol(float value)
{
	value = max_c(min_c(1.0, value), 0.0);
	set_vol(value, BUTTONS_VOL);
}

void Whistler::set_vol(float value, int32 index)
{
	vols[index] = (int32)(127*value);
	if (vols[index] < 5)
		mutes[index] = true;
	else
		mutes[index] = false;
}


void Whistler::run(void)
{
	int32 port_code;
	status_t err;
	input_info ii;

	while (1) {
		if (!enabled) {
			snooze(1000000);
			continue;
		}
		if ((err=read_port_etc(whistle_port, 
							&port_code, 
							&ii, 
							sizeof(input_info),
							B_TIMEOUT,
							(PULSE*4))) >= 0) {
			Play(port_code, &ii);
		}
		else {
			switch(err) {
			case B_INTERRUPTED:
				continue;
		
			case B_TIMED_OUT:
				NoteOff(MOUSE_MOVE_BASS_1);
				NoteOff(MOUSE_MOVE_BASS_2);
				NoteOff(MOUSE_MOVE_COMP_1);
				NoteOff(MOUSE_MOVE_COMP_2);
				NoteOff(MOUSE_MOVE_COMP_3);
				NoteOff(MOUSE_MOVE_COMP_4);
				break;
		
			default:
				return;
			}
		}
	}
}
	
void Whistler::ReckonScreen(void)
{
	BRect screen = BScreen().Frame();
	trans_width = screen.bottom/PITCH_RANGE;
	pan_width = screen.right/127.0;
}

void Whistler::prep_synth()
{
	synth = new BMidiSynth();
	synth->EnableInput(true, true);
	be_synth->SetReverb(B_REVERB_CLOSET);
	ToggleMouseInst(false);
	synth->ProgramChange(MOUSE_BUTTONS_CHAN, B_TIMPANI);
}

void Whistler::Enable(bool tf)
{
	enabled = tf;
}

void Whistler::ToggleMouseInst(bool isDown)
{
	if (isDown) {
		synth->ProgramChange(MOUSE_MOVE_BASS_1_CHAN, B_XYLOPHONE);
		synth->ProgramChange(MOUSE_MOVE_BASS_2_CHAN, B_XYLOPHONE);
		synth->ProgramChange(MOUSE_MOVE_COMP_CHAN, B_XYLOPHONE);
	}
	else {
		synth->ProgramChange(MOUSE_MOVE_BASS_1_CHAN, B_ACOUSTIC_GRAND);
		synth->ProgramChange(MOUSE_MOVE_BASS_2_CHAN, B_ACOUSTIC_GRAND);
		synth->ProgramChange(MOUSE_MOVE_COMP_CHAN, B_ACOUSTIC_GRAND);
	}
}

void Whistler::SetPan(int32 pan)
{
	synth->ControlChange(MOUSE_MOVE_BASS_1_CHAN, B_PAN, pan);
	synth->ControlChange(MOUSE_MOVE_BASS_2_CHAN, B_PAN, pan);
	synth->ControlChange(MOUSE_MOVE_COMP_CHAN, B_PAN, pan);
	synth->ControlChange(MOUSE_BUTTONS_CHAN, B_PAN, pan);
}

void Whistler::Play(int32 code, input_info *ii)
{
	switch (code) {
	case B_MOUSE_MOVED :
		SetPan((int)(127-(ii->x/pan_width)));
		transposition = (int)((PITCH_RANGE>>1)-(ii->y/trans_width));
		if (!mutes[MOUSE_VOL])
			PlayMouseMusic();
		break;

	case B_MOUSE_DOWN :
		SetPan((int)(127-(ii->x/pan_width)));
		if (!mutes[BUTTONS_VOL]) {
			ToggleMouseInst(true);
			NoteOn(MOUSE_DOWN, 
				MOUSE_DOWN_NOTE+(ii->item*3)+transposition/4, 
				vols[BUTTONS_VOL]);
		}
		break;

	case B_MOUSE_UP :
		SetPan((int)(127-(ii->x/pan_width)));
		if (!mutes[BUTTONS_VOL]) {
			ToggleMouseInst(false);
			NoteOn(MOUSE_UP, 
				MOUSE_UP_NOTE+(ii->item*3)+transposition/4, 
				vols[BUTTONS_VOL]);
		}
		break;

	case B_KEY_DOWN :
		if (!mutes[KEYBOARD_VOL])
			NoteOn(KEYBOARD, ((ii->item)%60)+27, vols[KEYBOARD_VOL]);
		break;
	}


}	

void Whistler::MessageReceived(BMessage *msg)
{
	const char *c;
	input_info ii;
	BPoint point;
	int32 buttons;
	
	switch (msg->what) {
	case B_WORKSPACE_ACTIVATED:
		ReckonScreen();
		break;

	case B_KEY_DOWN :
		msg->FindString("bytes", &c);
		ii.item = (int32)c[0];
		break;
		
	case B_MOUSE_MOVED:
		msg->FindPoint("where", &point);
		ii.x = point.x;
		ii.y = point.y;
		break;

	case B_MOUSE_DOWN:
		msg->FindInt32("buttons", &buttons);
		ii.item = button_num = buttons;
		msg->FindPoint("where", &point);
		ii.x = point.x;
		ii.y = point.y;
		break;
		
	case B_MOUSE_UP:
		ii.item = button_num;
		msg->FindPoint("where", &point);
		ii.x = point.x;
		ii.y = point.y;
		break;
		
	default:
		return (BHandler::MessageReceived(msg));
	}
	
	write_port(whistle_port, msg->what, 
		(void *)&ii, sizeof(input_info));
}

void Whistler::NoteOn(int32 which, int32 note, int32 vol)
{
	synth->NoteOn(chans[which], notes[which] = note, vol);
}

void Whistler::NoteOff(int32 which)
{
	if (notes[which] != -1) {
		synth->NoteOff(chans[which], notes[which], 0);
	}
	notes[which] = -1;
}

void Whistler::PlayMouseMusic()
{
	static bigtime_t last_time = 0;
	static int32 note_ktr = 0;
	bigtime_t this_time = system_time();
	
	if (this_time - last_time < PULSE)
		 return;
		
	if (this_time - last_time > PULSE*2) 
		note_ktr = 0;

	switch (note_ktr++ % 4) {
	case 0:
		NoteOff(MOUSE_MOVE_BASS_1);
		NoteOff(MOUSE_MOVE_BASS_2);
		snooze(1000);
		
		NoteOn(MOUSE_MOVE_BASS_1, BNOTE1+transposition, vols[MOUSE_VOL]);
		NoteOn(MOUSE_MOVE_BASS_2, BNOTE1+BASS2_OFFSET+transposition, 
			vols[MOUSE_VOL]);
		break;

	case 1:
		NoteOff(MOUSE_MOVE_COMP_1);
		NoteOff(MOUSE_MOVE_COMP_2);
		NoteOff(MOUSE_MOVE_COMP_3);
		NoteOff(MOUSE_MOVE_COMP_4);
		snooze(1000);
		NoteOn(MOUSE_MOVE_COMP_1, NOTE1+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_2, NOTE2A+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_3, NOTE3+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_4, NOTE4A+transposition, vols[MOUSE_VOL]);
		break;

	case 2:
		NoteOff(MOUSE_MOVE_BASS_1);
		NoteOff(MOUSE_MOVE_BASS_2);
		snooze(1000);
		NoteOn(MOUSE_MOVE_BASS_1, BNOTE2+transposition, vols[MOUSE_VOL]);
		NoteOn(MOUSE_MOVE_BASS_2, BNOTE2+BASS2_OFFSET+transposition, 
			vols[MOUSE_VOL]);
		break;
	case 3:
		NoteOff(MOUSE_MOVE_COMP_1);
		NoteOff(MOUSE_MOVE_COMP_2);
		NoteOff(MOUSE_MOVE_COMP_3);
		NoteOff(MOUSE_MOVE_COMP_4);
		snooze(1000);
		NoteOn(MOUSE_MOVE_COMP_1, NOTE1+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_2, NOTE2B+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_3, NOTE3+transposition, vols[MOUSE_VOL]);
		snooze(10000);
		NoteOn(MOUSE_MOVE_COMP_4, NOTE4B+transposition, vols[MOUSE_VOL]);
		break;
	};
	last_time = this_time;
	return;
}
