////////////////////////////////////////////////////////////////////////////////
// PigTail.cpp
// -----------
// Implements the PigTail slider control.
//
// Copyright 1999, Be Incorporated. All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include "PigTail.h"
#include <stdlib.h>
#include <OS.h>
#include <Application.h>

static void shift(int32 *c, bool *up);

PigTail::PigTail(BRect frame, int32 i) :
	BView(frame, "", B_FOLLOW_BOTTOM|B_FOLLOW_TOP, B_WILL_DRAW)
{
	index = i;

	srand(system_time());
	shape = new BShape();
	breaks[0][0] = -1;
	picture = 0x0;
	height = (int32)(Bounds().bottom - Bounds().top);
	bubble = (int32)(height * .35);
	r = rand()%255;
	g = rand()%255;
	b = rand()%255;
	r_up = true;
	g_up = false;
	b_up = true;
	
	SetViewColor(B_TRANSPARENT_COLOR);
	tracking = false;
	key_down = false;
}
	

void PigTail::AttachedToWindow()
{
	NewShape();
	NewColors();
}

void PigTail::NewShape()
{
	if (picture)
		delete picture;
	picture = new BPicture();

	shape->Clear();
	if (breaks[0][0] == -1)
		make_shape(shape);
	else
		bend_shape(shape);
	BeginPicture(picture);
	FillShape(shape, B_SOLID_HIGH);
	picture = EndPicture();
}	

#define MAX_THROW 19.0
#define MIN_THROW 3.0
#define BUBBLE_HEIGHT 2.0
#define RUN 0
#define LEFT 1
#define RIGHT 2

void PigTail::make_shape(BShape *shape)
{
	int32 ktr;
	float total = 0;
	float width = Bounds().right - MAX_THROW;

	for (ktr = 0; ktr < 11; ktr++) {
		if (ktr == 0) 
			breaks[0][RUN] = 0.0;
		else {
			total += (rand()%100);
			breaks[ktr][RUN] = total;
		}
		breaks[ktr][LEFT] = (rand()%1000) / 1000.0;
		breaks[ktr][RIGHT] = (rand()%1000) / 1000.0;
	}
		
	for (ktr = 0; ktr < 11; ktr++) {
		if (ktr > 0) {
			breaks[ktr][RUN] /= total ;
			breaks[ktr][RUN] *= height;
		}
			
		breaks[ktr][LEFT] = breaks[ktr][LEFT] * width;
		breaks[ktr][RIGHT] = breaks[ktr][LEFT] +
			max_c((breaks[ktr][RIGHT]*MAX_THROW), MIN_THROW);
	}
	breaks[10][RUN] = height;
		
	shape->MoveTo(BPoint(breaks[0][LEFT], breaks[0][RUN]));
	for (ktr = 1; ktr < 11; ktr++) {
		shape->LineTo(BPoint(breaks[ktr][LEFT], breaks[ktr][RUN]));
	}
	
	for (ktr = 9; ktr >= 1; ktr--) {
		shape->LineTo(BPoint(breaks[ktr][RIGHT], breaks[ktr][RUN]));
	}

		
	shape->Close();
}

#define MAX_BEND 6
void PigTail::bend_shape(BShape *shape)
{
	int32 ktr;
	float width = Bounds().right - MAX_THROW;

	for (ktr = 0; ktr < 11; ktr++) {
		if (ktr == 0) 
			breaks[0][RUN] = 0.0;
		else {
			breaks[ktr][RUN] = min_c(max_c(breaks[ktr-1][RUN]+(4*ktr),
				 breaks[ktr][RUN]+(rand()%MAX_BEND)-(rand()%MAX_BEND+1)),
				 height-(4*(11-ktr)));
			
		}
		breaks[ktr][LEFT] = min_c(max_c(0,
			breaks[ktr][LEFT]+(rand()%MAX_BEND)-(rand()%MAX_BEND)),
			width);
		breaks[ktr][RIGHT] = 
			max_c(min_c(breaks[ktr][RIGHT]+(rand()%MAX_BEND)-(rand()%MAX_BEND),
				breaks[ktr][LEFT]+MAX_THROW), breaks[ktr][LEFT]+MIN_THROW);
	}
		
	breaks[10][RUN] = height;
		
	shape->MoveTo(BPoint(breaks[0][LEFT], breaks[0][RUN]));
	for (ktr = 1; ktr < 11; ktr++) {
		shape->LineTo(BPoint(breaks[ktr][LEFT], breaks[ktr][RUN]));
	}
	
	for (ktr = 9; ktr >= 1; ktr--) {
		shape->LineTo(BPoint(breaks[ktr][RIGHT], breaks[ktr][RUN]));
	}

	shape->Close();
}

#define MAX_SHIFT 10
void shift(int32 *c, bool *up)
{
	if (*up) {
		*c += rand() % MAX_SHIFT;
		if (*c > 255) {
			*c = 255;
			*up = false;
		}
	}
	else {
		*c -= rand() % MAX_SHIFT;
		if (*c < 0) {
			*c = 0;
			*up = true;
		}
	}
}

void PigTail::NewColors()
{
	guage_hi.red = 255;
	guage_hi.green = 185+rand()%65;
	guage_hi.blue = rand()%40;
	guage_hi.alpha = 0;

	guage_lo.red = 215+rand()%35;
	guage_lo.green = 0;
	guage_lo.blue = 125+rand()%25;
	guage_lo.alpha = 0;

	guage_lo.red = 215+rand()%35;
	guage_lo.green = 0;
	guage_lo.blue = 125+rand()%25;
	guage_lo.alpha = 0;

	shift(&r, &r_up);
	shift(&g, &g_up);
	shift(&b, &b_up);
}
void PigTail::Draw(BRect update)
{
	if (tracking || key_down) {
		NewShape();
		NewColors();
		key_down = false;
	}
	BRect top=Bounds(), bottom, guage;
	bottom = top;
	top.bottom = bubble;
	bottom.top = bubble;
	
	SetHighColor(guage_hi);
	SetLowColor(guage_lo);
	ClipToPicture(picture, BPoint(0,0));
	FillRect(top, B_SOLID_HIGH);
	FillRect(bottom, B_SOLID_LOW);

	guage = top;
	guage.top = bubble-BUBBLE_HEIGHT;
	guage.bottom = bubble+BUBBLE_HEIGHT;

	SetHighColor(0, 0, 0);
	FillRect(guage, B_SOLID_HIGH);
	
	SetHighColor(r,g,b);
	ClipToInversePicture(picture, BPoint(0,0));
	FillRect(Bounds(), B_SOLID_HIGH);
}

void PigTail::SetBubble(float level, bool inc)
{
	float current;
	if (inc)  {
		current = (float)(height-bubble)/(float)height;
		SetBubble((int32)(height * (1.0 - (current+level))));
	}
	else
		SetBubble((int32)(height * (1.0 - level)));
}

#include <stdio.h>
void PigTail::SetBubble(int32 coord)
{
	bubble = coord;
	bubble = max_c(0, min_c(bubble, height));
	Invalidate();
}

void PigTail::MouseDown(BPoint where)
{
	tracking = true;
	SetMouseEventMask(B_POINTER_EVENTS, 
		B_NO_POINTER_HISTORY|B_LOCK_WINDOW_FOCUS);
	Window()->Activate(true);
	MakeFocus(true);
	SetBubble((int32)where.y);
	send_pig_message();
}


void PigTail::MouseMoved(BPoint where, uint32 code, const BMessage*)
{
	if (tracking) {
		SetBubble((int32)where.y);
		send_pig_message();
	}
}

void PigTail::MouseUp(BPoint where)
{
	tracking = false;
}

void PigTail::KeyDown(const char *bytes, int32 numBytes)
{
	BMessage msg(PIG_FOCUS_MSG);
	
	switch (bytes[0]) {
	case B_UP_ARROW:
		key_down = true;
		SetBubble(.05, true);
		send_pig_message();
		break;
		
	case B_DOWN_ARROW:
		key_down = true;
		SetBubble(-.05, true);
		send_pig_message();
		break;

	case B_LEFT_ARROW:
		msg.AddBool("up", false);
		Window()->PostMessage(&msg);
		break;

	case B_TAB:
	case B_RIGHT_ARROW:
		msg.AddBool("up", true);
		Window()->PostMessage(&msg);
		break;
	default:
		break;
	}
}

void PigTail::send_pig_message()
{
	BMessage msg(PIG_MSG);
	msg.AddFloat("value", 1.0 - ((float)bubble / (float)height));
	msg.AddInt32("index", index);
	be_app->PostMessage(&msg);
}

