/*******************************************************************************
/
/	File:			rtm_test.cpp
/
/   Description:	A simple test of the Media Kit real-time allocation
/                   facility.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <MediaDefs.h>

// If this is 0, regular allocation will be used.
// Otherwise, real-time allocation will be turned on.
#define USE_RTM_ALLOC 0

#if USE_RTM_ALLOC
// This is where all that rtm_alloc stuff comes from!
#include <RealtimeAlloc.h>
#endif

#define MAX_CH 100

// This is a dummy class that might look like something you'd use
// to maintain channels of information in a media node.
class Channel
{
public:
	media_format m_format;
	
#if USE_RTM_ALLOC
private:
	// RTMPoolManager simply creates and deletes a pool when it is
	// constructed and destroyed.
	struct RTMPoolManager {
		const static size_t poolSize = MAX_CH*400; // plenty o' stuffing!
		rtm_pool* pool;
		status_t init;
		
		RTMPoolManager()
		{ init = rtm_create_pool(&pool, poolSize); }
		~RTMPoolManager()
		{ rtm_delete_pool(pool); }
	};
	
	// We define a static RTMPoolManager object, which will be constructed
	// when the class's code is loaded, and destroyed when the class's
	// code is unloaded. This means that the pool will be available for
	// all instances of this class.
	static RTMPoolManager m_poolManager;
	
public:
	// The real-time allocatin' versions of operator new and delete.
	// Note that they throw bad_alloc if something goes screwy.
	void* operator new (size_t size) throw (bad_alloc)
	{
		if (m_poolManager.init != B_OK) throw bad_alloc();
		void* ret = rtm_alloc(m_poolManager.pool, size);
		if (! ret) throw bad_alloc();
		return ret;
	}
	
	void* operator new[] (size_t size) throw (bad_alloc)
	{
		if (m_poolManager.init != B_OK) throw bad_alloc();
		void* ret = rtm_alloc(m_poolManager.pool, size);
		if (! ret) throw bad_alloc();
		return ret;
	}
	
	void operator delete (void* ptr) throw ()
	{
		rtm_free(ptr);
	}
	
	void operator delete[] (void* ptr) throw ()
	{
		rtm_free(ptr);
	}
#endif
};

#if USE_RTM_ALLOC
// C++ sez we have to define the static variable!
Channel::RTMPoolManager Channel::m_poolManager;
#endif

// A simple stop watch -- like BStopWatch except that it does
// lap minima/maxima and averaging instead.
class MyStopWatch
{
public:
	MyStopWatch(const char* name)
		: m_minLap(-1), m_maxLap(-1), m_numLaps(0)
	{ m_begin = system_time(); m_end = m_begin; m_name = strdup(name); }
	
	~MyStopWatch()
	{
		m_end = system_time();
		bigtime_t tot = m_end - m_begin;
		printf("StopWatch %s: tot=%Ld, laps=%Ld, min_lap=%Ld, "
			"max_lap=%Ld, avg_lap=%Ld\n", m_name, tot, m_numLaps,
			m_minLap, m_maxLap, tot / m_numLaps);
		free(m_name);
	}
		 
	bigtime_t Lap()
	{
		m_numLaps++;
		bigtime_t cur = system_time();
		bigtime_t lap = cur - m_end;
		if (m_minLap == -1) m_minLap = lap;
		else if (lap < m_minLap) m_minLap = lap;
		if (m_maxLap == -1) m_maxLap = lap;
		else if (lap > m_maxLap) m_maxLap = lap;	
		m_end = cur;
		return cur;
	}

private:	 
	bigtime_t m_begin;
	bigtime_t m_end;
	bigtime_t m_minLap;
	bigtime_t m_maxLap;
	uint64 m_numLaps;
	char* m_name;
};

int main()
{
#if 1
	// Test 1: Create lots of small objects separately and delete them.
	const int32 kMaxIter = 1000;
	Channel *ch[MAX_CH];
	
	int i, j;
	MyStopWatch watch("rtm test");
	for (i=0; i<kMaxIter; i++) {
		for (j=0; j<MAX_CH; j++) {
			ch[j] = new Channel;
		}
		for (j=0; j<MAX_CH; j++) {
			delete ch[j];
		}
		watch.Lap();
	}
#else
	// Test 2: Create a big array of objects and delete it.
	const int32 kMaxIter = 100;
	Channel* ch;
	int i, j;
	MyStopWatch watch("rtm test");
	for (i=0; i<kMaxIter; i++) {
		for (j=0; j<kMaxIter; j++) {
			ch = new Channel[MAX_CH];
			delete [] ch;
		}
		watch.Lap();
	}
#endif
	return 0;
}
