//******************************************************************************
//
//	File: 			ticker.cpp
//
//	Description: 	ticker object class.
//
//	Written by:		Douglas Wright
//
//	Copyright 1998, Be Incorporated
//
//****************************************************************************

#include "ticker.h"
#include "tick.h"

/*------------------------------------------------------------*/

ticker::ticker(const char *name, uint32 frequency, long aPriority)
	: thread(name, aPriority)
{
	printf("ticker::ticker - frequency: %d\n", frequency);
	fFrequency = frequency;
	fPeriod = 1000000.0 / frequency;
	fIsStopping = false;
  	for (long i = 0; i < MAX_CONNECTIONS; i++) {
		fConnectionList[i] = 0;
  	}
}


/*------------------------------------------------------------*/

ticker::ticker(bigtime_t period, const char *name, long aPriority)
	: thread(name, aPriority)
{
	printf("ticker::ticker - period: %d\n", period);
	fPeriod = period;
	fFrequency = 1000000.0 / period;
  	for (long i = 0; i < MAX_CONNECTIONS; i++) {
		fConnectionList[i] = 0;
  	}
}

/*------------------------------------------------------------*/

uint32	
ticker::Frequency()
{
	return fFrequency;
}
	
/*------------------------------------------------------------*/

void	
ticker::SetFrequency(uint32 frequency)
{
	fFrequency = frequency;
	fPeriod = 1000000.0 / frequency;
}
	
/*------------------------------------------------------------*/

bigtime_t	
ticker::Period()
{
	return fPeriod;
}
	
/*------------------------------------------------------------*/

void	
ticker::SetPeriod(bigtime_t period)
{
	fPeriod = period;
	fFrequency = 1000000.0 / period;
}
	
/*------------------------------------------------------------*/

status_t
ticker::Start()
{
	fIsStopping = false;
	return Run();
}

/*------------------------------------------------------------*/

void
ticker::Stop()
{
	fIsStopping = true;
	Suspend();
}

/*------------------------------------------------------------*/

bool 
ticker::IsStopping() const
{
	return fIsStopping;
}

/*------------------------------------------------------------*/

status_t	
ticker::Connect(port_id id)
{
  if (!IsConnected(id)){
    for (long i = 0; i < MAX_CONNECTIONS; i++) {
	  if (!fConnectionList[i]) {
	    fConnectionList[i] = id;
	    return B_OK;
	  }
	  return B_ERROR;	//todo - be more descriptive here
    }
  }
  return B_OK;
}

/*------------------------------------------------------------*/

void	
ticker::Disconnect(port_id id)
{
  for (long i = 0; i < MAX_CONNECTIONS; i++) {
	if (fConnectionList[i] == id) {
	  fConnectionList[i] = 0;
	}
  }
}

/*------------------------------------------------------------*/

bool		
ticker::IsConnected(port_id id) const
{ 
  for (long i = 0; i < MAX_CONNECTIONS; i++) {
	if (fConnectionList[i] == id) {
	  return true;
	}
  }
  return false;
}

/*------------------------------------------------------------*/

void	
ticker::OutputTick(bigtime_t time, bigtime_t period, uint64 count)
{
	tick	t;
	
	t.fTime = time;
	t.fPeriod = period;
	t.fNumber = count;
  	for (long i = 0; i < MAX_CONNECTIONS; i++) {
		if (fConnectionList[i]) {
	  		write_port(fConnectionList[i], TICK_MSG, &t, sizeof(tick));
		}
  	}
}

/*------------------------------------------------------------*/

status_t
ticker::ThreadMain()
{
	bigtime_t	start  = system_time();
	bigtime_t 	current = start;
	bigtime_t	last;
	bigtime_t	period = fPeriod;

	int64		target = period;
	int64		snoot = 1500;		//snooze margin
	int64		drift = 0;
	int64		jitter = 0;
	int64		maxjitter = 0;

	uint64		count = 0;

	uint32		interval = 0;		//for analysis
	uint32		loopcounter = 0;	//for analysis

	while(!fIsStopping){
		//send the tick
		OutputTick(current, period, count);
		++count;
		last = current;
		
		//snooze til a little bit before next tick
		snooze(target-snoot);		//eventually use snooze_until()
		current = system_time();
		
		//wait until you're really there
		while((current - last) < target){
			current = system_time();
			++loopcounter;
		}
		
		//drift compensation
		jitter = current - last - period;
		drift += jitter;
		target = period - (drift >> 2);	//shift instead of divide for speed

		//refresh the period in case it changes
		period = fPeriod;					
		
		//ANALYSIS - test for maxjitter - comment out for performance
		if(abs(jitter) > abs(maxjitter)) maxjitter = jitter;
		//~ANALYSIS
		
		//do something every 60 ticks
		if(count > interval){
			//ANALYSIS - display stats
			printf("drift = %Ld, avg period = %Ld, loopcount = %Ld, maxjitter = %Ld\n", drift, (current-start)/count, loopcounter/count, maxjitter);
			maxjitter = 0;
			//~ANALYSIS
			interval = count + 60;
		}
	}
	return B_OK;
}

/*------------------------------------------------------------*/
