//	SimpleRecord.cpp
//	----------------
//	Copyright 1999, Be Incorporated.   All Rights Reserved.
//	This file may be used under the terms of the Be Sample Code License.

#include <Application.h>
#include <Button.h>
#include <FilePanel.h>
#include "MediaRecorder.h"
#include "BlockFIFO.h"
#include <Bitmap.h>
#include <Debug.h>
#include <Message.h>
#include <Messenger.h>
#include <Directory.h>
#include <Entry.h>
#include <MediaFile.h>
#include <MediaTrack.h>
#include <Alert.h>
#include <MediaNode.h>
#include <MediaRoster.h>
#include <MediaAddOn.h>
#include <File.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "SimpleRecord.h"


#define SET_FILE 'file'
#define RECORD 'reco'
#define STOP 'stop'
#define DEVICE 'devi'


PeakView::PeakView(BRect area) :
	BView(area, "peak", B_FOLLOW_NONE, B_WILL_DRAW | B_PULSE_NEEDED),
	mLock("PeakViewLock")
{
	mBits = new BBitmap(BRect(0,0,256,18), B_CMAP8);
	mCurMaxL = 0;
	mAvgMaxL = 0;
	mCurMaxR = 0;
	mAvgMaxR = 0;
	SetViewColor(216, 216, 216);
	Generate();
}

PeakView::~PeakView()
{
	delete mBits;
}

void
PeakView::Draw(BRect area)
{
	DrawBitmapAsync(mBits, B_ORIGIN);
	DrawString("0", BPoint(252, 30));
	DrawString("-6", BPoint(122, 30));
	DrawString("-12", BPoint(58, 30));
}

void
PeakView::Pulse()
{
	mLock.Lock();
	mAvgMaxL = 0.9*mAvgMaxL;
	if (mCurMaxL > mAvgMaxL) mAvgMaxL = mCurMaxL;
	mAvgMaxR = 0.9*mAvgMaxR;
	if (mCurMaxR > mAvgMaxR) mAvgMaxR = mCurMaxR;
	Generate();
	mCurMaxL = 0.0;
	mCurMaxR = 0.0;
	mLock.Unlock();
	DrawBitmapAsync(mBits, B_ORIGIN);
	Flush();
}

void
PeakView::Generate()
{
	union {
		uchar uc[4];
		uint32 ui;
	} z, w, p, r;
	z.uc[0] = BGCOLOR;
	z.uc[1] = SEGCOLOR;
	z.uc[2] = SEGCOLOR;
	z.uc[3] = SEGCOLOR;
	w.uc[0] = BGCOLOR;
	w.uc[1] = LITCOLOR;
	w.uc[2] = LITCOLOR;
	w.uc[3] = LITCOLOR;
	p.uc[0] = BGCOLOR;
	p.uc[1] = PEAKCOLOR;
	p.uc[2] = PEAKCOLOR;
	p.uc[3] = PEAKCOLOR;
	r.uc[0] = BGCOLOR;
	r.uc[1] = CLIPCOLOR;
	r.uc[2] = CLIPCOLOR;
	r.uc[3] = CLIPCOLOR;
	uint32 rb = mBits->BytesPerRow()/4;
	uint32 * b = (uint32 *)mBits->Bits();

	int limit = (int)(mCurMaxL*64);
	memset(b, BGCOLOR, rb*4);
	b += rb;
	for (int ix=0; ix<63; ix++) {
		uint32 * i = b+ix;
		if (ix < limit) {
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui;
		}
		else {
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix));
			break;
		}
	}
	if (mCurMaxL >= 0.99) {
		uint32 * i = b+63;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
	}
	else {
		uint32 * i = b+(int)(64*mAvgMaxL);
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
	}
	b += 8*rb;
	memset(b, BGCOLOR, rb*4);
	b += rb;

	limit = (int)(mCurMaxR*64);
	for (int ix=0; ix<63; ix++) {
		uint32 * i = b+ix;
		if (ix < limit) {
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui; i += rb;
			*i = w.ui;
		}
		else {
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix)); i += rb;
			memset(i, BGCOLOR, 4*(rb-ix));
			break;
		}
	}
	if (mCurMaxR >= 0.99) {
		uint32 * i = b+63;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
		*i = r.ui; i += rb;
	}
	else {
		uint32 * i = b+(int)(64*mAvgMaxR);
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
		*i = p.ui; i += rb;
	}
	b += 8*rb;
	memset(b, BGCOLOR, rb*4);
}


RecWindow::RecWindow() :
	BWindow(BRect(100,100,383+92,200), "Recorder", B_TITLED_WINDOW,
			B_NOT_ZOOMABLE | B_NOT_RESIZABLE),
	mRecorder("SimpleRecord"),
	mFIFO(65536, 1, 3),
	mFilePanel(B_SAVE_PANEL, new BMessenger(this), 0, 0, false,
		new BMessage(B_REFS_RECEIVED), 0, false, true),
	mDevice(0),
	mDevWin(0)
{
	mFile = 0;
	mTrack = 0;
	mRecording = false;
	mFrameSize = 0;
	mThread = -1;
	mPeakFormat = 0;
	BView * bg = new BView(Bounds(), "bg", B_FOLLOW_ALL, B_WILL_DRAW);
	bg->SetViewColor(216, 216, 216);
	bg->SetLowColor(216, 216, 216);
	BRect r(bg->Bounds());
	r.InsetBy(10, 10);
	r.top = r.bottom-21;
	r.right = r.left+80;
	mFileButton = new BButton(r, "file", "File", new BMessage(SET_FILE));
	bg->AddChild(mFileButton);
	r.OffsetBy(r.Width()+12, 0);
	mRecordButton = new BButton(r, "record", "Record", new BMessage(RECORD));
	bg->AddChild(mRecordButton);
	r.OffsetBy(r.Width()+12, 0);
	mStopButton = new BButton(r, "stop", "Stop", new BMessage(STOP));
	bg->AddChild(mStopButton);
	r.OffsetBy(r.Width()+12, 0);
	mDeviceButton = new BButton(r, "device", "Device", new BMessage(DEVICE));
	bg->AddChild(mDeviceButton);
	r.bottom = r.top-12;
	r.top = 10;
	r.left = 10;
	r.right = r.left+256;
	mPeakView = new PeakView(r);
	bg->AddChild(mPeakView);
	AddChild(bg);
	SetPulseRate(50000);
	mRecorder.SetHook(this, RecordHook);
	SetButtons();
}

RecWindow::~RecWindow()
{
	if (mRecording) {
		StopRecording();
	}
	CloseFile();
	if (mDevice != 0) {
		BMediaRoster::CurrentRoster()->ReleaseNode(*mDevice);
		delete mDevice;
	}
	if (mDevWin->Lock()) {
		mDevWin->Quit();
	}
}

bool
RecWindow::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void
RecWindow::MessageReceived(
	BMessage * message)
{
	switch (message->what) {
	case SET_FILE:
		SelectFile();
		break;
	case B_REFS_RECEIVED:
		GotFile(message);
		break;
	case RECORD:
		StartRecording();
		break;
	case STOP:
		StopRecording();
		break;
	case DEVICE:
		SelectDevice(message);
		break;
	default:
		BWindow::MessageReceived(message);
		break;
	}
	SetButtons();
}

void 
RecWindow::SetButtons()
{
	if (mRecording) {
		mFileButton->SetEnabled(false);
		mRecordButton->SetEnabled(false);
		mStopButton->SetEnabled(true);
		mDeviceButton->SetEnabled(false);
	}
	else {
		mFileButton->SetEnabled(true);
		mRecordButton->SetEnabled(mFile != 0);
		mStopButton->SetEnabled(false);
		mDeviceButton->SetEnabled(true);
	}
}

void 
RecWindow::SelectFile()
{
	if (!mFilePanel.IsShowing()) {
		mFilePanel.Show();
	}
	else {
		mFilePanel.Window()->Activate();
	}
}

bool
RecWindow::ConnectIt()
{
	// connect thing if not connected
	status_t err = B_OK;
	media_format fmt;
	fprintf(stderr, "ConnectIt(): %s\n", mRecorder.IsConnected() ? "already connected" : "not yet connected");
	if (!mRecorder.IsConnected()) {
		fmt.type = B_MEDIA_RAW_AUDIO;
		char str[200];
		string_for_format(fmt, str, 200);
		fprintf(stderr, "connect format %s\n", str);
		if (mDevice != 0) {
			fprintf(stderr, "mDevice is non-NULL, connecting to id %ld\n", mDevice->node);
			mRecorder.Connect(*mDevice, 0, &fmt);
		}
		else {
			fprintf(stderr, "connecting to default for format\n");
			err = mRecorder.Connect(fmt);
		}
		if (err < B_OK) {
			fprintf(stderr, "cannot connect recorder to input\n");
			(new BAlert("", strerror(err), "Stop"))->Go();
			return false;
		}
		fmt = mRecorder.Format();
		mFrameSize = (fmt.u.raw_audio.format & 0xf)*fmt.u.raw_audio.channel_count;
		fprintf(stderr, "frame size: %ld\n", mFrameSize);
		fprintf(stderr, "data rate: %.3f MB/s\n", fmt.u.raw_audio.frame_rate*mFrameSize/(1024.0*1024.0));
		mPeakFormat = fmt.u.raw_audio.format;
		mRecorder.Start();
	}
	return true;
}

void 
RecWindow::GotFile(BMessage *msg)
{
	if (mRecording) {
		StopRecording();
	}
	CloseFile();
	if (!ConnectIt()) {
		return;
	}
	status_t err = B_OK;
	entry_ref ref;
	media_file_format ff;
	int32 cookie = 0;
	while ((err = get_next_file_format(&cookie, &ff)) == B_OK) {
#if 1
		if (!strcasecmp(ff.file_extension, "wav")) {
			fprintf(stderr, "chose WAV file format\n");
			break;
		}
#else
		uint32 mask = media_file_format::B_WRITABLE | media_file_format::B_KNOWS_RAW_AUDIO;
		if ((ff.capabilities & mask) == mask) {
			fprintf(stderr, "chose %s file format\n", ff.file_extension);
			break;
		}
#endif
	}
	if (msg->FindRef("refs", &ref) < B_OK) {
		const char * name;
		if (((err = msg->FindRef("directory", &ref)) < B_OK) ||
				((err = msg->FindString("name", &name)) < B_OK)) {
			fprintf(stderr, "file message has neither refs nor directory/name\n");
			(new BAlert("", strerror(err), "Stop"))->Go();
			return;
		}
		BDirectory dir(&ref);
		BEntry ent;
		BFile file;
		if ((err = dir.CreateFile(name, &file)) < B_OK) {
			fprintf(stderr, "cannot create output file\n");
			(new BAlert("", strerror(err), "Stop"))->Go();
			return;
		}
		if (ff.mime_type[0] != 0) {
			file.RemoveAttr("BEOS:TYPE");
			file.WriteAttr("BEOS:TYPE", 'MIMS', 0, ff.mime_type, strlen(ff.mime_type));
		}
		if ((err = dir.FindEntry(name, &ent)) < B_OK) {
			fprintf(stderr, "cannot get entry for output file\n");
			(new BAlert("", strerror(err), "Stop"))->Go();
			return;
		}
		ent.GetRef(&ref);
	}
	if (err != B_OK) {
		fprintf(stderr, "no file format found\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
		return;
	}
	mFile = new BMediaFile(&ref, &ff);
	if ((err = mFile->InitCheck()) < B_OK) {
		fprintf(stderr, "could not create BMediaFile\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
		delete mFile;
		mFile = 0;
		return;
	}
	media_format fmt(mRecorder.Format());
#if 1
	char str[200];
	string_for_format(fmt, str, 200);
	fprintf(stderr, "format: %s (%g;0x%lx;%ld;%ld;%ld)\n", str, fmt.u.raw_audio.frame_rate,
			fmt.u.raw_audio.format, fmt.u.raw_audio.channel_count, fmt.u.raw_audio.byte_order,
			fmt.u.raw_audio.buffer_size);
#endif
	if (0 == (mTrack = mFile->CreateTrack(&fmt))) {
		fprintf(stderr, "could not add track to file\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
		delete mFile;
		mFile = 0;
		return;
	}
	err = mFile->CommitHeader();
	if (err < B_OK) {
		fprintf(stderr, "CommitHeader() failed\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
		delete mFile;
		mFile = 0;
		mTrack = 0;
		return;
	}
}

void 
RecWindow::StartRecording()
{
	if (mRecording) {
		return;
	}
	// connect thing if not connected
	if (!ConnectIt()) {
		return;
	}
	mFIFO.Reset();
	// start the thread
	mThread = spawn_thread(buffer_flusher, "buffer_flusher", 100, this);
	if (mThread < B_OK) {
		fprintf(stderr, "could not spawn buffer_flusher\n");
		(new BAlert("", strerror(mThread), "Stop"))->Go();
		return;
	}
	// set mRecording to true, and start recorder
	mRecording = true;
	status_t err = resume_thread(mThread);
	if (err < B_OK) {
		fprintf(stderr, "could not resume buffer_flusher\n");
		kill_thread(mThread);
		mThread = -1;
		(new BAlert("", strerror(err), "Stop"))->Go();
		return;
	}
}

void 
RecWindow::StopRecording()
{
	// set mRecording to false
	mRecording = false;
	// disconnect thing
	status_t err = mRecorder.Disconnect();
	if (err < B_OK) {
		fprintf(stderr, "Disconnect() failed\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
	}
	status_t status;
	mFIFO.CopyNextBufferIn(&status, 0, B_INFINITE_TIMEOUT, true);
	wait_for_thread(mThread, &status);
}

void 
RecWindow::CloseFile()
{
	ASSERT(mRecording == false);
	if (!mFile) return;
	// flush file
	status_t err = mFile->CloseFile();
	if (err < B_OK) {
		fprintf(stderr, "CloseFile() failed\n");
		(new BAlert("", strerror(err), "Stop"))->Go();
	}
	//	close file
	delete mFile;
	mFile = 0;
	mTrack = 0;
}

void 
RecWindow::RecordHook(void *cookie, void *data, size_t size, const media_header &header)
{
	RecWindow * rw = (RecWindow *)cookie;
//fprintf(stderr, "RecordHook: %s\n", rw->mRecording ? "true" : "false");
	if (rw->mRecording) {
//		status_t err = rw->mTrack->WriteFrames(data, size /* /rw->mFrameSize */, B_MEDIA_KEY_FRAME);
		status_t err = rw->mFIFO.CopyNextBufferIn(data, size, B_INFINITE_TIMEOUT, false);
		if (err < (int32)size) {
			fprintf(stderr, "CopyNextBufferIn(%p, %ld, B_INFINITE_TIMEOUT, false) failed with %ld.\n",
				data, size, err);
			fprintf(stderr, "error while recording: %s; bailing\n", strerror(err));
			rw->mRecording = false;
		}
	}
//	do peak data
	if (rw->mPeakFormat == 0x2) {
		int16 m = 0;
		int16 n = 0;
		int16 * ptr = (int16 *)data;
		int cnt = size/4;
		while (cnt-- > 0) {
			//	we don't handle -32768 correctly
			if (*ptr > m)
				m = *ptr;
			else if (-*ptr > m)
				m = -*ptr;
			ptr++;
			if (*ptr > n)
				n = *ptr;
			else if (-*ptr > n)
				n = -*ptr;
			ptr++;
		}
		rw->mPeakView->SetMax(m/32767.0, n/32767.0);
	}
	else if (rw->mPeakFormat == 0x4) {
		int32 m = 0;
		int32 n = 0;
		int32 * ptr = (int32 *)data;
		int cnt = size/8;
		while (cnt-- > 0) {
			//	we don't handle -32768 correctly
			if (*ptr > m)
				m = *ptr;
			else if (-*ptr > m)
				m = -*ptr;
			ptr++;
			if (*ptr > n)
				n = *ptr;
			else if (-*ptr > n)
				n = -*ptr;
			ptr++;
		}
		rw->mPeakView->SetMax(m/(32767.0*65536.0), n/(32767.0*65536.0));
	}
	else if (rw->mPeakFormat == 0x24) {
		float m = 0;
		float n = 0;
		float * ptr = (float *)data;
		int cnt = size/8;
		while (cnt-- > 0) {
			if (*ptr > m)
				m = *ptr;
			else if (-*ptr > m)
				m = -*ptr;
			ptr++;
			if (*ptr > n)
				n = *ptr;
			else if (-*ptr > n)
				n = -*ptr;
			ptr++;
		}
		rw->mPeakView->SetMax(m, n);
	}
}

status_t
RecWindow::buffer_flusher(
	void * data)
{
	RecWindow * rw = (RecWindow *)data;
	const void * ptr;
	ssize_t size;
	off_t total = 0;
	while (rw->mRecording && ((size = rw->mFIFO.BeginGet(&ptr, 65536L, 3000000LL)) > 0)) {
		if (0 > rw->mTrack->WriteFrames(ptr, size, B_MEDIA_KEY_FRAME)) {
			fprintf(stderr, "buffer_flusher couldn't write data; bailing\n");
			rw->mRecording = false;
		}
		else {
//			fprintf(stderr, "buffer_flusher: WriteFrames(%ld)\n", size);
		}
		rw->mFIFO.EndGet();
		total += size;
	}
	fprintf(stderr, "buffer_flusher: terminates, size = %ld, rw->mRecording = %s, total = %Ld\n",
			size, rw->mRecording ? "true" : "false", total);
	if (size == 0) {
		fprintf(stderr, "This probably means a 3 second timeout expired without any data to record.\n");
	}
	return B_OK;
}

void
RecWindow::SelectDevice(
	BMessage * msg)
{
	if (mRecording) {
		return;
	}
	CloseFile();
	if (!mDevWin) {
		mDevWin = new DevWindow(this);
		mDevWin->Show();
		return;
	}
	int32 node_id = -1;
	status_t n_err, d_err;
	ssize_t sz;
	const void * dnip = 0;
	if ((n_err = msg->FindInt32("node_id", &node_id)) && (d_err = msg->FindData("dormant_info", B_RAW_TYPE, &dnip, &sz))) {
		mDevWin->Lock();
		if (mDevWin->IsHidden()) {
			mDevWin->Show();
		}
		else {
			mDevWin->Activate();
		}
		mDevWin->Unlock();
		return;
	}
	media_node instantiated;
	if (n_err < B_OK) {
		d_err = BMediaRoster::CurrentRoster()->InstantiateDormantNode(
			*reinterpret_cast<const dormant_node_info *>(dnip),
			&instantiated, B_FLAVOR_IS_GLOBAL);
		if (d_err < B_OK) {
			fprintf(stderr, "InstantiateDormantNode() failed: %s\n", strerror(d_err));
			(new BAlert("", strerror(d_err), "Stop"))->Go();
			return;
		}
	}
	if (mDevice != 0) {
		BMediaRoster::CurrentRoster()->ReleaseNode(*mDevice);
		delete mDevice;
		mDevice = 0;
	}
	mDevice = new media_node;
	if ((node_id >= 0) && !n_err) {
		status_t err = BMediaRoster::Roster()->GetNodeFor(node_id, mDevice);
		if (err < B_OK) {
			fprintf(stderr, "GetNodeFor() failed\n");
			(new BAlert("", strerror(err), "Stop"))->Go();
			delete mDevice;
			mDevice = 0;
			return;
		}
	}
	else {
		ASSERT(instantiated.node >= 0);
		*mDevice = instantiated;
	}
	if (mRecorder.IsConnected()) {
		mRecorder.Disconnect();
	}
	ConnectIt();
}



DevWindow::DevWindow(BLooper * l) :
	BWindow(BRect(200,200,350,300), "Choose Device", B_TITLED_WINDOW,
			B_NOT_RESIZABLE | B_NOT_CLOSABLE | B_NOT_ZOOMABLE),
	mThat(l)
{
	BView * v = new BView(Bounds(), "bg", B_FOLLOW_ALL, B_WILL_DRAW);
	v->SetViewColor(216, 216, 216);

	BRect r(10,10,140,32);
	BMessage * msg = new BMessage(DEVICE);
	msg->AddInt32("node_id", -1);
	BButton * def = new BButton(r, "default", "Default", msg);
	v->AddChild(def);

	live_node_info lni[30];
	int32 ioCount = 30;
	media_format out;
	out.type = B_MEDIA_RAW_AUDIO;
	status_t err = BMediaRoster::Roster()->GetLiveNodes(lni, &ioCount, 0, &out, 0, B_BUFFER_PRODUCER | B_PHYSICAL_INPUT);
	if (err < B_OK) {
		fprintf(stderr, "GetLiveNodes() failed: %s\n", strerror(err));
		(new BAlert("", strerror(err), "Stop"))->Go();
	}
	else for (int ix=0; ix<ioCount; ix++) {
		r.OffsetBy(0, r.Height()+12);
		msg = new BMessage(DEVICE);
		msg->AddInt32("node_id", lni[ix].node.node);
		char s[12];
		sprintf(s, "%d", ix+1);
		def = new BButton(r, s, lni[ix].name, msg);
		v->AddChild(def);
	}

	dormant_node_info dni[30];
	ioCount = 30;
	out.type = B_MEDIA_RAW_AUDIO;
	err = BMediaRoster::Roster()->GetDormantNodes(dni, &ioCount, 0, &out, 0, B_BUFFER_PRODUCER | B_PHYSICAL_INPUT);
	if (err < B_OK) {
		fprintf(stderr, "GetDormantNodes() failed: %s\n", strerror(err));
		(new BAlert("", strerror(err), "Stop"))->Go();
	}
	else for (int ix=0; ix<ioCount; ix++) {
		media_node_id running = -1;
		if ((B_OK == BMediaRoster::CurrentRoster()->GetInstancesFor(
				dni[ix].addon, dni[ix].flavor_id, &running)) && (running > -1)) {
			fprintf(stderr, "%s: already running\n", dni[ix].name);
			continue;
		}
		r.OffsetBy(0, r.Height()+12);
		msg = new BMessage(DEVICE);
		msg->AddData("dormant_info", B_RAW_TYPE, &dni[ix], sizeof(dni[ix]));
		char s[12];
		sprintf(s, "%d", ix+1);
		def = new BButton(r, s, dni[ix].name, msg);
		v->AddChild(def);
	}

	v->ResizeTo(r.right+11, r.bottom+11);
	ResizeTo(r.right+11, r.bottom+11);

	AddChild(v);
}


DevWindow::~DevWindow()
{
}

void 
DevWindow::MessageReceived(BMessage *msg)
{
	if (msg->what == DEVICE) {
		Hide();
		mThat.SendMessage(msg);
	}
	else {
		BWindow::MessageReceived(msg);
	}
}

bool 
DevWindow::QuitRequested()
{
	Hide();
	mThat.SendMessage(B_QUIT_REQUESTED);
	return false;
}


int
main(
	int argc,
	char * argv[])
{
	if ((argc > 2) || ((argc == 2) && (argv[1][0] == '-'))) {
		fprintf(stderr, "usage: record [filename]\n");
		return 1;
	}
	BApplication app("application/x-vnd.hplus.simplerecorder");
	RecWindow * w = new RecWindow;
	if (argc == 2) {
		entry_ref ref;
		if (get_ref_for_path(argv[1], &ref) < 0) {
			fprintf(stderr, "cannot record to: %s\n", argv[1]);
		}
		else {
			BMessage msg(B_REFS_RECEIVED);
			msg.AddRef("refs", &ref);
			w->PostMessage(&msg);
		}
	}
	w->Show();
	app.Run();
	return 0;
}


