//	PlayNode.cpp
//	------------
//	Copyright 1999, Be Incorporated.   All Rights Reserved.
//	This file may be used under the terms of the Be Sample Code License.

#include <Application.h>
#include <Window.h>
#include <View.h>
#include <SoundPlayer.h>
#include <MediaRoster.h>
#include <MediaAddOn.h>
#include <MediaNode.h>
#include <TimeSource.h>
#include <MediaTrack.h>
#include <MediaFile.h>
#include <OS.h>
#include <Entry.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


media_node node;
media_multi_audio_format format;
BMediaRoster * r;
BSoundPlayer * s;

#define AMPLITUDE 20000*65536L





BMediaFile * mediaFile;
BMediaTrack * mediaTrack;
media_format mediaFormat;

static bool
suitable_track(BMediaTrack * trk)
{
	mediaFormat.type = B_MEDIA_RAW_AUDIO;
	if ((trk->DecodedFormat(&mediaFormat) < B_OK) || (mediaFormat.type != B_MEDIA_RAW_AUDIO)) {
		return false;
	}
	return true;
}

static thread_id loadThread = -1;
static sem_id loadData = -1;
static sem_id loadFree = -1;
static char * loadBuffer = 0;
static int32 loadTogo = 0;
static size_t loadPos = 0;

static status_t
load_thread(void *)
{
	status_t err;
	int32 frsize = (mediaFormat.u.raw_audio.format & 0xf) *
		mediaFormat.u.raw_audio.channel_count;
	int cnt = 0;
	while (1) {
		err = acquire_sem_etc(loadFree, mediaFormat.u.raw_audio.buffer_size, 0, 0);
		if (err < B_OK) {
			break;
		}
		int64 frs = mediaFormat.u.raw_audio.buffer_size/frsize;
		err = mediaTrack->ReadFrames(loadBuffer+cnt*mediaFormat.u.raw_audio.buffer_size, &frs);
		if (err < 0) {
			break;
		}
//fprintf(stderr, "ReadFrames(%Ld)\n", frs);
		atomic_add(&loadTogo, mediaFormat.u.raw_audio.buffer_size);
		release_sem_etc(loadData, mediaFormat.u.raw_audio.buffer_size, B_DO_NOT_RESCHEDULE);
		cnt = (cnt+1)&3;
	}
	delete_sem(loadData);
	return 0;
}

static void
init_loading()
{
	loadBuffer = new char[mediaFormat.u.raw_audio.buffer_size*4];
	loadData = create_sem(0, "loadData");
	loadFree = create_sem(mediaFormat.u.raw_audio.buffer_size*4, "loadFree");
	loadThread = spawn_thread(load_thread, "load_thread", 100, 0);
	resume_thread(loadThread);
	//	sync with first buffer
	acquire_sem(loadData);
	release_sem_etc(loadData, 1, B_DO_NOT_RESCHEDULE);
}

static void
stop_loading()
{
	delete_sem(loadFree);
	status_t status = 0;
	wait_for_thread(loadThread, &status);
	delete_sem(loadData);
	loadFree = -1;
	loadData = -1;
	loadThread = -1;
	delete[] loadBuffer;
	loadBuffer = 0;
	loadTogo = 0;
}

static bool
get_data(
	void * into,
	size_t size)
{
	status_t err = acquire_sem_etc(loadData, size, 0, 0);
	int there = size;
	if (err < B_OK) {
		if (loadTogo <= 0) {
			memset(into, 0, size);
			return false;
		}
		if (there > loadTogo) there = loadTogo;
	}
	if ((size_t) there < size) {
		memset((char *)into + there, 0, size-there);
	}
	if (loadPos + there >= mediaFormat.u.raw_audio.buffer_size*4) {
		int temp = mediaFormat.u.raw_audio.buffer_size*4-loadPos;
		memcpy(into, loadBuffer+loadPos, temp);
		atomic_add(&loadTogo, -temp);
		release_sem_etc(loadFree, temp, B_DO_NOT_RESCHEDULE);
		there -= temp;
		loadPos = 0;
		into = (char *)into + temp;
	}
	memcpy(into, loadBuffer+loadPos, there);
	loadPos += there;
	atomic_add(&loadTogo, -there);
	release_sem_etc(loadFree, there, B_DO_NOT_RESCHEDULE);
	return true;
}


class W : public BWindow {
public:
	W() : BWindow(BRect(100,100,300,200), "playnode", B_TITLED_WINDOW,
			B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
		{
		}
	bool QuitRequested()
		{
			be_app->PostMessage(B_QUIT_REQUESTED);
			return true;
		}
};


#define PHASE 0x40

//	This could be made smarter, to handle various
//	formats and channel counts.
void
PlayBuffer(
	void *,
	void * buffer,
	size_t size,
	const media_raw_audio_format &)
{
#if 0
	static int i;
	int * s = (int *)buffer;
	for (int ix=0; ix<size/(sizeof(*s)*2); ix++) {
//	this is deliberately out of phase!
		s[ix*2] = (i & PHASE) ? -AMPLITUDE : AMPLITUDE;
		s[ix*2+1] = (i & PHASE) ? AMPLITUDE : -AMPLITUDE;
		i++;
	}
#else
	if (!get_data(buffer, size)) {
		be_app->PostMessage(B_QUIT_REQUESTED);
	}
#endif
}

int
main(
	int argc,
	char * argv[])
{
	bool start = true;
	if ((argc != 3) && !((argc == 2) && !strcmp(argv[1], "--list"))) {
		fprintf(stderr, "usage: playnode { nodename file } | { --list [ name[*] ] }\n");
		return 1;
	}
	bool list = false;
	if (!strcmp(argv[1], "--list")) {
		list = true;
		argv++;
		argc--;
		if (!argv[1]) argv[1] = "*";
	}
	BApplication app("application/x-vnd.be.playnode");
	status_t err;
	format.frame_rate = 48000.0;
	format.format = 0x4;
	format.channel_count = 2;
//	format.byte_order = B_HOST_IS_BENDIAN ? B_MEDIA_BIG_ENDIAN : B_MEDIA_LITTLE_ENDIAN;
//	format.buffer_size = 2048;
	r = BMediaRoster::Roster(&err);
	if ((r == NULL) || (err != B_OK)) {
		fprintf(stderr, "Could not find media roster: %s (%lx)\n",
				strerror(err), err);
		start = false;
	}
	else {
		//	list live nodes
		live_node_info *info = new live_node_info[100];
		int32 count = 100;
		status_t err;
		media_format input;
		input.type = B_MEDIA_RAW_AUDIO;
		input.u.raw_audio = format;
		err = r->GetLiveNodes(info, &count, &input, NULL, argv[1], B_BUFFER_CONSUMER);
		bool error = false;
		bool found = false;
		if (list) {
			fprintf(stderr, "%ld Live Nodes found\n", count);
			for (int ix=0; ix<count; ix++) {
				fprintf(stderr, "%ld: %s\n", info[ix].node.node, info[ix].name);
			}
			fprintf(stderr, "\n");
		}
		else {
			//	check for match
			if (!error) switch (count) {
			case 0:
				break;
			case 1:
				found = true;
				//	remember node
				node = info[0].node;
				fprintf(stderr, "using live node '%s'\n", info[0].name);
				break;
			default:
				fprintf(stderr, "Found %ld sound output Nodes matching '%s'\n", count, argv[1]);
				error = true;
				for (int ix=0; ix<count; ix++) {
					fprintf(stderr, "%s\n", info[ix].name);
				}
				break;
			}
			else {
				fprintf(stderr, "Error finding Nodes: %s (0x%lx)\n", strerror(err), err);
			}
		}
		delete[] info;
		if ((!found && !error) || list) {
			//	list dormant nodes
			input.type = B_MEDIA_RAW_AUDIO;
			input.u.raw_audio = format;
			dormant_node_info * dni = new dormant_node_info[100];
			count = 100;
			err = r->GetDormantNodes(dni, &count, &input, NULL, argv[1], B_BUFFER_CONSUMER);
			//	check for match
			if (list) {
				fprintf(stderr, "%ld Dormant Nodes found\n", count);
				for (int ix=0; ix<count; ix++) {
					fprintf(stderr, "%ld/%ld: %s\n", dni[ix].addon, dni[ix].flavor_id,
							dni[ix].name);
				}
			}
			else {
				error = (err != B_OK);
				if ((count == 1) && !error) {
					//	check for existing instance
					int32 nid;
					err = r->GetInstancesFor(dni->addon, dni->flavor_id, &nid);
					if (err == B_OK) {
						fprintf(stderr, "There is an existing instance: id %ld\n", nid);
						err = r->GetNodeFor(nid, &node);
					}
					else {
						//	instantiate
						//	remember node
						err = r->InstantiateDormantNode(*dni, &node, 0);
					}
				}
				if (err) {
					fprintf(stderr, "Error finding dormant nodes: %s (0x%lx)\n",
							strerror(err), err);
				}
				else if (count != 1) {
					fprintf(stderr, "Found %ld matches for dormant nodes\n", count);
					for (int ix=0; ix<count; ix++) {
						fprintf(stderr, "%s\n", dni[ix].name);
					}
				}
				else {
					fprintf(stderr, "using dormant node '%s', live id %ld\n",
						dni[0].name, node.node);
					found = true;
					if ((node.kind & B_PHYSICAL_OUTPUT) && (node.kind & B_TIME_SOURCE)) {
						r->SetTimeSourceFor(node.node, node.node);
					}
					BTimeSource * ts = r->MakeTimeSourceFor(node);
					if (ts == NULL) {
						fprintf(stderr, "TimeSource is NULL!\n");
					}
					else {
						r->StartNode(node, ts->PerformanceTimeFor(ts->RealTimeFor(ts->Now(), 0)+40000));
					}
					ts->Release();
				}
			}
			delete[] dni;
		}
		if (error || list) start = false;
	}
	if (start) {
		entry_ref ref;
		if (get_ref_for_path(argv[2], &ref) < B_OK) {
			goto file_error;
		}
		mediaFile = new BMediaFile(&ref, B_MEDIA_FILE_UNBUFFERED);
		if (mediaFile->InitCheck() < B_OK) {
			goto file_error;
		}
		for (int ix=0; ix<mediaFile->CountTracks(); ix++) {
			BMediaTrack * trk = mediaFile->TrackAt(ix);
			if (suitable_track(trk)) {
				mediaTrack = trk;
				init_loading();
				goto done;
			}
		}
file_error:
		start = false;
		fprintf(stderr, "%s: could not open sound file\n", argv[2]);
	}
done:
	if (start && (node.node <= 0)) {
		fprintf(stderr, "did not find node: %s\n", argv[1]);
		start = false;
	}
	if (start) {
		(new W)->Show();
		format = mediaFormat.u.raw_audio;
		format.buffer_size = 0;
		s = new BSoundPlayer(node, &format, "TestPlayer", NULL, PlayBuffer);
		s->Start();
		s->SetHasData(true);
	}
	else {
		app.PostMessage(B_QUIT_REQUESTED);
	}
	app.Run();
	if (start) {
		r->ReleaseNode(node);
		delete s;
		stop_loading();
		delete mediaFile;
	}
	return 0;
}
