/*
	A simple node that generates tones
*/

#include "EmptyAudio.h"

#include <Buffer.h>
#include <BufferGroup.h>
#include <EndPoint.h>
#include <OS.h>
#include <TimedEventQueue.h>

/* debugging? */
#define DEBUG 1

/* printf when debugging */
#ifdef DEBUG
#define PRINTF printf
#else
#define PRINTF
#endif

/* define sets of debug messages */
#define TERROR PRINTF

/* quit message */
#define B_ABSTAIN_YOU_DUMMY 0x60000000

/* endians */
#if B_HOST_IS_LENDIAN
#define ENDIAN B_MEDIA_LITTLE_ENDIAN
#else
#define ENDIAN B_MEDIA_BIG_ENDIAN
#endif

/* guess at latency */
#define PROCESSING_LATENCY 1000

BEmptyAudio::BEmptyAudio(const char *name) :
	BMediaNode(name),
	BBufferProducer(B_MEDIA_RAW_AUDIO),
	BControllable(),
	mTimeToQuit(false),
	mRunning(false),
	mHookedUp(false),
	mOutput(1, B_OUTPUT, "oscillator"),
	mMyGroup(NULL)
{
	mControlPort = create_port(5, "oscillator control");
	mRunThread = spawn_thread(RunThreadEntry, "oscillator thread", B_NORMAL_PRIORITY, this);

	mFlexibleFormat.type = B_MEDIA_RAW_AUDIO;
	mFlexibleFormat.u.raw_audio.frame_rate = media_raw_audio_format::wildcard.frame_rate;
	mFlexibleFormat.u.raw_audio.channel_count = media_raw_audio_format::wildcard.channel_count;
	mFlexibleFormat.u.raw_audio.format = media_raw_audio_format::wildcard.format;
	mFlexibleFormat.u.raw_audio.byte_order = ENDIAN;
	mFlexibleFormat.u.raw_audio.buffer_size = media_raw_audio_format::wildcard.buffer_size;

	mDefaultFormat.type = B_MEDIA_RAW_AUDIO;
	mDefaultFormat.u.raw_audio.frame_rate = 44100.0;
	mDefaultFormat.u.raw_audio.channel_count = 2;
	mDefaultFormat.u.raw_audio.format = B_AUDIO_FLOAT;
	mDefaultFormat.u.raw_audio.byte_order = ENDIAN;
	mDefaultFormat.u.raw_audio.buffer_size = 2048;
	
	media_source src(mControlPort, 1);
	mOutput.SetSource(src);
}


BEmptyAudio::~BEmptyAudio()
{
	status_t status;
	mTimeToQuit = true;
	write_port(mControlPort, B_ABSTAIN_YOU_DUMMY, NULL, 0);
	wait_for_thread(mMixThread, &status);
	delete_port(mControlPort);
	
	if (mEvents.HasEvents())
	{
		mEvents.FlushEvents(0, BTimedEventQueue::B_ALWAYS);
	}

	if (mOutput.NextBuffer())
	{
		mOutput.NextBuffer()->Recycle();
		mOutput.SetNextBuffer(NULL);
	}
	mOutput.SetBufferGroup(NULL, false);
}

/* from BMediaNode */
void 
BEmptyAudio::Start(bigtime_t performance_time)
{
	mEvents.PushEvent(performance_time, BTimedEventQueue::B_START, NULL, 0, 0);
}

void 
BEmptyAudio::Stop(bigtime_t performance_time, bool immediate)
{
	if (immediate)
	{
		performance_time = mInQueue.NextEventTime() - 1;
	}
	mEvents.PushEvent(performance_time, BTimedEventQueue::B_STOP, NULL, 0, 0);
}

void 
BEmptyAudio::Seek(bigtime_t media_time, bigtime_t performance_time)
{
	mEvents.PushEvent(performance_time, BTimedEventQueue::B_SEEK, NULL, 0, media_time);
}

void 
BEmptyAudio::TimeWarp(bigtime_t at_real_time, bigtime_t to_performance_time)
{
	mEvents.PushEvent(TimeSource()->PerformanceTimeFor(at_real_time),
						BTimedEventQueue::B_WARP, NULL, 0, performance_time);
}

void 
BEmptyAudio::Preroll()
{
}

status_t 
BEmptyAudio::HandleMessage(int32 message, const void *data, size_t size)
{
	if (message == B_ABSTAIN_YOU_DUMMY)
	{
		mTimeToQuit = true;
		return B_OK;
	}
	
	if (BBufferProducer::HandleMessage(message, data, size) != B_OK)
		if (BControllable::HandleMessage(message, data, size) != B_OK)
			if (BMediaNode::HandleMessage(message, data, size) != B_OK)
			{
				BMediaNode::HandleBadMessage(message, data, size);
				return B_ERROR;
			}
	return B_OK;
}

/* from BBufferProducer */
status_t 
BEmptyAudio::FormatSuggestionRequested(media_type type, int32 quality, media_format *format)
{
	if (type != B_MEDIA_RAW_AUDIO)
		return B_BAD_FORMAT;
	else
		format = mDefaultFormat;
		return B_OK;
}

status_t 
BEmptyAudio::FormatProposal(const media_source &output, media_format *format)
{
	status_t err = B_OK;
	if (format_is_compatible(*format, mFlexibleFormat)) 
		return err;
	
	if (format.type != B_MEDIA_RAW_AUDIO)
	{
		format = mFlexibleFormat;
		err = B_BAD_FORMAT;
	}
	if (format.u.raw_audio.frame_rate != mFlexibleFormat.u.raw_audio.frame_rate &&
		format.u.raw_audio.frame_rate != media_raw_audio_format::wildcard.frame_rate)
	{
		format.u.raw_audio.frame_rate = mFlexibleFormat.u.raw_audio.frame_rate;
		err = B_BAD_FORMAT;
	}
	if (format.u.raw_audio.channel_count != mFlexibleFormat.u.raw_audio.channel_count &&
		format.u.raw_audio.channel_count != media_raw_audio_format::wildcard.channel_count)
	{
		format.u.raw_audio.channel_count = mFlexibleFormat.u.raw_audio.channel_count;
		err = B_BAD_FORMAT;
	}
	if (format.u.raw_audio.format != mFlexibleFormat.u.raw_audio.format &&
		format.u.raw_audio.format != media_raw_audio_format::wildcard.format)
	{
		format.u.raw_audio.format = mFlexibleFormat.u.raw_audio.format;
		err = B_BAD_FORMAT;
	}
	if (format.u.raw_audio.byte_order != mFlexibleFormat.u.raw_audio.byte_order &&
		format.u.raw_audio.byte_order != media_raw_audio_format::wildcard.byte_order)
	{
		format.u.raw_audio.byte_order = mFlexibleFormat.u.raw_audio.byte_order;
		err = B_BAD_FORMAT;
	}
	if (format.u.raw_audio.buffer_size != mFlexibleFormat.u.raw_audio.buffer_size &&
		format.u.raw_audio.buffer_size != media_raw_audio_format::wildcard.buffer_size)
	{
		format.u.raw_audio.buffer_size = mFlexibleFormat.u.raw_audio.buffer_size;
		err = B_BAD_FORMAT;
	}
	return err;
}

status_t 
BEmptyAudio::FormatChangeRequested(const media_source &source, const media_destination &destination, media_format *format, int32 *_deprecated_)
{
	status_t err = B_OK;
	if (format_is_compatible(*format, mFlexibleFormat)) 
		err = B_OK;
	else
		err = B_BAD_FORMAT;
	
	if (format.type != B_MEDIA_RAW_AUDIO)
	{
		format = mFlexibleFormat;
	}
	if (format.u.raw_audio.frame_rate == media_raw_audio_format::wildcard.frame_rate)
	{
		format.u.raw_audio.frame_rate = mDefaultFormat.u.raw_audio.frame_rate;
	}
	if (format.u.raw_audio.channel_count == media_raw_audio_format::wildcard.channel_count)
	{
		format.u.raw_audio.channel_count = mDefaultFormat.u.raw_audio.channel_count;
	}
	if (format.u.raw_audio.format == media_raw_audio_format::wildcard.format)
	{
		format.u.raw_audio.format = mDefaultFormat.u.raw_audio.format;
	}
	if (format.u.raw_audio.byte_order == media_raw_audio_format::wildcard.byte_order ||
		format.u.raw_audio.byte_order != ENDIAN)
	{
		format.u.raw_audio.byte_order = mDefaultFormat.u.raw_audio.byte_order;
	}
	if (format.u.raw_audio.buffer_size == media_raw_audio_format::wildcard.buffer_size)
	{
		format.u.raw_audio.buffer_size = mDefaultFormat.u.raw_audio.buffer_size;
	}
	return err;
	
}

status_t 
BEmptyAudio::GetNextOutput(int32 *cookie, media_output *output)
{
	if (cookie != 0)
		return B_BAD_VALUE;
	*output = mOutput.Output();
	return B_OK;
}

status_t 
BEmptyAudio::DisposeOutputCookie(int32 cookie)
{
	/* nada */
}

status_t 
BEmptyAudio::SetBufferGroup(const media_source &for_source, BBufferGroup *group)
{
	mOutput.SetBufferGroup(group, true);
}

status_t 
BEmptyAudio::VideoClippingChanged(const media_source &for_source, int16 num_shorts, int16 *clip_data, const media_video_display_info &display, int32 *_deprecated_)
{
	/* nada */
}

status_t 
BEmptyAudio::PrepareToConnect(const media_source &what, const media_destination &where, media_format *format, media_source *out_source, char *out_name)
{
	if (what.id != mOutput.ID() || mOutput.Destination() != media_destination::null)
		return B_BAD_INDEX;
		
	stastus_t err = FormatChangeRequested(what, where, format);
	if (err < B_OK)
		return err;
	
	mOutput.SetDestination(where);
	*out_source = mOutput.Source();
	sprintf(out_name, mOutput.Name());
}

void 
BEmptyAudio::Connect(status_t error, const media_source &source, const media_destination &destination, const media_format &format, char *io_name)
{
	if (error != B_OK)
		mOutput.SetDestination(media_destination::null);

	mOutput.SetDestination(where);

	mOutput.SetFormat(format);
	mFrameSize = (format.u.raw_audio.format & 0xf) * format.u.raw_audio.channel_count;
	mFrameCount = format.u.raw_audio.buffer_size / (format.u.raw_audio.format & 0xf) * format.u.raw_audio.channel_count;
	mFrameDuration = 1000000.0 / (double)format.u.raw_audio.frame_rate;
	mBufferDuration = (bigtime_t) (mFrameCount * 1000000.0 / format.u.raw_audio.frame_rate);

	media_node_id  timesource;
	bigtime_t latency;
	FindLatencyFor(where, &latency, &timesource);
	if (timesource != TimeSource()->ID())
		latency = 0;
	mOutput.SetLatencies(PROCESSING_LATENCY, latency);

	int32 numbufs = MAX(3, (int32)ceil((float)latency / mBufferDuration));
	mOutput.SetBufferGroup(NULL);
	delete mMyBufferGroup;
	mMyBufferGroup = new BBufferGroup(format.u.raw_audio.buffer_size, MAX(numBufs, 2));
	if (mMyBufferGroup->InitCheck() != B_OK)
	{
		TERROR("error in output buffer group creation!!!!!\n");
		delete mMyBufferGroup;
		mMyBufferGroup = NULL;
		mOutput.SetDestination(media_destination::null);
	}
	else
		mOutput.SetBufferGroup(mMyBufferGroup, false);
		
	mHookedUp = true;
}

void 
BEmptyAudio::Disconnect(const media_source &what, const media_destination &where)
{
	mOutput.SetDestination(media_destination::NULL);
	mHookedUp = false;
}

void 
BEmptyAudio::LateNoticeReceived(const media_source &what, bigtime_t how_much, bigtime_t performance_time)
{
	mOutput.SetLatencies(&how_much, NULL);
}

void 
BEmptyAudio::EnableOutput(const media_source &what, bool enabled, int32 *_deprecated_)
{
	mOutput.SetOutputEnabled(enabled);
}

/* from BControllable */
status_t 
BEmptyAudio::GetParameterValue(int32 id, bigtime_t *last_change, void *value, size_t *ioSize)
{
}

void 
BEmptyAudio::SetParameterValue(int32 id, bigtime_t when, const void *value, size_t size)
{
}

status_t 
BEmptyAudio::StartControlPanel(BMessenger *out_messenger)
{
}

/* implementation */
void 
BEmptyAudio::RunThreadEntry(void *data)
{
	((BEmptyAudio *)data)->Run();
}

void 
BEmptyAudio::Run()
{
	status_t err = B_OK;
	schedulingLatency = estimate_max_scheduling_latency();
	bigtime_t latency = 0;
	bigtime_t wait_until = B_INFINITE_TIMEOUT;

	while(!mTimeToQuit)
	{
		//setup time
		latency = mOutput.DownStreamLatency() + schedulingLatency;
		if (mEvents.HasEvents())
		{
			RUN("Run - NextEventTime: %Ld\n", mEvents.NextEventTime());
			wait_until = TimeSource()->RealTimeFor(mEvents.NextEventTime(), latency);
			RUN("Run - wait for %Ld micros\n", wait_until - system_time());
		}
		else
		{
			wait_until = B_INFINITE_TIMEOUT;
			RUN("Run - wait forever\n");
		}
		
		//wait
		err = WaitForMessages();
		
		//handle something
		if (err == B_TIMED_OUT)
		{
			RUN("Run - handle event\n");
			HandleEvent();
		}
		else if (err != B_INTERRUPTED)
		{
			TERROR("Run - Unexpected error:\n %s exiting...", strerror(err));
			return;
		}
	}

}

#define ENOUGH_TIME_FOR_SOMEONE_ELSE_TO_DO_SOMETHING_USEFUL 100

status_t
BEmptyAudio::WaitForMessages(bigtime_t wait_until)
{
	status_t err = B_OK;
	int32 code = 0;
	char message[B_MEDIA_MESSAGE_SIZE];

	if (system_time() < wait_until - ENOUGH_TIME_FOR_SOMEONE_ELSE_TO_DO_SOMETHING_USEFUL)
		err = read_port_etc(mControlPort, &code, &message, B_MEDIA_MESSAGE_SIZE, B_ABSOLUTE_TIMEOUT, wait_until);
	else
		err = B_TIMED_OUT;
	
	if (err >= 0)
		HandleMessage(code, &message, err);
	else
		return err;
}

void
BEmptyAudio::HandleEvent()
{
	if (!mEvents.HasEvents())
		return;
		
	bigtime_t time = 0;
	int32 what = BTimedEventQueue::B_NO_EVENT;
	void *pointer = NULL;
	uint32 flags = BTimedEventQueue::B_NO_CLEANUP;
	int64 data = 0;
	
	status_t err = mEvents.PopEvent(&time, &what, &pointer, &flags, &data);
	if (err < B_OK)
		return;
		
	switch(what)
	{
		case BTimedEventQueue::B_START:
			HANDLE("B_START %Ld\n", time);
			if (!mRunning)
			{
				mRunning = true;
				mStartTime = time;
				mNextTime = mStartTime;
				mFrameTotal = 0;
				SetupBuffer(&mOutput);
			}
			break;
		case BTimedEventQueue::B_STOP:
			HANDLE("B_STOP %Ld\n", time);
			if (mRunning)
			{
				mRunning = false;
				mInQueue.FlushEvents(time, BTimedEventQueue::B_AFTER_TIME);
				if (mOutput.NextBuffer())
				{
					mOutput.NextBuffer(NULL);
				}
				SetDataStatus(B_DATA_NOT_AVAILABLE);
				SendDataStatus(B_DATA_NOT_AVAILABLE, mOutput.Destination(), time);
			}
			break;
		case BTimedEventQueue::B_SEEK:
			HANDLE("B_SEEK %Ld\n", time);
			/* nothing */
			break;
		case BTimedEventQueue::B_WARP:
			HANDLE("B_WARP %Ld\n", time);
			if	(mRunning)
			{
				mInQueue.FlushEvents(0, BTimedEventQueue::B_ALWAYS, true, HANDLE_BUFFER);
				mNextTime = data;
				//adjust start time back from new next time
				mStartTime = mNextTime - (bigtime_t)floor((mFrameTotal + mFrameCount) * 1000000.0
															/ (double)mOutput.Format().u.raw_audio.frame_rate);
				SetupBuffer(&mOutput);
			}
			break;
		case BTimedEventQueue::B_HANDLE_BUFFER:
			HANDLE("B_HANDLE_BUFFER %Ld\n", time);
			if (!mRunning)
				((BBuffer *)pointer)->Recycle();
			else
			{
				SendBuffer((BBuffer *)pointer, mOutput.Destination());
				mNextTime = mStartTime + (bigtime_t)floor((mFrameTotal + mMixFrameCount) * 1000000.0
												/ (double)mOutput.Format().u.raw_audio.frame_rate);
				SetupBuffer(&mOutput);
			}
			break;
		case BTimedEventQueue::B_DATA_STATUS:
			HANDLE("B_DATA_STATUS %Ld\n", time);
			/* nothing */
			break;
		case BTimedEventQueue::B_NO_EVENT:
			FUNCTION("B_NO_EVENT %Ld\n", time);
		default:
			return;
	}	
}

void
BEmptyAudio::SetupBuffer(BEndPoint *output)
{
	BBuffer *buffer = output->Buffers()->RequestBuffer(mOutput.Format().u.raw_audio.buffer_size);
	if (buffer)
	{
		buffer->Header()->start_time = mNextTime;
 		output->SetNextBuffer(buffer);
		FillBuffer(output);
		mEvents.PushEvent(mNextTime, BTimedEventQueue::B_HANDLE_BUFFER, buffer, 0, 0);
	}
	else
		output->SetNextBuffer(NULL);
}

void
BEmptyAudio::FillBuffer(BEndPoint *output)
{
	char *data = output->NextBuffer()->Header()->Data();
	
	switch(output->Format().u.raw_audio.format)
	{
		case media_raw_audio_format::B_AUDIO_FLOAT:
			memset(data, output->Format().u.raw_audio.buffer_size, 0);
			break;
		case media_raw_audio_format::B_AUDIO_INT:
			break;
		case media_raw_audio_format::B_AUDIO_SHORT:
			break;
		case media_raw_audio_format::B_AUDIO_UCHAR:
			break;
		default:
			return;
	}
}

