/*******************************************************************************
/
/	File:			CaptureWindow.cpp
/
/   Description:	Window for recording sound and arranging clips.
/
/	Copyright 1998, Be Incorporated, All Rights Reserved
/
*******************************************************************************/


#include <Application.h>
#include <Alert.h>
#include <Screen.h>
#include <Button.h>
#include <CheckBox.h>
#include <TextControl.h>
#include <MenuField.h>
#include <PopUpMenu.h>
#include <MenuItem.h>
#include <Box.h>
#include <ScrollView.h>
#include <Beep.h>
#include <StringView.h>
#include <Slider.h>
#include <Message.h>

#include <Path.h>
#include <FindDirectory.h>

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/fcntl.h>

#include <MediaRoster.h>
#include <TimeSource.h>

#include "CaptureWindow.h"
#include "SoundConsumer.h"
#include "SoundProducer.h"
#include "SoundListView.h"
#include "array_delete.h"
#include "FileUtils.h"

#if ! NDEBUG
#define FPRINTF(args) fprintf args
#else
#define FPRINTF(args)
#endif

#define DEATH FPRINTF
#define CONNECT FPRINTF
#define VOX FPRINTF
#define WINDOW FPRINTF

// default window positioning 
static const float MIN_WIDTH = 440.0f;
static const float MIN_HEIGHT = 350.0f;
static const float XPOS = 100.0f;
static const float YPOS = 200.0f;

// defaults for voice activation
static const int16 VOX_TMIN = 100;
static const int16 VOX_TMAX = 32767;
static const int16 VOX_TDEF  = 2000;
static const uint16 VOX_FMIN = 1;
static const uint16 VOX_FMAX = 50;
static const uint16 VOX_FDEF = 20;

// Attribute info for the BeOS raw audio file format.
// Note that we write the sound's header info as file
// attributes. It makes reading and writing the file
// a piece of cake, but if the attributes are somehow
// stripped, the file's useless.
static const char * SAMPLE_RATE_ATTR = "be:AUDIO:sample_rate";
static const char * CHANNELS_ATTR = "be:AUDIO:channels";
static const char * FORMAT_ATTR = "be:AUDIO:format";
static const char * ENDIAN_ATTR = "be:AUDIO:big_endian";
//static const char * OFFSET_ATTR = "be:AUDIO:data_offset";
static const char * LENGTH_ATTR = "be:AUDIO:num_samples";
static const char * TYPE_ATTR = "BEOS:TYPE";
static const char * file_type = "application/x-vnd.Be-RAWM";


CaptureWindow::CaptureWindow() :
	BWindow(BRect(XPOS,YPOS,XPOS+MIN_WIDTH,YPOS+MIN_HEIGHT), "Sound Recorder", B_DOCUMENT_WINDOW, B_ASYNCHRONOUS_CONTROLS),
	m_savePanel(B_SAVE_PANEL)
{
	m_roster = 0;
	m_background = 0;
	m_record = 0;
	m_play = 0;
	m_stop = 0;
	m_save = 0;
	m_vox = 0;
	m_length = 0;
	m_input = 0;
	m_recordNode = 0;
	m_playNode = 0;
	m_recording = false;
	m_voxFrames = VOX_FDEF;
	m_voxThreshold = m_voxFrames*VOX_TDEF;
	m_soundList = 0;
	m_scroller = 0;
	m_tempCount = 0;

	CalcSizes(MIN_WIDTH, MIN_HEIGHT);
}

CaptureWindow::~CaptureWindow()
{
	DEATH((stderr, "CaptureWindow::~CaptureWindow()\n"));
	//	The sound consumer and producer are Nodes; it has to be Release()d and the Roster
	//	will reap it when it's done.
	if (m_recordNode) {
		m_recordNode->Release();
	}
	if (m_playNode) {
		m_playNode->Release();
	}
	//	Clean up items in list view.
	if (m_soundList) {
		m_soundList->DeselectAll();
		for (int ix=0; ix<m_soundList->CountItems(); ix++) {
			WINDOW((stderr, "clean up item %d\n", ix+1));
			SoundListItem * item = dynamic_cast<SoundListItem *>(m_soundList->ItemAt(ix));
			if (item) {
				if (item->IsTemp()) {
					item->Entry().Remove();	//	delete temp file
				}
				delete item;
			}
		}
		m_soundList->MakeEmpty();
	}
	//	Clean up currently recording file, if any.
	m_recEntry.Remove();
	m_recEntry.Unset();
}

void
CaptureWindow::CalcSizes(
	float min_wid,
	float min_hei)
{
	//	Set up size limits based on new screen size
	BScreen screen;
	BRect r = screen.Frame();
	float wid = r.Width()-12;
	SetSizeLimits(min_wid, wid, min_hei, r.Height()-24);

	//	Don't zoom to cover all of screen; user can resize last bit if necessary.
	//	This leaves other windows visible.
	if (wid > 640) {
		wid = 640 + (wid-640)/2;
	}
	SetZoomLimits(wid, r.Height()-24);
}

void
CaptureWindow::WorkspaceActivated(
	int32 workspace,
	bool active)
{
	BWindow::WorkspaceActivated(workspace, active);
	//	re-calculate window size
	if (active) {
		CalcSizes(MIN_WIDTH, MIN_HEIGHT);
	}
}

status_t
CaptureWindow::IWindow()
{
	BPopUpMenu * popup = 0;
	status_t error;
	int max_input_count = 64;
	live_node_info * lni = new live_node_info[max_input_count];
	array_delete<live_node_info> lni_delete(lni);	//	auto-delete[]
	
	try {
		//	Find temp directory for recorded sounds.
		BPath path;
		if (!(error = find_directory(B_COMMON_TEMP_DIRECTORY, &path))) {
			error = m_tempDir.SetTo(path.Path());
		}
		if (error < 0) {
			goto bad_mojo;
		}

		//	Make sure the media roster is there (which means the server is there).
		m_roster = BMediaRoster::Roster(&error);
		if (!m_roster) {
			goto bad_mojo;
		}

		error = m_roster->GetAudioInput(&m_audioInputNode);
		if (error < B_OK) {	//	there's no input?
			goto bad_mojo;
		}

		error = m_roster->GetAudioMixer(&m_audioMixerNode);
		if (error < B_OK) {	//	there's no mixer?
			goto bad_mojo;
		}

		//	Create our internal Node which records sound, and register it.
		m_recordNode = new SoundConsumer("Sound Recorder");
		error = m_roster->RegisterNode(m_recordNode);
		if (error < B_OK) {
			goto bad_mojo;
		}

		// Create our internal Node which plays sound, and register it.
		m_playNode = new SoundProducer("Sound Player");
		error = m_roster->RegisterNode(m_playNode);
		if (error < B_OK) {
			goto bad_mojo;
		}

		//	Create the window header with controls
		BRect r(Bounds());
		r.bottom = r.top + 130;
		m_background = new BBox(r, "_background", B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP,
			B_WILL_DRAW|B_FRAME_EVENTS|B_NAVIGABLE_JUMP, B_PLAIN_BORDER);

		//	Input selection lists all available physical inputs that produce
		//	buffers with B_MEDIA_RAW_AUDIO format data.
		popup = new BPopUpMenu("Input");
		int32 real_count = max_input_count;
		media_format output_format;
		output_format.type = B_MEDIA_RAW_AUDIO;
		output_format.u.raw_audio = media_raw_audio_format::wildcard;
		error = m_roster->GetLiveNodes(lni, &real_count, 0, &output_format,
			0, B_BUFFER_PRODUCER | B_PHYSICAL_INPUT);
		if (real_count > max_input_count) {
			WINDOW((stderr, "dropped %ld inputs\n", real_count - max_input_count));
			real_count = max_input_count;
		}
		char selected_name[B_MEDIA_NAME_LENGTH] = "Default Input";
		BMessage * msg;
		BMenuItem * item;
		for (int ix=0; ix<real_count; ix++) {
			msg = new BMessage(INPUT_SELECTED);
			msg->AddData("node", B_RAW_TYPE, &lni[ix].node, sizeof(lni[ix].node));
			item = new BMenuItem(lni[ix].name, msg);
			popup->AddItem(item);
			if (lni[ix].node == m_audioInputNode) {
				strcpy(selected_name, lni[ix].name);
			}
		}

		//	Create the actual widget
		BRect frame(m_background->Bounds());
		r = frame;
		r.right = (r.left + r.right) / 2;
		r.InsetBy(10,10);
		r.bottom = r.top+18;
		m_input = new BMenuField(r, "Input", "Input:", popup);
		m_input->SetDivider(50);
		m_background->AddChild(m_input);

		//	Text field for entering length to record (in seconds)
		r.OffsetBy(r.Width()+20, 1);
		r.right -= 70;
		msg = new BMessage(LENGTH_CHANGED);
		m_length = new BTextControl(r, "Length", "Length:", "8", msg);
		m_length->SetDivider(60);
		m_background->AddChild(m_length);

		r.left += r.Width()+5;
		r.right += 70;
		r.top += 2;
		r.bottom += 2;
		BStringView* lenUnits = new BStringView(r, "Seconds", "seconds");
		m_background->AddChild(lenUnits);

		//	Check box for the start-on-sound feature
		r = m_input->Frame();
		r.OffsetBy(0, r.Height() + 10 + 18);
		r.right = r.left + 100;
		r.bottom = r.top + 30;
		msg = new BMessage(VOX_START);
		m_vox = new BCheckBox(r, "Vox Start", "Vox Start", msg);
		m_background->AddChild(m_vox);

		// Vox frames is the number of frames to average over when
		// calculating whether the sound has started.
		const rgb_color cBlue = { 0, 0, 229, 255 };
		r.OffsetBy(r.Width()+10, -18);
		r.bottom += 20;
		r.right += 50;
		msg = new BMessage(VOX_FRAMES);
		m_voxFrameSlider = new BSlider(r, "Frames", "Vox Frames", msg, VOX_FMIN, VOX_FMAX,
			B_TRIANGLE_THUMB);
		m_voxFrameSlider->SetValue(VOX_FDEF);
		m_voxFrameSlider->SetLimitLabels("1", "50");
		m_voxFrameSlider->SetBarColor(cBlue);
		m_background->AddChild(m_voxFrameSlider);
		
		// Threshold represents the minimum volume at which recording will begin,
		// stored internally as a product of the average value and the vox frames.
		r.OffsetBy(r.Width()+10, 0);
		msg = new BMessage(VOX_THRESHOLD);
		m_voxThresholdSlider = new BSlider(r, "Threshold", "Threshold", msg, VOX_TMIN, VOX_TMAX,
			B_TRIANGLE_THUMB);
		m_voxThresholdSlider->SetValue(VOX_TDEF);
		m_voxThresholdSlider->SetLimitLabels("Low", "High");
		m_voxThresholdSlider->SetBarColor(cBlue);
		m_background->AddChild(m_voxThresholdSlider);
  
		//	Button to start recording (or waiting for sound)
		r.left = frame.left+10;
		r.right = frame.right-10;
		r.OffsetBy(0, r.Height()+10);
		r.bottom = frame.bottom-10;
		r.right = r.left + (int)((r.right-r.left-30)/4);
		msg = new BMessage(RECORD);
		m_record = new BButton(r, "Record", "Record", msg);
		m_background->AddChild(m_record);

		//	Button for stopping recording or playback
		r.OffsetBy(r.Width()+10, 0);
		msg = new BMessage(STOP);
		m_stop = new BButton(r, "Stop", "Stop", msg);
		m_background->AddChild(m_stop);

		//	Button for starting playback of selected sound
		r.OffsetBy(r.Width()+10, 0);
		msg = new BMessage(PLAY);
		m_play = new BButton(r, "Play", "Play", msg);
		m_background->AddChild(m_play);

		//	Button for saving selected sound
		r.OffsetBy(r.Width()+10, 0);
		msg = new BMessage(SAVE);
		m_save = new BButton(r, "Save", "Save", msg);
		m_background->AddChild(m_save);

		//	The actual list of recorded sounds (initially empty) sits
		//	below the header with the controls.
		r = Bounds();
		r.top = frame.bottom+1;
		r.right -= B_V_SCROLL_BAR_WIDTH;
		r.bottom -= B_H_SCROLL_BAR_HEIGHT;
		m_soundList = new SoundListView(r, "Sound List", B_FOLLOW_ALL);
		msg = new BMessage(SOUND_SELECTED);
		m_soundList->SetSelectionMessage(msg);
		m_soundList->SetViewColor(216, 216, 216);
		m_scroller = new BScrollView("scroller", m_soundList, B_FOLLOW_ALL,
			0, true, true, B_PLAIN_BORDER);
		AddChild(m_scroller);

		AddChild(m_background);
		popup->Superitem()->SetLabel(selected_name);
		
		// Make sure the save panel is happy.
		m_savePanel.SetTarget(this);
	}
	catch (...) {
		goto bad_mojo;
	}
	UpdateButtons();
	return B_OK;

	//	Error handling.
bad_mojo:
	if (error >= 0) {
		error = B_ERROR;
	}
	if (m_recordNode) {
		m_recordNode->Release();
	}
	if (m_playNode) {
		m_playNode->Release();
	}
	delete m_background;
	if (!m_input) {
		delete popup;
	}
	return error;
}


bool
CaptureWindow::QuitRequested()	//	this means Close pressed
{
	StopRecording();
	StopPlaying();
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}


void
CaptureWindow::MessageReceived(
	BMessage * message)
{
	//	Your average generic message dispatching switch() statement.
	switch (message->what) {
	case INPUT_SELECTED:
		Input(message);
		break;
	case LENGTH_CHANGED:
		Length(message);
		break;
	case VOX_START:
		Vox(message);
		break;
	case VOX_FRAMES:
		VoxFrames(message);
		break;
	case VOX_THRESHOLD:
		VoxThreshold(message);
		break;
	case SOUND_SELECTED:
		Selected(message);
		break;
	case STOP_PLAYING:
		StopPlaying();
		break;
	case STOP_RECORDING:
		StopRecording();
		break;
	case RECORD:
		Record(message);
		break;
	case STOP:
		Stop(message);
		break;
	case PLAY:
		Play(message);
		break;
	case SAVE:
		Save(message);
		break;
	case B_SAVE_REQUESTED:
		DoSave(message);
		break;
	default:
		BWindow::MessageReceived(message);
		break;
	}
}

void
CaptureWindow::Record(
	BMessage * message)
{	
	//	User pressed Record button
	m_recording = !m_recOnVox;
	int seconds = atoi(m_length->Text());
	if (seconds < 1) {
		ErrorAlert("record a sound that's shorter than a second", B_ERROR);
		return;
	}

	if (m_buttonState != btnPaused) {
		return;			//	user is too fast on the mouse
	}
	SetButtonState(btnRecording);

	char name[256];
	//	Create a file with a temporary name
	status_t err = NewTempName(name);
	if (err < B_OK) {
		ErrorAlert("find an unused name to use for the new recording", err);
		return;
	}
	//	Find the file so we can refer to it later
	err = m_tempDir.FindEntry(name, &m_recEntry);
	if (err < B_OK) {
		ErrorAlert("find the temporary file created to hold the new recording", err);
		return;
	}
	err = m_recFile.SetTo(&m_tempDir, name, O_RDWR);
	if (err < B_OK) {
		ErrorAlert("open the temporary file created to hold the new recording", err);
		m_recEntry.Unset();
		return;
	}
	//	Reserve space on disk (creates fewer fragments)
	err = m_recFile.SetSize(seconds*4*44100LL);
	if (err < B_OK) {
		ErrorAlert("record a sound that long", err);
		m_recEntry.Remove();
		m_recEntry.Unset();
		return;
	}
	m_recLimit = seconds*4*44100LL;
	m_recSize = 0;

	m_recFile.Seek(0, SEEK_SET);

	//	Hook up input
	err = MakeRecordConnection(m_audioInputNode);
	if (err < B_OK) {
		ErrorAlert("connect to the selected sound input", err);
		m_recEntry.Remove();
		m_recEntry.Unset();
		return;
	}

	//	And get it going...
	bigtime_t then = m_recordNode->TimeSource()->Now()+50000LL;
	m_roster->StartNode(m_recordNode->Node(), then);
	if (m_audioInputNode.kind & B_TIME_SOURCE) {
		m_roster->StartNode(m_audioInputNode, m_recordNode->TimeSource()->RealTimeFor(then, 0));
	}
	else {
		m_roster->StartNode(m_audioInputNode, then);
	}
}

void
CaptureWindow::Play(
	BMessage * message)
{
	//	User pressed Play button.
	//	Get selection.
	int32 selIdx = m_soundList->CurrentSelection();
	SoundListItem* pItem = dynamic_cast<SoundListItem*>(m_soundList->ItemAt(selIdx));
	if (! pItem) {
		return;
	}

	BEntry& entry = pItem->Entry();
	status_t err;
	err = m_playFile.SetTo(&entry, B_READ_ONLY);
	if (err != B_OK) {
		ErrorAlert("get the file to play", err);
		return;
	}

	if (m_buttonState != btnPaused) {
		return;			//	user is too fast on the mouse
	}
	SetButtonState(btnPlaying);

	media_raw_audio_format fmt;
	m_playFile.ReadAttr(SAMPLE_RATE_ATTR, B_FLOAT_TYPE, 0, &fmt.frame_rate, sizeof(float));
	m_playFile.ReadAttr(CHANNELS_ATTR, B_INT32_TYPE, 0, &fmt.channel_count, sizeof(int32));
	m_playFile.ReadAttr(FORMAT_ATTR, B_INT32_TYPE, 0, &fmt.format, sizeof(int32));
	int32 endian;
	m_playFile.ReadAttr(ENDIAN_ATTR, B_INT32_TYPE, 0, &endian, sizeof(int32));
	fmt.byte_order = (endian ? B_MEDIA_BIG_ENDIAN : B_MEDIA_LITTLE_ENDIAN);
	m_playFile.ReadAttr(LENGTH_ATTR, B_INT64_TYPE, 0, &m_playSize, sizeof(int64));
	fmt.buffer_size = media_raw_audio_format::wildcard.buffer_size;

	m_playLimit = m_playFile.Seek(0, SEEK_END);
	if (m_playLimit <= B_OK) {
		ErrorAlert("get the size of the file", m_playLimit);
		return;
	}
	m_playSize = 0;

	//	Hook up output
	err = MakePlayConnection(fmt);
	if (err < B_OK) {
		ErrorAlert("connect to the mixer", err);
		return;
	}

	//	And get it going...
	bigtime_t then = m_playNode->TimeSource()->Now()+50000LL;
	m_roster->StartNode(m_playNode->Node(), then);
}

void
CaptureWindow::Stop(
	BMessage * message)
{
	//	User pressed Stop button.
	//	Stop recorder.
	StopRecording();
	//	Stop player.
	StopPlaying();
}

void
CaptureWindow::Save(
	BMessage * message)
{
	//	User pressed Save button.
	//	Find the item to save.
	int32 selIdx = m_soundList->CurrentSelection();
	SoundListItem* pItem = dynamic_cast<SoundListItem*>(m_soundList->ItemAt(selIdx));
	if ((! pItem) || (pItem->Entry().InitCheck() != B_OK)) {
		return;
	}
	
	// Update the save panel and show it.
	char filename[B_FILE_NAME_LENGTH];
	pItem->Entry().GetName(filename);
	BMessage saveMsg(B_SAVE_REQUESTED);
	entry_ref ref;
	pItem->Entry().GetRef(&ref);
	
	saveMsg.AddPointer("sound list item", pItem);
	m_savePanel.SetSaveText(filename);
	m_savePanel.SetMessage(&saveMsg);
	m_savePanel.Show();
}

void
CaptureWindow::DoSave(
	BMessage * message)
{
	// User picked a place to put the file.
	// Find the location of the old (e.g.
	// temporary file), and the name of the
	// new file to save.
	entry_ref old_ref, new_dir_ref;
	const char* new_name;
	SoundListItem* pItem;
		
	if ((message->FindPointer("sound list item", (void**) &pItem) == B_OK)
		&& (message->FindRef("directory", &new_dir_ref) == B_OK)
		&& (message->FindString("name", &new_name) == B_OK))
	{
		BEntry& oldEntry = pItem->Entry();
		BFile oldFile(&oldEntry, B_READ_WRITE);
		if (oldFile.InitCheck() != B_OK)
			return;

		BDirectory newDir(&new_dir_ref);
		if (newDir.InitCheck() != B_OK)
			return;
			
		BFile newFile;		
		newDir.CreateFile(new_name, &newFile); 
		
		if (newFile.InitCheck() != B_OK)
			return;
				
		status_t err = CopyFile(newFile, oldFile);
		
		if (err == B_OK) {
			// clean up the sound list and item
			if (pItem->IsTemp())
				oldEntry.Remove(); // blows away temp file!
			oldEntry.SetTo(&newDir, new_name);
			pItem->SetTemp(false);	// don't blow the new entry away when we exit!
			m_soundList->Invalidate();
		}
	} else {
		WINDOW((stderr, "Couldn't save file.\n"));
	}
}

void
CaptureWindow::Vox(
	BMessage * message)
{
	//	User pressed Vox checkbox.
	//	We detect this in Record().
	BCheckBox* checkBox;
	if (message->FindPointer("source", (void**) &checkBox) == B_OK) {
		m_recOnVox = (checkBox->Value() == B_CONTROL_ON);
		if (m_recOnVox) {
			VOX((stderr, "vox start activated.\n"));
		} else {
			VOX((stderr, "vox start deactivated.\n"));
		}
	}
}

void
CaptureWindow::VoxFrames(
	BMessage * message)
{
	BSlider* slider;
	if (message->FindPointer("source", (void**) &slider) == B_OK) {
		m_voxFrames = (uint16) slider->Value();
		VOX((stderr, "vox frames now %u.\n", m_voxFrames));
	}
}

void
CaptureWindow::VoxThreshold(
	BMessage * message)
{
	BSlider* slider;
	if (message->FindPointer("source", (void**) &slider) == B_OK) {
		m_voxThreshold = m_voxFrames*slider->Value();
	}
	VOX((stderr, "vox threshold now %lu.\n", m_voxThreshold));
}

void
CaptureWindow::Length(
	BMessage * message)
{
	//	User changed the Length field -- validate
	const char * ptr = m_length->Text();
	const char * start = ptr;
	const char * anchor = ptr;
	const char * end = m_length->Text()+m_length->TextView()->TextLength();
	while (ptr < end) {
		//	Remember the last start-of-character for UTF-8 compatibility
		//	needed in call to Select() below (which takes bytes).
		if (*ptr & 0x80) {
			if (*ptr & 0xc0 == 0xc0) {
				anchor = ptr;
			}
		}
		else {
			anchor = ptr;
		}
		if (!isdigit(*ptr)) {
			m_length->TextView()->MakeFocus(true);
			m_length->TextView()->Select(anchor-start, m_length->TextView()->TextLength());
			beep();
			break;
		}
		ptr++;
	}
}

void
CaptureWindow::Input(
	BMessage * message)
{
	//	User selected input from pop-up
	const media_node * node = 0;
	ssize_t size = 0;
	if (message->FindData("node", B_RAW_TYPE, (const void **)&node, &size)) {
		return;		//	bad input selection message
	}
	m_audioInputNode = *node;
}

void
CaptureWindow::Selected(
	BMessage * message)
{
	//	User selected a sound in list view
	UpdateButtons();
}

status_t
CaptureWindow::MakeRecordConnection(
	const media_node & input)
{
	CONNECT((stderr, "CaptureWindow::MakeRecordConnection()\n"));
	
	//	Find an available output for the given input node.
	int32 count = 0;
	status_t err = m_roster->GetFreeOutputsFor(input, &m_audioOutput, 1, &count, B_MEDIA_RAW_AUDIO);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): couldn't get free outputs from audio input node\n"));
		return err;
	}
	if (count < 1) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): no free outputs from audio input node\n"));
		return B_BUSY;
	}

	//	Find an available input for our own Node. Note that we go through the
	//	MediaRoster; calling Media Kit methods directly on Nodes in our app is
	//	not OK (because synchronization happens in the service thread, not in
	//	the calling thread).
	// TODO: explain this
	err = m_roster->GetFreeInputsFor(m_recordNode->Node(), &m_recInput, 1, &count, B_MEDIA_RAW_AUDIO);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): couldn't get free inputs for sound recorder\n"));
		return err;
	}
	if (count < 1) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): no free inputs for sound recorder\n"));
		return B_BUSY;
	}

	//	Find out what the time source of the input is.
	//	For most nodes, we just use the preferred time source (the DAC) for synchronization.
	//	However, nodes that record from an input need to synchronize to the audio input node
	//	instead for best results.
	//	MakeTimeSourceFor gives us a "clone" of the time source node that we can manipulate
	//	to our heart's content. When we're done with it, though, we need to call Release()
	//	on the time source node, so that it keeps an accurate reference count and can delete
	//	itself when it's no longer needed.
	// TODO: what about filters connected to audio input?
	media_node use_time_source;
	BTimeSource * tsobj = m_roster->MakeTimeSourceFor(input);
	if (! tsobj) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): couldn't clone time source from audio input node\n"));
		return B_MEDIA_BAD_NODE;
	}

	//	Apply the time source in effect to our own Node.
	err = m_roster->SetTimeSourceFor(m_recordNode->Node().node, tsobj->Node().node);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): couldn't set the sound recorder's time source\n"));
		tsobj->Release();
		return err;
	}

	//	Get a format, any format.
	media_format fmt;
	fmt.u.raw_audio = m_audioOutput.format.u.raw_audio;
	fmt.type = B_MEDIA_RAW_AUDIO;

	//	Tell the consumer where we want data to go.
	err = m_recordNode->SetHooks(RecordFile, NotifyRecordFile, this);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): couldn't set the sound recorder's hook functions\n"));
		tsobj->Release();
		return err;
	}

	//	Using the same structs for input and output is OK in BMediaRoster::Connect().
	err = m_roster->Connect(m_audioOutput.source, m_recInput.destination, &fmt, &m_audioOutput, &m_recInput);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakeRecordConnection(): failed to connect sound recorder to audio input node.\n"));
		tsobj->Release();
		m_recordNode->SetHooks(0, 0, 0);
		return err;
	}

	//	Start the time source if it's not running.
	if ((tsobj->Node() != input) && !tsobj->IsRunning()) {
		m_roster->StartNode(tsobj->Node(), BTimeSource::RealTime());
	}
	tsobj->Release();	//	we're done with this time source instance!
	return B_OK;
}

status_t
CaptureWindow::BreakRecordConnection()
{
	status_t err;

	//	If we are the last connection, the Node will stop automatically since it
	//	has nowhere to send data to.
	err = m_roster->StopNode(m_recInput.node, 0);
	err = m_roster->Disconnect(m_audioOutput.node.node, m_audioOutput.source, m_recInput.node.node, m_recInput.destination);
	m_audioOutput.source = media_source::null;
	m_recInput.destination = media_destination::null;
	return err;
}

status_t
CaptureWindow::StopRecording()
{
	m_recording = false;
	BreakRecordConnection();
	m_recordNode->SetHooks(0, 0, 0);
	if (m_recSize > 0) {
		//	Write attributes of newly created file.
		m_recFile.WriteAttr(SAMPLE_RATE_ATTR, B_FLOAT_TYPE, 0, &m_audioOutput.format.u.raw_audio.frame_rate, sizeof(float));
		m_recFile.WriteAttr(CHANNELS_ATTR, B_INT32_TYPE, 0, &m_audioOutput.format.u.raw_audio.channel_count, sizeof(int32));
		m_recFile.WriteAttr(FORMAT_ATTR, B_INT32_TYPE, 0, &m_audioOutput.format.u.raw_audio.format, sizeof(int32));
		//	Note that the "endian" attribute is different from the media_format value
		//	since it's taken from the "/dev/audio/raw" interface. Screwy but true.
		int32 endian = (m_audioOutput.format.u.raw_audio.byte_order == B_MEDIA_BIG_ENDIAN) ? 1 : 0;
		m_recFile.WriteAttr(ENDIAN_ATTR, B_INT32_TYPE, 0, &endian, sizeof(int32));
		int64 zero = 0;
		m_recFile.WriteAttr(LENGTH_ATTR, B_INT64_TYPE, 0, &zero, sizeof(int64));
		m_recFile.WriteAttr(LENGTH_ATTR, B_INT64_TYPE, 0, &m_recSize, sizeof(int64));
		m_recFile.WriteAttr(TYPE_ATTR, 'MIMS', 0, file_type, strlen(file_type)+1);
		m_recFile.SetSize(m_recSize);	//	We reserve space; make sure we cut off any excess at the end.
		AddSoundItem(m_recEntry, true);
	}
	else {
		m_recEntry.Remove();
	}
	//	We're done for this time.
	m_recEntry.Unset();
	//	Close the file.
	m_recFile.Unset();
	//	No more recording going on.
	m_recLimit = 0;
	m_recSize = 0;
	SetButtonState(btnPaused);
	return B_OK;
}


status_t
CaptureWindow::MakePlayConnection(const media_raw_audio_format & format)
{
	//	Find an available input for the mixer node.
	int32 count = 0;
	status_t err = m_roster->GetFreeInputsFor(m_audioMixerNode, &m_audioMixerInput, 1, &count, B_MEDIA_RAW_AUDIO);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): no free mixer inputs\n"));
		return err;
	}
	if (count < 1) {
		return B_BUSY;
	}

	//	Find an available output for our own Node. Note that we go through the
	//	MediaRoster; calling Media Kit methods directly on Nodes in our app is
	//	not OK (because synchronization happens in the service thread, not in
	//	the calling thread).
	// TODO: explain this last bit
	err = m_roster->GetFreeOutputsFor(m_playNode->Node(), &m_playOutput, 1, &count, B_MEDIA_RAW_AUDIO);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): no free play node outputs\n"));
		return err;
	}
	if (count < 1) {
		return B_BUSY;
	}

	//	Find out what the preffered (DAC) time source is.
	media_node tsnode;
	err = m_roster->GetTimeSource(&tsnode);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): no default time source: using system time source\n"));
		m_roster->GetSystemTimeSource(&tsnode);
	}

	//	Apply the time source in effect to our own Node.
	err = m_roster->SetTimeSourceFor(m_playNode->ID(), tsnode.node);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): set time source for play node failed\n"));
		return err;
	}

	// Create a media format with the raw audio format we were given.
	media_format fmt;
	fmt.u.raw_audio = format;
	fmt.type = B_MEDIA_RAW_AUDIO;

	//	Tell the producer where we want data to go.
	err = m_playNode->SetHooks(PlayFile, NotifyPlayFile, this);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): set hooks for play node failed\n"));
		return err;
	}

	//	Using the same structs for input and output is OK in BMediaRoster::Connect().
	err = m_roster->Connect(m_playOutput.source, m_audioMixerInput.destination, &fmt, &m_playOutput, &m_audioMixerInput);
	if (err < B_OK) {
		CONNECT((stderr, "CaptureWindow::MakePlayConnection(): failed to connect play node to mixer\n"));
		m_playNode->SetHooks(0, 0, 0);
		return err;
	}

	return B_OK;
}

status_t
CaptureWindow::BreakPlayConnection()
{
	CONNECT((stderr, "Breaking play connection...\n"));
	status_t err;

	if (m_playOutput.source == media_source::null
		|| m_audioMixerInput.destination == media_destination::null)
	{
		// someone's already done our dirty work
		CONNECT((stderr, "Play connection already broken.\n"));
		return B_OK;
	}

	// Stop the node so that it's no longer sending buffers,
	// then disconnect it from the mixer.
	CONNECT((stderr, "Stopping node...\n"));
	err = m_roster->StopNode(m_playNode->Node(), m_playNode->TimeSource()->Now(),
		true);
	CONNECT((stderr, "Disconnecting node...\n"));
	err = m_roster->Disconnect(m_playOutput.node.node, m_playOutput.source,
		m_audioMixerInput.node.node, m_audioMixerInput.destination);
	CONNECT((stderr, "Play connection broken.\n"));
	m_playOutput.source = media_source::null;
	m_audioMixerInput.destination = media_destination::null;
	return err;
}

status_t
CaptureWindow::StopPlaying()
{
	BreakPlayConnection();
	m_playNode->SetHooks(0, 0, 0);
	SetButtonState(btnPaused);
	return B_OK;	
}

void
CaptureWindow::SetButtonState(
	BtnState state)
{
	m_buttonState = state;
	UpdateButtons();
}

void
CaptureWindow::UpdateButtons()
{
	bool hasSelection = (m_soundList->CurrentSelection() >= 0);
	m_record->SetEnabled(m_buttonState == btnPaused);
	m_play->SetEnabled((m_buttonState == btnPaused) && hasSelection);
	m_stop->SetEnabled(m_buttonState != btnPaused);
	m_save->SetEnabled(hasSelection);
	m_vox->SetEnabled(m_buttonState != btnRecording);
	m_voxFrameSlider->SetEnabled(m_buttonState != btnRecording);
	m_voxThresholdSlider->SetEnabled(m_buttonState != btnRecording);
	m_length->SetEnabled(m_buttonState != btnRecording);
	m_input->SetEnabled(m_buttonState != btnRecording);
}

void
CaptureWindow::ErrorAlert(
	const char * action,
	status_t err)
{
	char msg[300];
	sprintf(msg, "Cannot %s: %s.\n[%lx]", action, strerror(err), (int32) err);
	(new BAlert("", msg, "Stop"))->Go();
}

status_t
CaptureWindow::NewTempName(
	char * name)
{
	int init_count = m_tempCount;
again:
	if (m_tempCount-init_count > 25) {
		return B_ERROR;
	}
	else {
		m_tempCount++;
		sprintf(name, "NewSound:%d", m_tempCount);
		BPath path;
		status_t err;
		BEntry tempEnt;
		if ((err = m_tempDir.GetEntry(&tempEnt)) < B_OK) {
			return err;
		}
		if ((err = tempEnt.GetPath(&path)) < B_OK) {
			return err;
		}
		path.Append(name);
		int fd;
		//	Use O_EXCL so we know we created the file (sync with other instances)
		if ((fd = open(path.Path(), O_RDWR | O_CREAT | O_EXCL, 0666)) < 0) {
			goto again;
		}
		close(fd);
	}
	return B_OK;
}

void
CaptureWindow::AddSoundItem(const BEntry& entry, bool temp)
{
	//	Create list item to display.
	SoundListItem * li = new SoundListItem(entry, temp);
	m_soundList->AddItem(li, 0);
}

void
CaptureWindow::RecordFile(
	void * cookie, 
	bigtime_t timestamp, 
	void * data,
	size_t size, 
	const media_raw_audio_format & format)
{
	//	Callback called from the SoundConsumer when receiving buffers.
	assert((format.format & 0x02) && format.channel_count);
	CaptureWindow * window = (CaptureWindow *)cookie;
	if (window->m_recOnVox) {
		uint32 voxSum = 0;
		int16* data16 = (int16*) data;
		for (uint32 i=0; i < window->m_voxFrames; i++) {
			int16 maxChannel = 0;
			for (uint32 j=0; j < format.channel_count; j++) {
				int16 val = abs(*(data16+(i*format.channel_count)+j));
				if (val > maxChannel) maxChannel = val;
			}
			voxSum += maxChannel;
		}
		if ((! window->m_recording) && (voxSum > window->m_voxThreshold)) {
			VOX((stderr, "CaptureWindow::RecordFile(): recording activated on vox sum = %lu\n", voxSum));
			window->m_recording = true;
		} else {
			VOX((stderr, "CaptureWindow::RecordFile(): vox sum = %lu\n", voxSum));
		}
	}

	if (window->m_recording) {
		//	Write the data to file (we don't buffer or guard file access
		//	or anything)
		if (window->m_recSize < window->m_recLimit) {
			window->m_recFile.WriteAt(window->m_recSize, data, size);
			window->m_recSize += size;
		}
		else {
			// We're done!
			window->PostMessage(STOP_RECORDING);
		}
	}
}

void
CaptureWindow::NotifyRecordFile(
	void * cookie,
	int32 code,
	...)
{
	if ((code == B_WILL_STOP) || (code == B_NODE_DIES)) {
		CaptureWindow * window = (CaptureWindow *)cookie;
		// Tell the window we've stopped, if it doesn't
		// already know.
		window->PostMessage(STOP_RECORDING);
	}
}


void
CaptureWindow::PlayFile(
	void * cookie,
	bigtime_t timestamp,
	void * data,
	size_t size,
	const media_raw_audio_format & format)
{
	//	Callback called from the SoundProducer when producing buffers.
	CaptureWindow * window = (CaptureWindow *)cookie;
	if (window->m_playSize < window->m_playLimit) {
		window->m_playFile.ReadAt(window->m_playSize, data, size);
		window->m_playSize += size;
	} else {
		//	we're done!
		window->PostMessage(STOP_PLAYING);
	}
}

void
CaptureWindow::NotifyPlayFile(
	void * cookie,
	int32 code,
	...)
{
	if ((code == B_WILL_STOP) || (code == B_NODE_DIES)) {
		CaptureWindow * window = (CaptureWindow *)cookie;
		// tell the window we've stopped, if it doesn't
		// already know.
		window->PostMessage(STOP_PLAYING);
	}
}

