/*******************************************************************************
/
/	File:			SoundListView.cpp
/
/   Description:	View for organizing and playing back sound clips.
/
/	Copyright 1998-1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <Application.h>
#include <Bitmap.h>
#include <Font.h>
#include <File.h>
#include <ScrollBar.h>
#include <fs_attr.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SoundListView.h"


SoundListView::SoundListView(
	const BRect & area,
	const char * name,
	uint32 resize) :
	BListView(area, name, B_SINGLE_SELECTION_LIST, resize)
{
}

SoundListView::~SoundListView()
{
}

void
SoundListView::Draw(BRect updateRect)
{
	BListView::Draw(updateRect);
	
	// draw the drag focus rect if necessary. The list view
	// relies on its surrounding scroll view to draw the
	// keyboard focus, but drawing the focus looks a lot
	// better when it's done this way.
	if (DragFocus()) {
		SetHighColor(ui_color(B_KEYBOARD_NAVIGATION_COLOR));
		SetPenSize(1.0f);
		StrokeRect(Bounds());
	}
}

void
SoundListView::MouseDown(
	BPoint position)
{
	BListView::MouseDown(position);
}

void
SoundListView::MouseMoved(
	BPoint where, uint32 transit, const BMessage * dragMsg)
{
	if (dragMsg && (dragMsg->what == B_SIMPLE_DATA))
	{
		// something's being dragged
		SetDragFocus((transit == B_ENTERED_VIEW) || (transit == B_INSIDE_VIEW));
	}
}

void
SoundListView::FrameResized(
	float width,
	float height)
{
	BListView::FrameResized(width, height);
	Calibrate(width, height);
}

void
SoundListView::AttachedToWindow()
{
	BListView::AttachedToWindow();
	Calibrate(Frame().Width(), Frame().Height());
}

bool
SoundListView::AddItem(
	BListItem * item)
{
	bool ret = BListView::AddItem(item);
	Calibrate(Frame().Width(), Frame().Height());
	return ret;
}

bool
SoundListView::AddItem(
	BListItem * item,
	int32 index)
{
	bool ret = BListView::AddItem(item, index);
	Calibrate(Frame().Width(), Frame().Height());
	return ret;
}

void
SoundListView::Calibrate(
	float width,
	float height)
{
	// Fixup the scroll bars.
	//
	// Note: our calculations here make dangerous assumptions
	// about the list item. This should be cleaner.
	float wid = 0;
	float hei = 0;
	float itemHei = 0;
	for (int ix=0; ix<CountItems(); ix++) {
		SoundListItem * item = dynamic_cast<SoundListItem *>(ItemAt(ix));
		if (item) {
			// wid of list view is the width of the longest bitmap item
			if (item->m_bitmap->Bounds().Width() > wid) {
				wid = item->m_bitmap->Bounds().Width() + 6;
			}
			// hei is the sum of sound list item heights
			itemHei = item->m_bitmap->Bounds().Height()+4;
			hei += itemHei-1;
		}
	}
	
	float babySteps, giantSteps;
	babySteps = 8.0f;
	giantSteps = (width/8 > 32) ? width/8 : 32;
	ScrollBar(B_HORIZONTAL)->SetSteps(babySteps, giantSteps);
	if (wid > width) {
		ScrollBar(B_HORIZONTAL)->SetRange(0, wid-width);
		ScrollBar(B_HORIZONTAL)->SetProportion(width/wid);
	}
	else {
		ScrollBar(B_HORIZONTAL)->SetRange(0, 0);
	}
	// for steps, calculate steps to an integral item height,
	// assuming all items are the same height
	babySteps = itemHei-1;
	giantSteps = 4*babySteps;
	ScrollBar(B_VERTICAL)->SetSteps(babySteps, giantSteps);
	if (hei > height) {
		ScrollBar(B_VERTICAL)->SetRange(0, hei-height);
	}
	else {
		ScrollBar(B_VERTICAL)->SetRange(0, 0);
	}
}

void
SoundListView::MessageReceived(BMessage* message)
{
   	entry_ref ref;
 	switch ( message->what )	{
   		case B_SIMPLE_DATA:
   		{
   			int32 i=0;
   			BMessage appMsg(B_REFS_RECEIVED);
   			// Look for any refs in the message, and pass
   			// them on to the app for it to handle
   			while( message->FindRef("refs", i++, &ref) == B_OK ){
				appMsg.AddRef("refs", &ref);
			}
			if (i > 1) {
				be_app->PostMessage(&appMsg);
			} else {
				BListView::MessageReceived(message);
			}
			SetDragFocus(false);
   			break;
   		}
   		default:
   			// Call inherited if we didn't handle the message
   			BListView::MessageReceived(message);
   			break;
	}
}

void
SoundListView::SetDragFocus(bool focus)
{
	if (m_bDragFocus != focus) {
		m_bDragFocus = focus;
		Invalidate(Bounds());
	}
}

bool
SoundListView::DragFocus() const
{
	return m_bDragFocus;
}








SoundListItem::SoundListItem(
	const BEntry & entry,
	bool isTemp) : m_entry(entry)
{
	m_bitmap = NULL;
	m_isTemp = isTemp;
	MakeBitmap();
	m_height = m_bitmap->Bounds().Height() + 2.0f;
	m_baseline = m_height * 0.75f;
}

SoundListItem::~SoundListItem()
{
	delete m_bitmap;
}

void
SoundListItem::DrawItem(
	BView *owner,
	BRect bounds,
	bool complete)
{
	owner->SetDrawingMode(B_OP_COPY);
	if (IsSelected()) {
		owner->SetHighColor(0,0,229);
		owner->FillRect(bounds);
	}
	else {
		owner->SetHighColor(240, 240, 240);
		owner->FillRect(bounds);
	}
	owner->SetHighColor(0,0,0);
	BPoint pt(bounds.LeftTop());
	pt.x += 3;
	pt.y += 1;
	owner->DrawBitmapAsync(m_bitmap, pt);
	pt.y = m_baseline+bounds.top-1;
	pt.x += 6;
	char name[256];
	m_entry.GetName(name);
	owner->SetLowColor(240, 240, 240);
	BRect r(pt.x-2, pt.y-m_above-1, pt.x+owner->StringWidth(name)+1, pt.y+m_below+1);
	owner->FillRect(r, B_SOLID_LOW);
	owner->SetHighColor(128,64,64);
	owner->DrawString(name, pt);
	owner->SetHighColor(128,128,128);
	owner->StrokeRect(r);
}

void
SoundListItem::Update(
	BView *owner,
	const BFont * font)
{
	font_height fh;
	font->GetHeight(&fh);
	m_baseline = fh.ascent + fh.leading + 1;
	m_above = fh.ascent + fh.leading;
	m_below = fh.descent;
	m_height = m_baseline + fh.descent + 1;
	float bitHeight = m_bitmap->Bounds().Height() + 2;
	if (m_height < bitHeight) {
		m_baseline += (int)((bitHeight-m_height)*3/4);
		m_height = bitHeight;
	}
	SetHeight(m_height);
}

void
SoundListItem::MakeBitmap()
{
	//	This reads in an existing set of preview data,
	//	or creates it if it doesn't already exist,
	//	and creates a bitmap to represent the preview
	//	data.
	BFile file(&m_entry, O_RDONLY);
	if (file.InitCheck()) {
		m_bitmap = new BBitmap(BRect(0,0,255,31), B_COLOR_8_BIT, true);
		BView * v = new BView(m_bitmap->Bounds(), "error", B_FOLLOW_NONE, B_WILL_DRAW);
		m_bitmap->Lock();
		m_bitmap->AddChild(v);
		v->SetFont(be_bold_font);
		v->DrawString("Not found", BPoint(50,24));
		v->Sync();
		v->RemoveSelf();
		m_bitmap->Unlock();
		delete v;
		return;				//	error
	}
	attr_info info;
	uchar * data = NULL;
	const size_t max_samps = 1024;
	//	We assume 16 bit linear data.
	if (file.GetAttrInfo("be:AUDIO:preview", &info) < B_OK) {
		off_t size = file.Seek(0, SEEK_END); // the length of the file
		int div = 2048; // the default nybble length in bytes = 1024 samples
						// = 512 frames (in stereo)
		// Calculate the number of nybbles we'll be taking -- 0x1000 (4096) max.
		if (size/div > 0x1000) {
			div = size / 0x1000;	
		}
		int num_nybs = size/div;
		
		// data stores each nybble as an array of nybbles (i.e. 4-bit values).
		// Zero out the data to be safe.
		data = (uchar *)malloc((num_nybs+1)/2);
		memset(data, 0, (num_nybs+1)/2);
		
		// samp stores each chunk of nybble data as we read them in.
		// we only store up to max_samps at a time, but we always step
		// through the file div at a time, even if div is bigger.
		int16* samp = new int16[max_samps];
		off_t o; // current position in file
		int16 nyb_pos = 0;	// current nybble number
		fprintf(stderr, "creating preview...\n");
		for (o = 0; o < size; o += div) {
			// Zero out the data in case we're at the end of the file
			// and not all of the array gets read.
			memset(samp, 0, max_samps*sizeof(int16));
			file.ReadAt(o, samp, max_samps*sizeof(int16));
						
			// get the min and max values in the nibbling interval
			// and set max to be the greater of the absolute value
			// of these.
			int mi = 0, ma = 0;
			for (uint32 ix=0; ix<max_samps; ix++) {
				if (mi > samp[ix]) mi = samp[ix];
				else if (ma < samp[ix]) ma = samp[ix];
			}
			if (-ma > mi) ma = (mi == -32768) ? 32767 : -mi;
			
			// compute n = floor(log2(ma))
			uchar n = 0;
			for (int ix=15; ix>0; ix--) {
				if (ma & (1 << ix)) {
					n = ix;
					break;
				}
			}
			if (!(nyb_pos & 1)) {
				// odd values are stored in high nybble.
				data[nyb_pos/2] |= (n << 4);
			}
			else {
				// even values are stored in low nybble.
				data[nyb_pos/2] |= n;
			}
			nyb_pos++;
		}
		// We're done creating the preview data. Write the preview data
		// as an attribute so it can be read by other apps in the future.
		info.size = (num_nybs+1)/2;
		file.WriteAttr("be:AUDIO:preview", B_RAW_TYPE, 0, data, info.size);
		delete [] samp;
	}
	else {
		data = (uchar *)malloc(info.size);
		file.ReadAttr("be:AUDIO:preview", info.type, 0, data, info.size);
	}
	
	//	Make the bitmap big enough to encompass the data and a nice frame
	//	1 deep on each side.
	m_bitmap = new BBitmap(BRect(0,0,info.size*2+1,32), B_COLOR_8_BIT);
	
	//	Clear to light grey -- remember, we're using 8-bit palette index values!
	memset(m_bitmap->Bits(), 28, m_bitmap->BitsLength());
	uchar * base = (uchar *)m_bitmap->Bits();
	int rb = m_bitmap->BytesPerRow();
	
	// Draw the frame.
	memset(base, 63, rb);		// top row is white
	memset(base+32*rb, 16, rb); // bottom row is medium grey
	for (int iy=0; iy<33; iy++) {
		base[iy*rb] = 63; 		// leftmost column is white
		base[iy*rb+rb-1] = 16;	// rightmost column is medium grey
	}

	// Draw the data.
	// Note: in my kube, the noise floor is pretty high, ~ 9 bits or so,
	// so this preview drawing generally looks like a big black stripe
	// with slight variations. If this weren't going to be used to record
	// high-quality sound, it might be a good idea to truncate the lowest
	// several bits when drawing, so that the drawing is more compacted
	// and looks better.
	base += rb + 1; // start at (1, 1)
	for (int ix=0; ix<info.size*2; ix++) {
		int nyb;
		if (ix & 1) {
			// index is odd -- retrieve value from low nybble
			nyb = data[ix/2] & 15;
		}
		else {
			// index is even -- retrieve value from high nybble
			nyb = data[ix/2] >> 4;
		}
		for (int iy = 15-nyb; iy < 16+nyb; iy++) {
			// Draw a vertical black line, extending from the
			// baseline nyb units in both directions.
			base[rb * iy] = 0;
		}
		// increment base to go to next column
		base++;
	}
	free(data);
}
