/*******************************************************************************
/
/	File:			ArgvHandler.cpp
/
/   Description:	A simple class for processing argv's.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <stdlib.h>
#include <Errors.h>
#include "ArgvHandler.h"

void ArgvHandler::Init(int32 argc, char** argv, int32 initPos)
{
	m_argc = argc;
	m_argv = argv;
	m_curPos = initPos;
}

status_t ArgvHandler::NextArg(char** arg)
{
	status_t err;
	if (! check_range(m_curPos)) {
		err = B_BAD_INDEX;
	} else {
		*arg = *(m_argv + (m_curPos++));
		err = B_OK;
	}
	return err;
}

status_t ArgvHandler::PrevArg(char** arg)
{
	status_t err;
	if (! check_range(m_curPos-1)) {
		err = B_BAD_INDEX;
	} else {
		*arg = *(m_argv + (--m_curPos));
		err = B_OK;
	}
	return err;
}

status_t ArgvHandler::NextArg(int32* val)
{
	status_t err;
	char* next;

	err = NextArg(&next);
	if (err == B_OK) {
		err = ConvertToInt32(next, val);
		if (err != B_OK)
			PrevArg(&next);
	}
	
	return err;
}

status_t ArgvHandler::PrevArg(int32* val)
{
	status_t err;
	char* prev;

	err = PrevArg(&prev);
	if (err == B_OK) {
		err = ConvertToInt32(prev, val);
		if (err != B_OK)
			NextArg(&prev);
	}
	
	return err;
}

status_t ArgvHandler::NextArg(int8* val)
{
	int32 val32;
	status_t err = NextArg(&val32);
	if (err == B_OK) {
		if (val32 > 127L || val32 < -128L)
			err = ERANGE;
		else
			*val = static_cast<int8>(val32);
	}
	return err;
}

status_t ArgvHandler::PrevArg(int8* val)
{
	int32 val32;
	status_t err = PrevArg(&val32);
	if (err == B_OK) {
		if (val32 > 127L || val32 < -128L)
			err = ERANGE;
		else
			*val = static_cast<int8>(val32);
	}
	return err;
}

status_t ArgvHandler::NextArg(double* val)
{
	status_t err;
	char* next;

	err = NextArg(&next);
	if (err == B_OK) {
		err = ConvertToDouble(next, val);
		if (err != B_OK)
			PrevArg(&next);
	}
	
	return err;
}

status_t ArgvHandler::PrevArg(double* val)
{
	status_t err;
	char* prev;

	err = PrevArg(&prev);
	if (err == B_OK) {
		err = ConvertToDouble(prev, val);
		if (err != B_OK)
			NextArg(&prev);
	}
	
	return err;
}

status_t ArgvHandler::ConvertToInt32(char* arg, int32* val)
{
	status_t err;
	char* trailing;
	*val = strtol(arg, &trailing, 0);
	if (trailing == arg || (*trailing != '\0'))
		err = (*val == 0) ? B_BAD_VALUE : ERANGE;
	else
		err = B_OK;

	return err;
}

status_t ArgvHandler::ConvertToDouble(char* arg, double* val)
{
	status_t err;
	char* trailing;
	*val = strtod(arg, &trailing);
	if (trailing == arg || (*trailing != '\0'))
		err = (*val == 0.0) ? B_BAD_VALUE : ERANGE;
	else
		err = B_OK;

	return err;	
}
