/*******************************************************************************
/
/	File:			SelectorNode.h
/
/   Description:	A node that allows the user to select one of
/					several inputs to route to the output.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#ifndef _SelectorNode_h
#define _SelectorNode_h

// the number of possible input connections is currently hard-coded
#define MAX_SELECTORS 64

// this is the message protocol for initializing the node.
// To configure the node, write a message to the node's port,
// where what = SELECTOR_INIT and the data is the raw
// selector_init_struct.
const int32 SELECTOR_INIT = 0x60005000;
struct selector_init_struct {
	bool controls;
	int32 initSel;
};

#if GENKI_4
// genki/4 didn't have BMediaEventLooper. If you have Stephen's
// MotherNode article, though, it's a quite servicable substitute.
#include "MotherNode.h"
#define BMediaEventLooper MotherNode
#else
#include <MediaEventLooper.h>
#endif

#include <BufferProducer.h>
#include <BufferConsumer.h>
#include <Controllable.h>

class SelectorNode : public BMediaEventLooper, public BBufferProducer,
	public BBufferConsumer, public BControllable
{
public:
	SelectorNode(const char* name, BMediaAddOn* addon, int32 flavor_id);
	~SelectorNode();

// BMediaEventLooper methods
protected:
	virtual void HandleEvent(bigtime_t performance_time, int32 what, const void* pointer, uint32 cleanup, int64 data);

// BMediaNode methods
public:
	virtual BMediaAddOn* AddOn(int32* out_internal_id) const
	{ *out_internal_id = m_flavorID; return m_addon; }
	virtual status_t HandleMessage(int32 what, const void* data, size_t size);	
	virtual void NodeRegistered();
	
// BBufferProducer methods
public:
	virtual status_t FormatSuggestionRequested(media_type type, int32 quality, media_format* format);
	virtual status_t FormatProposal(const media_source& what, media_format* format);
	virtual status_t FormatChangeRequested(const media_source& what, const media_destination& where, media_format* io_format, int32* _deprecated_);
	virtual status_t GetNextOutput(int32* cookie, media_output* out_output);
	virtual status_t DisposeOutputCookie(int32 cookie);
	virtual status_t SetBufferGroup(const media_source& what, BBufferGroup* group);
	virtual status_t PrepareToConnect(const media_source& what, const media_destination& where, media_format* format, media_source* out_source, char* out_name);
	virtual void Connect(status_t error, const media_source& source, const media_destination& where, const media_format& format, char* io_name);
	virtual void Disconnect(const media_source& what, const media_destination& where);
	virtual void LateNoticeReceived(const media_source& what, bigtime_t how_much, bigtime_t tpAt);
	virtual void EnableOutput(const media_source& what, bool enabled, int32* _deprecated_);
	
// BBufferConsumer methods
public:
	virtual status_t AcceptFormat(const media_destination& where, media_format* format);
	virtual status_t GetNextInput(int32* cookie, media_input* out_input);
	virtual void DisposeInputCookie(int32 cookie);
	virtual void BufferReceived(BBuffer* buffer);
	virtual void ProducerDataStatus(const media_destination& where, int32 status, bigtime_t performance_time);
	virtual status_t GetLatencyFor(const media_destination& where, bigtime_t* out_latency, media_node_id* out_timesource);
	virtual status_t Connected(const media_source& what, const media_destination& where, const media_format& with, media_input* out_input);
	virtual void Disconnected(const media_source& what, const media_destination& where);
	virtual status_t FormatChanged(const media_source& what, const media_destination& where, int32 _deprecated_, const media_format& format);

// BControllable methods
protected:
	virtual status_t GetParameterValue(int32 id, bigtime_t* last_change, void* value, size_t* ioSize);
	virtual void SetParameterValue(int32 id, bigtime_t when, const void* value, size_t size);

#if GENKI_4
	// The latency change mechanism is new for genki/4. Fortunately, it's
	// easy to implement in genki/4.
	void SendLatencyChange(const media_source& what, const media_destination& where, bigtime_t new_latency, uint32 flags=0);
	status_t StartControlPanel(BMessenger* msgr);
#endif

// SelectorNode methods
private:
	// The ID of the parameter that selects the input -- it also doubles
	// as the ID for the associated media event.
	static const int32 INPUT_SELECTED;
	
	void Init();
	void UpdateWeb();
	void UpdateThreadPriority();
	void UpdateLatency();
	
	media_input* InputAt(const media_destination& where);
	bool IsConnected() const
	{ return (m_output.destination != media_destination::null); }
	bool check_selector_range(int32 index)
	{ return (index >= 0 && index < MAX_SELECTORS); }
	
	bigtime_t ProcessingLatency() const { return m_processingLatency; }
	void SetProcessingLatency(bigtime_t latency) { m_processingLatency = latency; }
	
	media_format* Format() const { return m_format; }
	void SetFormat(const media_format* format);

// SelectorNode data
private:
	media_format* m_format;
	media_input m_inputs[MAX_SELECTORS];
	int32 m_status[MAX_SELECTORS];
	int32 m_selectedInput;
	bigtime_t m_tpInputSelected;
	media_output m_output;
	bool m_enabled;
	BMediaAddOn* m_addon;
	int32 m_flavorID;
	bigtime_t m_processingLatency;
};

#endif /* _SelectorNode_h */