/*******************************************************************************
/
/	File:			utilfunctor.h
/
/   Description:	Useful functors, binders, and adapters for working with STL.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#ifndef _utilfunctor_h
#define _utilfunctor_h

// trinary_function is like unary_function and binary_function;
// it provides an STL-friendly way to specify functions that take
// three arguments.
template <class Arg1, class Arg2, class Arg3, class Result>
struct trinary_function {
	typedef Arg1 first_argument_type;
	typedef Arg2 second_argument_type;
	typedef Arg3 third_argument_type;
	typedef Result result_type;
};

// binder23 is an object that allows you to call a 3-argument function
// with a 1-argument function:
// * In the ctor, you pass it a trinary_function, and tell it what you
//   want the 2nd and 3rd arguments to be bound to.
// * It gets invoked with 1 argument, and calls the 3-argument function,
//   binding argument 1 to whatever argument it is invoked with, and
//   binding arguments 2 and 3 to whatever you specified in the ctor.
//
// Compare to binder2nd and binder1st in <functional>.
template <class Op>
class binder23 : public unary_function<typename Op::first_argument_type,
	typename Op::result_type>
{
protected:
	Op op;
	typename Op::second_argument_type v1;
	typename Op::third_argument_type v2;
	
public:
	binder23(const Op& x, const typename Op::second_argument_type& y, const typename Op::third_argument_type& z)
		: op(x), v1(y), v2(z) {}
	typename Op::result_type operator() (const typename Op::first_argument_type& x) const
	{ return op(x, v1, v2); }
};

// bind23 is a function that creates a binder23 for you. Compare to bind2nd
// and bind1st in <functional>.
template <class Op, class Arg1, class Arg2> binder23<Op>
inline bind23(const Op& op, const Arg1& v1, const Arg2& v2)
{ return binder23<Op>(op, v1, v2); }

// mem_fun2_ref_t represents a member function of T that takes two arguments.
// You invoke it with a reference to an object of type T, and it invokes that
// object's member function with the given arguments. Compare to
// mem_fun1_ref_t in <functional>.
template <class Result, class T, class Arg1, class Arg2> 
class mem_fun2_ref_t : public trinary_function<T, Arg1, Arg2, Result>
{
public:
	explicit mem_fun2_ref_t(Result (T::*f)(Arg1, Arg2)) : m_f(f) {}
	Result operator() (T& obj, Arg1 arg1, Arg2 arg2) const
	{ return (obj.*m_f)(arg1, arg2); }

private:
	Result (T::*m_f)(Arg1, Arg2);
};

// const_mem_fun2_ref_t represents a const member function of T that takes
// two arguments. Pretty much the same as above.
template <class Result, class T, class Arg1, class Arg2>
class const_mem_fun2_ref_t : public trinary_function<T, Arg1, Arg2, Result>
{
public:
	explicit const_mem_fun2_ref_t(Result (T::*f)(Arg1, Arg2) const) : m_f(f) {}
	Result operator() (const T& obj, Arg1 arg1, Arg2 arg2) const
	{ return (obj.*m_f)(arg1, arg2); }

private:
	Result (T::*m_f)(Arg1, Arg2) const;
};

// Overloading mem_fun_ref to handle member functions with two arguments.
// The semantics are exactly the same as the other mem_fun_ref variants
// defined in <functional>.
template <class Result, class T, class Arg1, class Arg2>
inline mem_fun2_ref_t<Result, T, Arg1, Arg2> mem_fun_ref(Result (T::*f)(Arg1, Arg2))
{ return mem_fun2_ref_t<Result, T, Arg1, Arg2>(f); }

template <class Result, class T, class Arg1, class Arg2>
inline const_mem_fun2_ref_t<Result, T, Arg1, Arg2> mem_fun_ref(Result (T::*f)(Arg1, Arg2) const)
{ return const_mem_fun2_ref_t<Result, T, Arg1, Arg2>(f); }

// A simple function to delete a pointer, which could be used in
// transform to mop up lists of pointers.
template <class T> T* delete_ptr(T* p) { delete p; return 0; }

#endif /* _utilfunctor_h */
