/*******************************************************************************
/
/	File:			TestBench.cpp
/
/   Description:	A generic framework for testing a media node.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <MediaAddOn.h>
#include <MediaNode.h>
#include <MediaRoster.h>
#include <ParameterWeb.h>
#include <TimeSource.h>
#include <Debug.h>
#include <string.h>
#include <functional>
#include <algorithm>
#include <Rect.h>
#include "utilfunctor.h"
#include "Connection.h"
#include "TestBench.h"

TestBench::TestBench()
{
	m_bRunning = false;
}

TestBench::~TestBench()
{
}

status_t TestBench::Start(const media_node& node, int32 what, const void* data, size_t size)
{
	PRINT(("TestBench::Start %ld\n", node.node));
	if (m_bRunning)
		// can't start an already-running test!
		return B_ERROR;
		
	m_testNode = node;

	// Connect everything together first.
	status_t err;
	err = Hookup();
	if (err == B_OK) {
		// Initialize the node by sending it a message.
		err = write_port(node.port, what, data, size);
		if (err == B_OK) {
			// Go ahead and start everything.
			err = StartNodes();
			m_bRunning = (err == B_OK);
		}
	}
	
	if (err != B_OK) {
		// Something went wrong -- stop and tear everything down.
		Stop();
	}	
	return err;
}

void TestBench::Stop()
{
	PRINT(("TestBench::Stop\n"));
	if (m_bRunning) {
		// Stop all of the nodes.
		StopNodes();
		m_bRunning = false;
	}
	
	// Disconnect and release all of the nodes.
	TearDown();
}

void TestBench::AddInput(const media_node& node, bool lock)
{
	PRINT(("TestBench::AddInput %ld\n", node.node));
	m_inputNodes.push_back(MediaNodeWrapper(node, lock));
}

void TestBench::AddOutput(const media_node& node, bool lock)
{
	PRINT(("TestBench::AddOutput %ld\n", node.node));
	m_outputNodes.push_back(MediaNodeWrapper(node, lock));
}

status_t TestBench::Hookup()
{
	PRINT(("TestBench::Hookup()\n"));

	// For the master time source of our entire chain,
	// we always use the time source that the first
	// output is slaved to.
	ASSERT(m_outputNodes.size() >= 1);
	MediaNodeWrapper masterOutput = m_outputNodes[0];
	m_testNode.SlaveTo(masterOutput);	

	// Clear all input connections.
	for (uint i=0; i<m_inputs.size(); i++)
		m_inputs[i] = delete_ptr(m_inputs[i]);
	m_inputs.clear();

	
	// Connect every input node to the test node, and stash the newly created
	// Connection objects in m_inputs.		
	transform(m_inputNodes.begin(), m_inputNodes.end(), back_inserter(m_inputs),
		bind2nd(ptr_fun(&Connection::Make), m_testNode));	
	
	// Slave every input node to the test node's time source.
	for_each(m_inputNodes.begin(), m_inputNodes.end(),
		bind2nd(mem_fun_ref(&MediaNodeWrapper::SlaveTo), m_testNode));

	// Clear all output connections.	
	for (uint i=0; i<m_outputs.size(); i++)
		m_outputs[i] = delete_ptr(m_outputs[i]);
	m_outputs.clear();
	
	// Connect the test node to every output node (this time the test node
	// is the upstream node), and stash the newly created Connections into
	// m_outputs.
	transform(m_outputNodes.begin(), m_outputNodes.end(), back_inserter(m_outputs),
		bind1st(ptr_fun(&Connection::Make), m_testNode));

	// Slave every output node to the test node's time source.
	// (Remember that SlaveTo has no effect on locked nodes).
	for_each(m_outputNodes.begin(), m_outputNodes.end(),
		bind2nd(mem_fun_ref(&MediaNodeWrapper::SlaveTo), m_testNode));
			
	PRINT(("Done with TestBench::Hookup()\n"));

	return B_OK;
}

status_t TestBench::StartNodes()
{
	// Figure out the soonest time we can start. This depends on the
	// total latency of our system (plus a little bit of fudge).
	BTimeSource* ts = m_testNode.GetTimeSource();
	ASSERT(ts);
	bigtime_t latency = ChainLatency();
	bigtime_t tpStart;
	tpStart = ts->Now() + latency + 1000;
	ts->Release();
	
	// Tell all the downstream nodes to start at the given time.
	for_each(m_outputNodes.begin(), m_outputNodes.end(),
		bind2nd(mem_fun_ref(&MediaNodeWrapper::Start), tpStart));
	
	// Now start the test node.
	m_testNode.Start(tpStart);

	// Finally, tell all the upstream nodes to start at the given time.
	for_each(m_inputNodes.begin(), m_inputNodes.end(),
		bind2nd(mem_fun_ref(&MediaNodeWrapper::Start), tpStart));
	
	return B_OK;
}

status_t TestBench::StopNodes()
{
	// Figure out the soonest time we can stop. Again, this depends on
	// the total latency of our system.
	BTimeSource* ts = m_testNode.GetTimeSource();
	ASSERT(ts);
	bigtime_t latency = ChainLatency();
	bigtime_t tpStop;
	tpStop = ts->Now() + latency + 1000;
	ts->Release();
	
	// Stop all the upstream nodes first.
	for_each(m_inputNodes.begin(), m_inputNodes.end(),
		bind23(mem_fun_ref(&MediaNodeWrapper::Stop), tpStop, false));
		
	// Then stop the test node.
	m_testNode.Stop(tpStop, false);
	
	// Finally, stop the downstream nodes.
	for_each(m_outputNodes.begin(), m_outputNodes.end(),
		bind23(mem_fun_ref(&MediaNodeWrapper::Stop), tpStop, false));

	return B_OK;
}

status_t TestBench::TearDown()
{
	// Clean up the connections.
	for (uint i=0; i<m_inputs.size(); i++)
		m_inputs[i] = delete_ptr(m_inputs[i]);
	for (uint i=0; i<m_outputs.size(); i++)
		m_outputs[i] = delete_ptr(m_outputs[i]);
	m_inputs.clear();
	m_outputs.clear();
	
	// Clean up and release the nodes
	for_each(m_inputNodes.begin(), m_inputNodes.end(),
		mem_fun_ref(&MediaNodeWrapper::Release));
	for_each(m_outputNodes.begin(), m_outputNodes.end(),
		mem_fun_ref(&MediaNodeWrapper::Release));
	m_inputNodes.clear();
	m_outputNodes.clear();
	
	m_testNode.Release();
	
	return B_OK;
}

// This is a simple helper function that is used to calculate the maximum
// latency of a list of nodes.
static void CollectMaxLatency(MediaNodeWrapper node, bigtime_t* max_latency)
{
	bigtime_t latency;
	status_t err = node.GetLatency(&latency);
	if (err == B_OK && latency > *max_latency)
		*max_latency = latency;
}

// This figures out the latency of the entire system by finding the
// worst-case latency among the input nodes.
bigtime_t TestBench::ChainLatency()
{
	bigtime_t max_latency = 0;
	for_each(m_inputNodes.begin(), m_inputNodes.end(),
		bind2nd(ptr_fun(CollectMaxLatency), &max_latency));
	return max_latency;
}
