/*******************************************************************************
/
/	File:			TestApp.h
/
/   Description:	A simple application that tests the Selector Node.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <Debug.h>
#include <File.h>
#include <MediaRoster.h>
#include <MediaAddOn.h>
#include <Button.h>
#include <Window.h>
#include "TestApp.h"
#include "TestBench.h"
#include "SoundFileReader.h"
// This header defines the message protocol
// for initializing the Selector node.
#include "../Selector_addon/SelectorNode.h"

TestApp::TestApp()
 : BApplication("application/x-vnd.Be-DTS.Selector_test")
{
	m_bHelp = false;
	m_bControls = false;
	m_initSel = 0;
	m_bHasFiles = false;
	m_bInitOK = false;
}

void TestApp::ReadyToRun()
{
	status_t err = B_ERROR;
	if (m_bHelp) {
		PrintHelp();
	} else if (! m_bHasFiles) {
		printf("No input sound files specified.\nTry 'Selector_test --help' for more information.\n");
	} else {
		StartTestWindow();
		err = StartTestNode();
	}
	
	if (err != B_OK)
		PostMessage(B_QUIT_REQUESTED);
}

void TestApp::ArgvReceived(int32 argc, char** argv)
{
	m_argv.Init(argc-1, argv+1);
	while (HandleArg());
}

void TestApp::Quit()
{
	PRINT(("TestApp::Quit\n"));
	m_tester.Stop();
	BApplication::Quit();
}

bool TestApp::HandleArg()
{
	status_t err;
	char* arg;
	if (m_argv.NextArg(&arg) != B_OK)
		return false;
	
	ASSERT(arg);
	
	if (! strncmp(arg, "--", 2)) {
		char* option = arg+2;
		if (! strcmp(option, "help")) {
			m_bHelp = true;
			return false;
		} else if (! strcmp(option, "controls")) {
			m_bControls = true;
		} else if (! strcmp(option, "selection")) {
			err = m_argv.NextArg(&m_initSel);
			if (err != B_OK) {
				fprintf(stderr, "Bad selection option: %s\n", strerror(err));
				m_bHelp = true;
				return false;
			}
		}
	} else {
		err = AddReader(arg);
		if (err == B_OK) {
			m_bHasFiles = true;
		} else {
			fprintf(stderr, "Not a valid sound file: %s\n", arg);
		}
	}	
	return true;
}

void
TestApp::PrintHelp()
{
	printf("\nApplication: Selector_test\n");
	printf("Tests the Selector media node, which allows you to select\n");
	printf("which one of a set of inputs you'd like to use.\n\n");
	printf("Usage: Selector_test {<option>}* {<input_file>}+\n");
	printf("%-29s%-25s\n", "<input_file>", "The name of an audio file (AIFF, WAV, AU)");
	printf("%-29s%-25s\n", "<option>", "One of the following:");
	printf("%4s%-25s%-25s\n", "", "--help", "Display this help and exit");
	printf("%4s%-25s%-25s\n", "", "--controls", "Display the selector control panel");
	printf("%4s%-25s%-25s\n", "", "--selection <sel_num>", "Set the initial selection");
	printf("%8s%-21s%25s\n", "", "<sel_num>", "The position of the file you want to select");
	printf("\nWhile running, press 'q' to quit.\n");	
}

status_t
TestApp::StartTestNode()
{
	media_node testNode;
	dormant_node_info dni;
	int32 count = 1;
	status_t err;

	// create the test node and tell the TestBench to start testing it...	
	err = BMediaRoster::Roster()->GetDormantNodes(&dni, &count, 0, 0, "Selector");
	if (err == B_OK) {
		err = BMediaRoster::Roster()->InstantiateDormantNode(dni, &testNode);
		if (err == B_OK) {
			AddOutput();
			selector_init_struct d = { m_bControls, m_initSel };
			err = m_tester.Start(testNode, SELECTOR_INIT, &d, sizeof(selector_init_struct));
		}
	}
	return err;
}

status_t
TestApp::AddReader(const char* filename)
{
	PRINT(("TestApp::AddReader(\"%s\")\n", filename));
	entry_ref ref;
	
	// Create a node to read this file, and tell the TestBench to
	// use this node as an input.
	status_t err = get_ref_for_path(filename, &ref);
	BMediaRoster::Roster(); // need to make sure the roster's instantiated
	if (err == B_OK) {
		SoundFileReader* reader = new SoundFileReader(filename);
		err = BMediaRoster::Roster()->RegisterNode(reader);
		if (err == B_OK) {
			err = reader->SetRef(&ref);
			if (err == B_OK) {
				media_node node;
				// Since we want to deal with the node through the Media
				// Roster, we should go to the Media Roster to get a
				// handle for this node, instead of just querying the
				// SoundFileReader directly. This will make sure the
				// reference counts in the Media Roster are maintained
				// properly. 
				BMediaRoster::Roster()->GetNodeFor(reader->Node().node, &node);
				m_tester.AddInput(node);
			} else {
				reader->Release();
			}
		}
	}
	return err;
}

status_t
TestApp::AddOutput()
{
	PRINT(("TestApp::AddOutput\n"));
	media_node node;
	
	// Tell the TestBench to use the audio mixer as its output.
	status_t err = BMediaRoster::Roster()->GetAudioMixer(&node);
	if (err == B_OK) {
		m_tester.AddOutput(node, true);
	}
	return err;
}



// TestWin is a cheesy little window with a "Stop" button in it.
// When pressed, it quits the application.
class TestWin : public BWindow
{
public:
	TestWin() : BWindow(BRect(50, 50, 51, 51), "Selector_test",
		B_TITLED_WINDOW, B_NOT_CLOSABLE | B_NOT_ZOOMABLE | B_NOT_RESIZABLE)
	{
		BView* bkg = new BView(Bounds(), "bkg view", B_FOLLOW_ALL,
			B_WILL_DRAW);
		bkg->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
		AddChild(bkg);
		BButton* stop = new BButton(BRect(5,5,6,6),
			"stop button", "Stop Selector", new BMessage(B_QUIT_REQUESTED));
		stop->ResizeToPreferred();
		bkg->AddChild(stop);
		stop->SetTarget(be_app);
		ResizeTo(stop->Frame().right + 5, stop->Frame().bottom + 5);	
		Show();
	}
};

void TestApp::StartTestWindow()
{
	// this creates a window with a "Stop" button
	new TestWin();
}
