/*
	
	SoundFileReader.cpp
	
	Implements a sound file reader node.
	
	Copyright 1999 Be Incorporated, All Rights Reserved.

*/

#include <stdio.h>
#include <string.h>
#include <Sound.h>
#include <StopWatch.h>
#include <Debug.h>

#include "SoundFileReader.h"

#if B_HOST_IS_LENDIAN
#define MEDIA_HOST_ENDIAN B_MEDIA_LITTLE_ENDIAN
#else
#define MEDIA_HOST_ENDIAN B_MEDIA_BIG_ENDIAN
#endif

SoundFileReader::SoundFileReader(const char* name, const entry_ref* ref)
	: BMediaNode(name ? name : "SoundFileReader"),
	SoundProducer(name), m_sound(0)
{
	SetRef(ref);
	// for now, we always loop
	m_bLoop = true;
}

SoundFileReader::~SoundFileReader()
{
	ClearSound();
}

status_t SoundFileReader::SetRef(const entry_ref* ref)
{
	// When we receive a reference to a file, we load the file
	// into a BSound and change the format. This interface
	// should be replaced by the BFileInterface methods at
	// some point.
	ClearSound();
	m_pos = 0;
	
	if (ref) {
		m_sound = new BSound(ref, false);
		status_t err = m_sound->InitCheck();
		if (err != B_OK) {
			fprintf(stderr, "SoundFileReader::SoundFileReader() - can't open sound file: %s\n", strerror(err));
			ClearSound();
			return err;
		}
		
		media_raw_audio_format format = m_sound->Format();
		SetFormat(&format);
	}
	return B_OK;
}

bigtime_t SoundFileReader::ProcessingLatency()
{
	// BSound is much better than SoundCapture at reading files!
	return 10000;
}

void SoundFileReader::Play(
	bigtime_t /* time */,
	void* buffer,
	size_t size,
	const media_raw_audio_format& /* format */)
{
	if (m_bPlaying) {
		ASSERT(m_sound);
		uint8* cur = static_cast<uint8*>(buffer);
		size_t sizeRead;

		while (size > 0) {
			m_sound->GetDataAt(m_pos, cur, size, &sizeRead);
			
			if (m_bLoop && sizeRead < size) {
				// time to loop! Reset our position...
				m_pos = 0;
			} else {
				m_pos += sizeRead;
			}
			cur += sizeRead;	
			size -= sizeRead;
		}
	} else {
		// we're not playing anything, so just fill the
		// buffer with the sound of silence...
		memset(buffer, 0, size);
	}
}

void SoundFileReader::Notify(
	int32 cause, ...)
{
	switch (cause) {
	case B_WILL_START:	
		m_bPlaying = true;
		break;
	case B_WILL_STOP:
	case B_DISCONNECTED:
	case B_FORMAT_CHANGED:
	case B_NODE_DIES:
		m_bPlaying = false;
		break;
	case B_WILL_SEEK:
		// note: we don't handle this, but we might if we were implementing
		// a more sophisticated looping mechanism.
		break;
	}
}

void SoundFileReader::ClearSound()
{
	if (m_sound) {
		m_sound->ReleaseRef();
		m_sound = 0;
	}
}
