/*******************************************************************************
/
/	File:			MediaNodeWrapper.cpp
/
/   Description:	A application-based object-oriented API for
/					working with media_nodes.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <Debug.h>
#include <MediaRoster.h>
#include <TimeSource.h>
#include "MediaNodeWrapper.h"

MediaNodeWrapper::MediaNodeWrapper()
	: m_init(false)
{
}

MediaNodeWrapper::MediaNodeWrapper(const media_node& node, bool locked)
	: m_node(node), m_locked(locked), m_init(true)
{
}

MediaNodeWrapper::MediaNodeWrapper(const MediaNodeWrapper& node)
	: m_node(node.m_node), m_locked(node.m_locked), m_init(node.m_init)
{
}

MediaNodeWrapper::~MediaNodeWrapper()
{
}

MediaNodeWrapper& MediaNodeWrapper::operator=(const MediaNodeWrapper& node)
{
	if (&node != this) {
		m_node = node.m_node;
		m_locked = node.m_locked;
		m_init = node.m_init;
	}
	return *this;
}

void MediaNodeWrapper::Start(bigtime_t tpStart) const
{
	if (m_init && (! m_locked))
		BMediaRoster::Roster()->StartNode(m_node, tpStart);
}

void MediaNodeWrapper::Stop(bigtime_t tpStop, bool synchronous) const
{
	if (m_init && (! m_locked))
		BMediaRoster::Roster()->StopNode(m_node, tpStop, synchronous);
}

void MediaNodeWrapper::SlaveTo(MediaNodeWrapper master) const
{
	if (m_init && (! m_locked)) {
		BTimeSource* ts = master.GetTimeSource();
		BMediaRoster::Roster()->SetTimeSourceFor(m_node.node, ts->Node().node);
	}
}

BTimeSource* MediaNodeWrapper::GetTimeSource() const
{
	if (m_init)
		return BMediaRoster::Roster()->MakeTimeSourceFor(m_node);
	else return 0;
}

status_t MediaNodeWrapper::GetFreeOutput(media_output* output) const
{
	if (m_init) {
		int32 count;
		return BMediaRoster::Roster()->GetFreeOutputsFor(m_node, output, 1, &count);
	} else return B_ERROR;
}

status_t MediaNodeWrapper::GetFreeInput(media_input* input) const
{
	if (m_init) {
		int32 count;
		return BMediaRoster::Roster()->GetFreeInputsFor(m_node, input, 1, &count);
	} else return B_ERROR;
}

status_t MediaNodeWrapper::GetLatency(bigtime_t* latency) const
{
	if (m_init)
		return BMediaRoster::Roster()->GetLatencyFor(m_node, latency);
	else return B_ERROR;
}

void MediaNodeWrapper::Release() const
{
	if (m_init) {
		BMediaRoster::Roster()->ReleaseNode(m_node);
	}
}
