/*******************************************************************************
/
/	File:			SelectorNode.h
/
/   Description:	A node that allows the user to select one of
/					several inputs to route to the output.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <Buffer.h>
#include <Debug.h>
#include <ParameterWeb.h>
#include <OS.h>
#include <Roster.h>
#include <MediaAddOn.h>
#include <scheduler.h>
#include <TimeSource.h>
#include "SelectorNode.h"
#include "utilmedia.h"

const int32 SelectorNode::INPUT_SELECTED = BTimedEventQueue::B_USER_EVENT;

SelectorNode::SelectorNode(const char* name, BMediaAddOn* addon, int32 flavor_id)
	: BBufferProducer(B_MEDIA_UNKNOWN_TYPE), BBufferConsumer(B_MEDIA_UNKNOWN_TYPE),
	BMediaNode(name), m_addon(addon), m_flavorID(flavor_id)
{
	Init();
}

SelectorNode::~SelectorNode()
{
	PRINT(("SelectorNode::~SelectorNode\n"));
	SetFormat(0);
}

void 
SelectorNode::HandleEvent(bigtime_t performance_time, int32 what, const void *pointer, uint32 cleanup, int64 data)
{
	switch (what) {
	case BTimedEventQueue::B_HANDLE_BUFFER:
		{
			// The pointer is const by mistake -- this should be fixed
			// in a later version...
			BBuffer* buffer = const_cast<BBuffer*>(static_cast<const BBuffer*>(pointer));
			ASSERT(buffer);

#if DEBUG
			bigtime_t receivedAt = data;
			bigtime_t now = TimeSource()->Now();
			bigtime_t offset = TimeSource()->Now() + EventLatency() - ProcessingLatency() - buffer->Header()->start_time;
			PRINT(("handle buffer (%Ld): ", now - receivedAt));
			if (offset <= 0)
				PRINT(("early by %Ld\n", -offset));
			else
				PRINT(("LATE! by %Ld\n", offset));
#endif
			
			if (buffer->Header()->destination != static_cast<uint32>(m_selectedInput)) {
				// this buffer didn't come from the selected input, so recycle it!
				buffer->Recycle();
				break;
			}
	
			// This buffer is from the selected input! Try to send the buffer
			// if we are connected to an output, if we're running, and if our
			// output is enabled.
			if (! (IsConnected() && (! IsStopped()) && m_enabled
				&& SendBuffer(buffer, m_output.destination) == B_OK))
			{
				// We couldn't send the buffer for some reason. Get rid of it!
				buffer->Recycle();
			}
		}
		break;
	case BTimedEventQueue::B_DATA_STATUS:
		{
			PRINT(("SelectorNode::HandleEvent: B_DATA_STATUS\n"));
			// The pointer is const by mistake -- this should be fixed
			// in a later version...
			media_input* input = const_cast<media_input*>(static_cast<const media_input*>(pointer));
			int32 status = static_cast<int32>(data);
			if (! input)
				break;
				
			int32 index = input->destination.id;
			if (! check_selector_range(index))
				break;
			
			// Upate the data status for the specified input.
			if (m_status[index] != status) {
				m_status[index] = status;
				if (index == m_selectedInput) {
					// Our output's status is the same as the selected input's
					// status, so tell the downstream node when the selected
					// input's status changes.
					SendDataStatus(status, m_output.destination, performance_time);
				}
			}
		}
		break;
	case BTimedEventQueue::B_START:
		{
			PRINT(("SelectorNode::HandleEvent: B_START\n"));
			// Buffers from the selected input will now be streaming to the
			// output, so update the status for our downstream node
			PRINT(("Data status is %d\n", m_status[m_selectedInput]));
			SendDataStatus(m_status[m_selectedInput], m_output.destination, performance_time);
		}
		break;
	case BTimedEventQueue::B_STOP:
		{
			PRINT(("SelectorNode::HandleEvent: B_STOP\n"));
			// No buffers will be sent now, so update the data status for our
			// downstream node.
			SendDataStatus(B_DATA_NOT_AVAILABLE, m_output.destination, performance_time);
		}
		break;
	case INPUT_SELECTED:
		{
			PRINT(("SelectorNode::HandleEvent: INPUT_SELECTED\n"));
			int32 index = static_cast<int32>(data);
			if (! check_selector_range(index))
				break;
				
			// Change the selected input. If the new input's status is different,
			// the downstream node should be informed!
			m_selectedInput = index;
			m_tpInputSelected = performance_time;
			SendDataStatus(m_status[index], m_output.destination, performance_time);
		}
		break;
	default:
		break;
	}
}

status_t
SelectorNode::HandleMessage(int32 what, const void* data, size_t size)
{
	status_t err = B_OK;
	switch (what) {
	case SELECTOR_INIT:
		{
			// Configure the node with the information we've received.
			ASSERT(size >= sizeof(selector_init_struct));
			selector_init_struct* init = static_cast<selector_init_struct*>(data);
			if (init->controls) {
				// The app wants us to start a control panel. Make it so.
				BMessenger msgr;
				StartControlPanel(&msgr);
			}
			if (check_selector_range(init->initSel)) {
				// The app has told us what our selection should now be.
				// Update it.
				m_selectedInput = init->initSel;
				m_tpInputSelected = 0;
				SendDataStatus(m_status[m_selectedInput], m_output.destination, 0);
				UpdateWeb();
			}	
			break;
		}
	default:
		err = B_ERROR;
		break;
	}
	
	return err;
}

void
SelectorNode::NodeRegistered()
{
	PRINT(("SelectorNode::NodeRegistered\n"));
	
	// Creating the parameter web correctly relies on us being registered!
	UpdateWeb();
	BMediaEventLooper::NodeRegistered();
}

status_t 
SelectorNode::FormatSuggestionRequested(media_type type, int32 /* quality */, media_format *format)
{
	// If we don't already have a format established, we don't have any sugggestions.
	if (! Format()) {
		return B_MEDIA_BAD_FORMAT;
	}
	
	// Otherwise, our existing format is our only suggestion.
	status_t err;
	if (type != Format()->type)
		err = B_MEDIA_BAD_FORMAT;
	else
		err = B_OK;
	
	*format = *(Format());
	return err;
}

status_t 
SelectorNode::FormatProposal(const media_source &what, media_format *format)
{
	if (what != m_output.source)
		return B_MEDIA_BAD_SOURCE;
	
	status_t err;
	media_format* curFmt = Format();
	if (! curFmt) {
		// At least one input must be connected first to establish the format!
		PRINT(("SelectorNode::FormatProposal: Must connect an input first!\n"));
		err = B_MEDIA_BAD_FORMAT;
	} else if (! format_is_compatible(*format, *curFmt)) {
		PRINT(("SelectorNode::FormatProposal: Bad proposed format!\n"));
		*format = *curFmt;
		err = B_MEDIA_BAD_FORMAT;
	} else {
		PRINT(("SelectorNode::FormatProposal: Format looks A-OK.\n"));
		// Fill in all of the wildcards -- we can't afford to be flexible.
		specialize_format(format, *curFmt);
		err = B_OK;
	}
	
	return err;
}

status_t 
SelectorNode::FormatChangeRequested(const media_source &what, const media_destination &where, media_format *io_format, int32 *_deprecated_)
{
	// We don't currently allow format changes. Consider this an exercise
	// for the reader for now.
	return B_ERROR;
}

status_t 
SelectorNode::GetNextOutput(int32 *cookie, media_output *out_output)
{
	if (*cookie != 0)
		return B_BAD_INDEX;
	
	if (m_output.destination == media_destination::null) {
		// This is an unconnected output, so fill it in on the spot.
		m_output.source.port = ControlPort();
		m_output.source.id = 0;
		m_output.destination = media_destination::null;
		if (Format()) {
			// We already know what our format has to be.
			m_output.format = *(Format());
		} else {
			// No input has been connected to us yet, so we
			// don't know what the format will be.
			m_output.format.type = B_MEDIA_NO_TYPE;
		}
	}
	
	strncpy(m_output.name, "Selector Output", B_MEDIA_NAME_LENGTH);

	*out_output = m_output;
	(*cookie)++;
	return B_OK;
}

status_t 
SelectorNode::DisposeOutputCookie(int32 cookie)
{
	// nothing to do
	return B_OK;
}

status_t 
SelectorNode::SetBufferGroup(const media_source &what, BBufferGroup *group)
{
	// We don't currently support changing buffer groups.
	return B_ERROR;
}

status_t 
SelectorNode::PrepareToConnect(const media_source &what, const media_destination &where, media_format* format, media_source* out_source, char *out_name)
{
	PRINT(("SelectorNode::PrepareToConnect\n"));

	if (what != m_output.source)
		return B_MEDIA_BAD_SOURCE;
	
	if (m_output.destination != media_destination::null)
		return B_MEDIA_ALREADY_CONNECTED;
	
	// Reserve the connection by filling in the tentative destination.
	*out_source = m_output.source;
	m_output.format = *format;
	m_output.destination = where;
	strncpy(out_name, m_output.name, B_MEDIA_NAME_LENGTH);
	return B_OK;
}

void 
SelectorNode::Connect(status_t error, const media_source &what, const media_destination& where, const media_format& format, char *io_name)
{
	PRINT(("SelectorNode::Connect\n"));
	ASSERT(what == m_output.source);
	
	if (error != B_OK) {
		// The connection has failed, so unreserve the connection.
		m_output.destination = media_destination::null;
	} else {
		// Finalize the connection by filling in the final destination and format.
		m_output.destination = where;
		m_output.format = format;
		strncpy(m_output.name, io_name, B_MEDIA_NAME_LENGTH);
		
		// Now that we have a connected output, we need to adjust our latency!
		UpdateLatency();
	}
}

void 
SelectorNode::Disconnect(const media_source &what, const media_destination &where)
{
	PRINT(("SelectorNode::Disconnect\n"));
	ASSERT(what == m_output.source);
	m_output.destination = media_destination::null;
}

void 
SelectorNode::LateNoticeReceived(const media_source &what, bigtime_t how_much, bigtime_t tpAt)
{
	PRINT(("SelectorNode::LateNotice received: by %Ld at %Ld\n", how_much, tpAt));
	ASSERT(what == m_output.source);
	// TODO: implement	
}

void 
SelectorNode::EnableOutput(const media_source &what, bool enabled, int32 *_deprecated_)
{
	PRINT(("SelectorNode::EnableOutput %hd\n", (int16)enabled));
	ASSERT(what == m_output.source);
	m_enabled = enabled;
}

status_t 
SelectorNode::AcceptFormat(const media_destination &where, media_format *format)
{
	PRINT(("SelectorNode::AcceptFormat for %ld\n", where.id));
	media_input* in = InputAt(where);
	if (! in)
		return B_MEDIA_BAD_DESTINATION;

	media_format* curFmt = Format();

	// If we haven't already established a format, we can take whatever
	// the upstream node wants to give us.
	if (! curFmt)
		return B_OK;
	
	// Otherwise, because we only support one format, we have to use
	// the already-established format.
	char str[256];
	string_for_format(*curFmt, str, 256);
	PRINT(("We accept %s, ", str));
	string_for_format(*format, str, 256);
	PRINT(("they're asking for %s\n", str));
	
	if (! format_is_compatible(*format, *curFmt))
		return B_MEDIA_BAD_FORMAT;
	
	// Make sure all the wildcards get filled in!	
	specialize_format(format, *curFmt);
	return B_OK;
}

status_t 
SelectorNode::GetNextInput(int32 *cookie, media_input *out_input)
{
	int32& index = *cookie;
	if (! check_selector_range(index))
		return B_BAD_INDEX;
	
	media_input& in = m_inputs[index];
	if (in.source == media_source::null) {
		// This input is unconnected -- fill in
		// its information on the spot.
		in.destination.port = ControlPort();
		in.destination.id = index;
		sprintf(in.name, "Input %ld\n", index);
		if (Format()) {
			in.format = *(Format());
		} else {
			in.format.type = B_MEDIA_NO_TYPE;
		}
	}
	*out_input = m_inputs[index];
	index++;
	return B_OK;
}

void 
SelectorNode::DisposeInputCookie(int32 cookie)
{
	// nothing to do
}

void 
SelectorNode::BufferReceived(BBuffer *buffer)
{
	ASSERT(buffer);
	
#if DEBUG
	bigtime_t now = TimeSource()->Now();
	bigtime_t offset = TimeSource()->Now() + EventLatency() - buffer->Header()->start_time;
	PRINT(("buffer received (%Ld): ", buffer->Header()->start_time - now));
	if (offset <= 0)
		PRINT(("early by %Ld\n", -offset));
	else
		PRINT(("LATE! by %Ld\n", offset));
#else
	bigtime_t now = 0;
#endif

	// Simply push the buffer onto the queue. We'll see it again
	// in HandleEvent. We also have room to pass the current
	// performance time for debugging purposes.		
	EventQueue()->PushEvent(buffer->Header()->start_time, BTimedEventQueue::B_HANDLE_BUFFER,
		buffer, BTimedEventQueue::B_RECYCLE, now);
}

void 
SelectorNode::ProducerDataStatus(const media_destination &where, int32 status, bigtime_t performance_time)
{
	PRINT(("SelectorNode::ProducerDataStatus for %ld\n", where.id));
	
	// Push the data status event onto the queue to be handled at the proper time.
	media_input* in = InputAt(where);
	EventQueue()->PushEvent(performance_time, BTimedEventQueue::B_DATA_STATUS,
		in, BTimedEventQueue::B_NO_CLEANUP, status);
}

status_t 
SelectorNode::GetLatencyFor(const media_destination &where, bigtime_t *out_latency, media_node_id *out_timesource)
{
	// Our latency is the downstream latency plus our processing latency.
	if (m_output.destination != media_destination::null)
		FindLatencyFor(m_output.destination, out_latency, out_timesource);
	else
		*out_latency = 0;

	*out_latency += ProcessingLatency();
	return B_OK;
}

status_t 
SelectorNode::Connected(const media_source &what, const media_destination &where, const media_format &with, media_input *out_input)
{
	PRINT(("SelectorNode::Connected at %ld\n", where.id));
	media_input* in = InputAt(where);
	if (! in)
		return B_MEDIA_BAD_DESTINATION;
	
	if (in->source != media_source::null)
		return B_MEDIA_ALREADY_CONNECTED;

	// Set the format, especially if this is the first input connection!	
	SetFormat(&with);
	
	// Finalize the information on this side of the connection.
	in->format = with;
	in->source = what;
	strncpy(in->name, out_input->name, B_MEDIA_NAME_LENGTH);
	*out_input = *in;

	// Now that we've definitely specified a format, we can calculate
	// what our priority and initial latency should be.
	UpdateThreadPriority();
	UpdateLatency();
	
	// Because we've added an input, our input selector parameter
	// needs to be changed.
	UpdateWeb();
	return B_OK;
}

void 
SelectorNode::Disconnected(const media_source &what, const media_destination &where)
{
	PRINT(("SelectorNode::Disconnected at %ld\n", where.id));
	media_input* in = InputAt(where);
	ASSERT(in && in->source == what);
	in->source = media_source::null;
	
	// Because we've removed an input, our input selector parameter
	// needs to be changed.
	UpdateWeb();
}

status_t 
SelectorNode::FormatChanged(const media_source &what, const media_destination &where, int32 _deprecated_, const media_format &format)
{
	// We don't currently allow format changes...
	return B_MEDIA_BAD_FORMAT;
}

status_t 
SelectorNode::GetParameterValue(int32 id, bigtime_t *last_change, void *value, size_t *ioSize)
{
	status_t err = B_OK;
	switch (id) {
	case INPUT_SELECTED:
		// This is our input selection parameter. Stuff value
		// with our last known parameter value.
		*last_change = m_tpInputSelected;
		if (*ioSize < sizeof(int32)) {
			err = B_NO_MEMORY;
		} else {
			int32* ival = static_cast<int32*>(value);
			*ival = m_selectedInput;
		}
		*ioSize = sizeof(int32);
		break;
	default:
		err = B_BAD_INDEX;
		break;
	}
		
	return err;
}

void 
SelectorNode::SetParameterValue(int32 id, bigtime_t when, const void *value, size_t size)
{
	switch (id) {
	case INPUT_SELECTED:
		{
			if (size < sizeof(int32))
				break;

			// This is our input selection parameter. Because we're
			// a discrete single-channel parameter, we know that we
			// just need to extract a single int32 value from the
			// value buffer, which we'll interpret as an index into
			// our array of inputs. Push an event onto the queue to
			// make sure we change the input at the correct time.			
			const int32* ival = static_cast<const int32*>(value);
			EventQueue()->PushEvent(when, INPUT_SELECTED, 0, BTimedEventQueue::B_NO_CLEANUP, *ival);
			break;
		}
	default:
		break;
	}
}

void
SelectorNode::Init()
{
	m_format = 0;
	m_selectedInput = 0;
	m_tpInputSelected = 0;
	m_enabled = true;
	for (int32 i=0; i<MAX_SELECTORS; i++)
		m_status[i] = B_DATA_NOT_AVAILABLE;
}

void
SelectorNode::UpdateWeb()
{
	PRINT(("SelectorNode: Updating parameter web\n"));
	
	// Create a brand-new parameter web for our node.
	BParameterWeb* web = new BParameterWeb;
	BParameterGroup* group = web->MakeGroup("Input");
	media_type type;
	if (Format()) {
		type = Format()->type;
	} else {
		type = B_MEDIA_NO_TYPE;
	}
	
	// This is our single discrete parameter that represents the
	// selected input. 
	BDiscreteParameter* p = group->MakeDiscreteParameter(INPUT_SELECTED, type,
		"Source", B_INPUT_MUX);
		
	// The inputs and outputs to our node are represented in the UI as
	// null parameters that are "connected" to the input selector parameter.
	for (int32 i=0; i<MAX_SELECTORS; i++) {
		if (m_inputs[i].source != media_source::null) {
			PRINT(("   making input: %s\n", m_inputs[i].name));
			BNullParameter* in = group->MakeNullParameter(i, type,
				m_inputs[i].name, B_WEB_BUFFER_INPUT);
			p->AddInput(in);
		}
	}
	p->MakeItemsFromInputs();
	if (m_output.destination != media_destination::null) {
		BNullParameter* out = group->MakeNullParameter(MAX_SELECTORS, type,
			m_output.name, B_WEB_BUFFER_OUTPUT);
		p->AddOutput(out);
	}
	SetParameterWeb(web);
	PRINT(("SelectorNode: Done updating parameter web.\n"));
}

media_input*
SelectorNode::InputAt(const media_destination& where)
{
	media_input* res;
	int32 index = where.id;
	if (check_selector_range(index)
		&& m_inputs[index].destination == where)
	{
		res = m_inputs + index;	
	} else {
		res = 0;
	}
	return res;
}

void
SelectorNode::SetFormat(const media_format* format)
{
	delete m_format;
	if (format) {
#if DEBUG
		char str[256];
		string_for_format(*format, str, 256);
		PRINT(("SelectorNode: setting format to %s\n", str));
#endif
		m_format = new media_format(*format);
	} else {
		PRINT(("SelectorNode: clearing format\n"));
		m_format = 0;
	}
}

void
SelectorNode::UpdateThreadPriority()
{
	int32 priority = suggest_priority_from_format(*(Format()));
	SetPriority(priority);
}

void
SelectorNode::UpdateLatency()
{
	PRINT(("SelectorNode: Updating latency\n"));
	// To avoid jitter, we'll set our processing latency to one
	// full buffer's duration, plus scheduling jitter for our
	// thread.
	bigtime_t procLatency = buffer_duration(*(Format()));
	PRINT(("   buffer duration is %Ld\n", procLatency));
	bigtime_t schedulingLatency = estimate_max_scheduling_latency();
	PRINT(("   scheduling jitter is %Ld\n", schedulingLatency));
	procLatency += schedulingLatency;
	SetProcessingLatency(procLatency);

	// GetLatencyFor will add the new processing latency to
	// the downstream latency to give us our new total latency.
	bigtime_t totalLatency = 0;
	media_node_id ts;
	GetLatencyFor(m_output.destination, &totalLatency, &ts);
	PRINT(("   downstream latency is %Ld\n", totalLatency - procLatency));
	PRINT(("   ...setting total latency to %Ld.\n", totalLatency));

	if (totalLatency != EventLatency()) {
		// Tell all our upstream nodes that the latency
		// just changed for our node!
		for (int32 i=0; i<MAX_SELECTORS; i++) {
			media_input& in = m_inputs[i];
			if (in.source != media_source::null)
				SendLatencyChange(in.source, in.destination, totalLatency);
		}
		// We store our total latency as the event queue's EventLatency.
		SetEventLatency(totalLatency);
	}
}

#if GENKI_4
// This implements the stuff that was added between genki/4 and genki/5:
// latency changes and a new implementation of StartControlPanel that passes
// the node ID as an argument to the control panel app.
void
SelectorNode::SendLatencyChange(const media_source& what, const media_destination& where,
	bigtime_t new_latency, uint32 flags)
{
	my_latency_change_request q;
	q.src = what;
	q.dest = where;
	q.latency = new_latency;
	q.flags = flags;
	
	write_port(what.port, MY_LATENCY_CHANGED, &q, sizeof(q));
}

status_t
SelectorNode::StartControlPanel(BMessenger* msgr)
{
	PRINT(("SelectorNode::StartControlPanel\n"));
	image_id img = m_addon->ImageID();
	if (img < 0)
		return img;
		
	image_info img_info;
	status_t err = get_image_info(img, &img_info);
	if (err != B_OK)
		return err;
		
	entry_ref ref;
	err = get_ref_for_path(img_info.name, &ref);
	if (err != B_OK)
		return err;
	
	team_id team = -1;
	char arg[256];
	sprintf(arg, "node=%ld", ID());
	char* argv[2] = { arg, 0 };
	err = be_roster->Launch(&ref, 1, argv, &team);
	if (err == B_OK)
		*msgr = BMessenger(0, team);
	
	return err;
}
#endif
