/*******************************************************************************
/
/	File:			SelectorAddOn.cpp
/
/   Description:	An add-on that contains the Selector Node.
/
/	Copyright 1999, Be Incorporated, All Rights Reserved
/
*******************************************************************************/

#include <Debug.h>
#include "SelectorAddOn.h"
#include "SelectorNode.h"
#include "utilmedia.h"
#include "MediaControlPanelApp.h"

// main() gets called when the user wants to launch a control panel
// for one of our nodes. MediaControlPanelApp takes care of everything
// for us in this case.
int main()
{
	MediaControlPanelApp app("application/x-vnd.Be-DTS.Selector.media_addon");
	app.Run();
	return 0;
}

// This is actually the main entry point for the add-on,
// and gives the Media Add-On Server access to our
// BMediaAddOn-derived object.
//
// The image_id refers to our add-on, not to the application
// that loaded us.
extern "C" _EXPORT BMediaAddOn* make_media_addon(image_id you)
{
	return new SelectorAddOn(you);
}

SelectorAddOn::SelectorAddOn(image_id image)
	: BMediaAddOn(image)
{
	// We support exactly one flavor; we'll fill in all of the
	// important fields here.
	m_flavor = new flavor_info;
	m_flavor->name = "Selector";
	m_flavor->info = "Select one output from a collection of inputs.";		
	m_flavor->kinds = B_BUFFER_PRODUCER | B_BUFFER_CONSUMER;
	m_flavor->internal_id = 0;
	m_flavor->possible_count = 0;
	
	// The selector node should be able to handle any kind of
	// format. We have to list each format type separately, so
	// we'll just stick to the common ones we know about for now.
	m_flavor->in_format_count = m_flavor->out_format_count = 5;
	m_formats = new media_format[5];
	m_formats[0].type = B_MEDIA_RAW_AUDIO; wildcard_format(m_formats);
	m_formats[1].type = B_MEDIA_RAW_VIDEO; wildcard_format(m_formats+1);
	m_formats[2].type = B_MEDIA_MULTISTREAM; wildcard_format(m_formats+2);
	m_formats[3].type = B_MEDIA_ENCODED_AUDIO; wildcard_format(m_formats+3);
	m_formats[4].type = B_MEDIA_ENCODED_VIDEO; wildcard_format(m_formats+4);
	m_flavor->in_formats = m_flavor->out_formats = m_formats;
}

SelectorAddOn::~SelectorAddOn()
{
	delete m_flavor;
	delete [] m_formats;
}

status_t
SelectorAddOn::InitCheck(const char** out_failure_test)
{
	return B_OK;	
}

int32
SelectorAddOn::CountFlavors()
{
	return 1;
}

status_t
SelectorAddOn::GetFlavorAt(int32 n, const flavor_info** info)
{
	status_t err;
	// there's exactly one flavor, so "0" is the only valid index
	if (n != 0) {
		*info = 0;
		err = B_BAD_INDEX;
	} else {
		*info = m_flavor;
		err = B_OK;
	}
	return err;
}

BMediaNode*
SelectorAddOn::InstantiateNodeFor(const flavor_info* info, BMessage* /* config */, status_t* out_error)
{
	PRINT(("SelectorAddOn::InstantiateNodeFor\n"));
	ASSERT(info->internal_id == m_flavor->internal_id);
	*out_error = B_OK;
	return new SelectorNode(info->name, this, 0);
}

status_t
SelectorAddOn::GetConfigurationFor(BMediaNode* /* your_node */, BMessage* /* into_message */)
{
	// This hasn't been implemented by the media_server yet,
	// so "future goes here..."
	return B_OK;
}

bool
SelectorAddOn::WantsAutoStart()
{
	return false;
}

status_t
SelectorAddOn::AutoStart(int /* in_count */, BMediaNode** /* out_node */, int32* /* out_internal_id */, bool* has_more)
{
	// But we told the Media Add-On Server that
	// we didn't want to auto-start!!
	TRESPASS();
	*has_more = false;
	return B_ERROR;
}
