#include "MotherNode.h"
#include <TimeSource.h>
#include <OS.h>
#include <stdio.h>


MotherNode::MotherNode() :
	BMediaNode("error"),
	_mControlThread(-1),
	_mPriority(B_NORMAL_PRIORITY),
	_mEventLatency(0),
	_mRunning(false)
{
}

MotherNode::~MotherNode()
{
	Quit();
}

void 
MotherNode::NodeRegistered()
{
	Go();
}

void 
MotherNode::Start(bigtime_t performance_time)
{
	_mEventQueue.PushEvent(performance_time, BTimedEventQueue::B_START, NULL, 0, 0);
}

void 
MotherNode::Stop(bigtime_t performance_time, bool immediate)
{
	if (immediate)
	{
		// always be sure to add to the front of the queue so we can make sure it is
		// handled before any buffers are sent!
		performance_time = 0;
		if (performance_time >= _mEventQueue.NextEventTime())
			performance_time = _mEventQueue.NextEventTime() -1;
	}
	_mEventQueue.PushEvent(performance_time, BTimedEventQueue::B_STOP, NULL, 0, 0);
}

void 
MotherNode::Seek(bigtime_t media_time, bigtime_t performance_time)
{
	_mEventQueue.PushEvent(performance_time, BTimedEventQueue::B_SEEK, NULL, 0, media_time);
}

void 
MotherNode::TimeWarp(bigtime_t at_real_time, bigtime_t to_performance_time)
{
	_mEventQueue.PushEvent(TimeSource()->PerformanceTimeFor(at_real_time),
						BTimedEventQueue::B_WARP, NULL, 0, to_performance_time);
}

bool 
MotherNode::IsStopped()
{
	return !_mRunning;
}


void 
MotherNode::HandleEvent(const bigtime_t performanceTime, const int32 what, const void *pointer, const uint32 cleanup, const int64 data)
{
	/* nothing */
}

void 
MotherNode::SetEventLatency(bigtime_t latency)
{
	// clamp to a valid value
	if (latency < 0)
		latency = 0;

	_mEventLatency = latency;
}

void 
MotherNode::SetPriority(int32 priority)
{
	// clamp to a valid value
	if (priority < 1)
		priority = 1;
		
	if (priority > 120)
		priority = 120;
		
	_mPriority = priority;
	if(_mControlThread > 0)
		set_thread_priority(_mControlThread, _mPriority);
}

void 
MotherNode::Go()
{
	char threadName[32];
	sprintf(threadName, "%.20s control", Name());
	_mControlThread = spawn_thread(_ControlThreadStart, threadName, _mPriority, this);
	resume_thread(_mControlThread);
}

void 
MotherNode::Quit()
{
	// this is a new call that will destroy the port but
	// still let you pull any waiting messages out of it.
	status_t err = close_port(ControlPort());
	wait_for_thread(_mControlThread, &err);
	_mControlThread = -1;
	_mRunning = false;
}

void 
MotherNode::_ControlLoop()
{
	while (true)
	{
		bigtime_t waitUntil = B_INFINITE_TIMEOUT;
		if(_mEventQueue.HasEvents())
			waitUntil = TimeSource()->RealTimeFor(_mEventQueue.NextEventTime(), _mEventLatency);
		
		status_t err = WaitForMessage(waitUntil);
		
		if (err == B_TIMED_OUT) {
			_PHandleEvent();
		}
		else if (err != B_OK)
		{
			ReportError(B_NODE_IN_DISTRESS);
			break;
		}
	}
}

status_t 
MotherNode::_Reserved_MotherNode_0(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_1(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_2(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_3(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_4(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_5(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_6(void *)
{
	return B_ERROR;
}

status_t 
MotherNode::_Reserved_MotherNode_7(void *)
{
	return B_ERROR;
}

int32 
MotherNode::_ControlThreadStart(void *data)
{
	((MotherNode *)data)->_ControlLoop();
	return 0;
}

void
MotherNode::_PHandleEvent()
{
	bigtime_t performanceTime = 0;
	int32 what = BTimedEventQueue::B_NO_EVENT;
	void *pointer = NULL;
	uint32 cleanup = BTimedEventQueue::B_NO_CLEANUP;
	int64 data = 0;
	
	if (_mEventQueue.PopEvent(&performanceTime, &what, &pointer, &cleanup, &data) != B_OK)
		return;
		
	HandleEvent(performanceTime, what, pointer, cleanup, data);

	switch(what) {
		case BTimedEventQueue::B_START:
			_mRunning = true;
			break;
		
		case BTimedEventQueue::B_STOP: 
			_mRunning = false;
			break;
	
		case BTimedEventQueue::B_SEEK:
			/* nothing */
			break;
		
		case BTimedEventQueue::B_WARP:
			/* nothing */
			break;
		
		default:
			break;
	}
}
